<?php
namespace Swissup\Core\Console\Command;

use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Exception\FileSystemException;
use Magento\Framework\Filesystem;
use Swissup\Core\Console\Command\ThemeCreateCommand;

class ThemeCreate
{
    const THEME_COMPOSER_TEMPLATE = <<<EOT
{
    "name": "{{THEME_PACKAGE_NAME}}",
    "type": "magento2-theme",
    "version": "1.0.0",
    "require": {
        "{{PARENT_THEME_PACKAGE_NAME}}": "*"
    },
    "autoload": {
        "files": [ "registration.php" ]
    }
}
EOT;

    const THEME_REGISTRATION_TEMPLATE = <<<EOT
<?php
use \Magento\Framework\Component\ComponentRegistrar;
ComponentRegistrar::register(
    ComponentRegistrar::THEME,
    '{{THEME_NAME}}',
    __DIR__
);
EOT;

    const THEME_XML = <<<EOT
<?xml version="1.0"?>
<theme xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="urn:magento:framework:Config/etc/theme.xsd">
    <title>{{THEME_NAME}}</title>
    <parent>{{PARENT_THEME_NAME}}</parent>
</theme>
EOT;

    /**
     * @var Filesystem\Directory\ReadInterface
     */
    private $appRead;

    /**
     * @var Filesystem\Directory\WriteInterface
     */
    private $appWrite;

    /**
     * constructor.
     *
     * @param Filesystem                  $fs
     * @throws FileSystemException
     */
    public function __construct(Filesystem $fs)
    {
        $this->appRead = $fs->getDirectoryRead(DirectoryList::APP);
        $this->appWrite = $fs->getDirectoryWrite(DirectoryList::APP);
    }

    /**
     * @param string $themeName
     * @return int
     * @throws FileSystemException
     */
    public function generateRegistration(string $themeName): int
    {
        $destinationPath = $this->getThemePath($themeName);

        $content = self::THEME_REGISTRATION_TEMPLATE;
        $content = str_replace(
            '{{THEME_NAME}}',
            ThemeCreateCommand::SECTION . \Magento\Framework\Config\Theme::THEME_PATH_SEPARATOR . $themeName,
            $content
        );

        return $this->appWrite->writeFile(
            $destinationPath . DIRECTORY_SEPARATOR . 'registration.php',
            $content
        );
    }

    /**
     *
     * @param  string  $themeName
     * @return boolean
     */
    public function isExist(string $themeName)
    {
        $path = $this->getThemePath($themeName);
        return $this->appWrite->isExist($path);
    }

    /**
     * @param string $themeName
     * @param string $parentThemeName
     * @return int
     * @throws FileSystemException
     */
    public function generateThemeXml(string $themeName, string $parentThemeName): int
    {
        $content = self::THEME_XML;
        $content = str_replace(
            '{{THEME_NAME}}',
            str_replace('/', ' ', $themeName . ' theme'),
            $content
        );
        $content = str_replace(
            '{{PARENT_THEME_NAME}}',
            $parentThemeName,
            $content
        );
        $destinationPath = $this->getThemePath($themeName);

        return $this->appWrite->writeFile(
            $destinationPath . DIRECTORY_SEPARATOR . 'theme.xml',
            $content
        );
    }

    /**
     * @param string $themeName
     * @param string $parentThemePackageName
     * @return int
     * @throws FileSystemException
     */
    public function generateComposerJson(string $themeName, string $parentThemePackageName): int
    {
        // local/argento-stripes-custom
        // swissup/theme-frontend-argento-stripe
        $content = self::THEME_COMPOSER_TEMPLATE;
        $content = str_replace(
            '{{THEME_PACKAGE_NAME}}',
            strtolower($themeName),
            // strtolower($themeName) . '-custom',
            $content
        );
        $content = str_replace(
            '{{PARENT_THEME_PACKAGE_NAME}}',
            $parentThemePackageName,
            $content
        );
        $destinationPath = $this->getThemePath($themeName);

        return $this->appWrite->writeFile(
            $destinationPath . DIRECTORY_SEPARATOR . 'composer.json',
            $content
        );
    }

    /**
     * @param string $themeName
     * @return int
     * @throws FileSystemException
     */
    public function generateCustomCss(string $themeName): int
    {
        $content = '/* Autogenerated  */';
        $destinationPath = $this->getThemePath($themeName);

        return $this->appWrite->writeFile(
            $destinationPath . DIRECTORY_SEPARATOR . 'web/css/source/_argento_custom.less',
            $content
        );
    }

    /**
     * @param $themeName
     * @return string
     */
    protected function getThemePath($themeName): string
    {
        return $destinationPath = $this->appRead->getAbsolutePath(
            ThemeCreateCommand::THEME_DIR . DIRECTORY_SEPARATOR . $themeName
        );
    }
}
