<?php

namespace Swissup\AddressValidation\Model\Service;

use Magento\Framework\App\CacheInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Quote\Api\Data\AddressInterface;
use Magento\Store\Model\ScopeInterface;
use Psr\Log\LoggerInterface;
use Swissup\AddressValidation\Model\ServiceContext;
use Swissup\AddressValidation\Model\Result;

abstract class AbstractService implements ServiceInterface
{
    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * @var CacheInterface
     */
    protected $cache;

    public function __construct(ServiceContext $context)
    {
        $this->scopeConfig = $context->getScopeConfig();
        $this->logger = $context->getLogger();
        $this->cache = $context->getCache();
    }

    public function validate(AddressInterface $address): Result
    {
        try {
            return $this->_validate($address);
        } catch (\Exception $e) {
            $this->logException($e);
            throw $e;
        }
    }

    protected function _validate(AddressInterface $address): Result
    {
        throw new \Exception('_validate method is not implemented');
    }

    /**
     * Get specific config value
     *
     * @param  string $path
     * @param  string $scope
     * @return string
     */
    protected function getConfigValue($path, $scope = ScopeInterface::SCOPE_STORE)
    {
        return $this->scopeConfig->getValue($path, $scope);
    }

    /**
     * @param  \Throwable $e
     * @return void
     */
    protected function logException(\Throwable $e)
    {
        $this->logger->error('Swissup_AddressValidation: ' . $e->getMessage());
    }

    /**
     * @param  string $key
     * @return mixed
     */
    protected function loadCache(string $key)
    {
        if (!$cached = $this->cache->load($key)) {
            return false;
        }

        return $this->unserialize($cached);
    }

    protected function saveCache(array $data, string $key)
    {
        if ($cached = $this->serialize($data)) {
            $this->cache->save(
                $cached,
                $key,
                [],
                60 * 10
            );
        }
    }

    protected function serialize(array $data)
    {
        return json_encode($data);
    }

    protected function unserialize(string $data)
    {
        return json_decode($data, true);
    }

    protected function getCacheKey(string $data)
    {
        return 'swissup_address_validation_' . sha1($data);
    }
}
