define([
    'jquery',
    'underscore',
    'mage/translate',
    'Magento_Ui/js/modal/modal',
    'Magento_Customer/js/customer-data'
], function ($, _, $t, modal, customerData) {
    'use strict';

    var popup,
        countryData = customerData.get('directory-data');

    if (_.isEmpty(countryData())) {
        customerData.reload(['directory-data'], false);
    }

    popup = {
        el: false,
        templates: {
            main: _.template(`
                <%= error %>
                <p>${$t('The address could not be verified. Please double-check the address before submitting the order. If the address is correct, just place the order without making any changes to it.')}</p>
                <ul>
                    <li class="field choice">
                        <input type="radio"
                            class="radio"
                            name="address-verification"
                            id="address-verification-skip"
                            value="skip"
                            checked="checked"/>
                        <label for="address-verification-skip">${$t('Use entered address')}</label>
                    </li>
                    <li class="field choice">
                        <input type="radio"
                            class="radio"
                            name="address-verification"
                            id="address-verification-edit"
                            value="edit"/>
                        <label for="address-verification-edit">${$t('Edit entered address')}</label>
                    </li>
                    <%= suggestions %>
                </ul>
            `),
            suggestion: _.template(`
                <li class="field choice">
                    <input type="radio" class="radio" name="address-verification" id="<%= id %>" value='<%= value %>'/>
                    <label for="<%= id %>"><%= label %></label>
                </li>
            `),
            error: _.template(`
                <div class="message error">
                    <div><%= error %></div>
                </div>
            `)
        },

        create: function () {
            var element = $('<div>').appendTo($('body'));

            $(document).on('click', '[name="address-verification"]', this.updateButtons.bind(this));

            modal({
                type: 'popup',
                responsive: true,
                innerScroll: true,
                trigger: '.show-modal',
                modalClass: 'sw-validation-popup',
                title: 'Address Verification',
                buttons: [{
                    text: $t('Edit Address'),
                    class: 'action primary edit-address',
                    click: editAddress
                }, {
                    text: $t('Proceed using selected option'),
                    class: 'action primary proceed',
                    click: proceed
                }, {
                    text: $t('Place order using selected option'),
                    class: 'action primary place-order',
                    click: proceed
                }]
            }, element);

            return element;
        },

        open: function (data) {
            var html,
                error = '',
                self = this;

            if (!this.el) {
                this.el = this.create();
            }

            if (data.error) {
                error = self.templates.error({
                    error: data.error
                });
            }

            html = _.reduce(data.suggestions, function (memo, suggestion) {
                var label = _.values(suggestion).join(' ');

                return memo + self.templates.suggestion({
                    id: 'verified-' + btoa(label),
                    label: label,
                    value: JSON.stringify(suggestion)
                });
            }, '');

            this.el.html(self.templates.main({
                error: error,
                suggestions: html
            }));

            this.updateButtons();

            this.el.modal('openModal');
        },

        close: function () {
            var instance;

            if (!this.isOpened()) {
                return;
            }

            instance = this.el.modal('instance');
            instance.focussedElement = false; // prevent laggy scroll
            instance.options.transitionEvent = false; // close without delay
            instance.closeModal();
        },

        isOpened: function () {
            return this.el && this.el.modal('instance').options.isOpen;
        },

        /**
         * 💩 Persist selected option for a few seconds to make it possible to
         * read it just right after the modal is closed. But not when it
         * closed for a while.
         */
        persistSelectedOption: function () {
            this.selectedOption = this.getSelectedOption();
            setTimeout(function () {
                this.selectedOption = false;
            }.bind(this), 1000);
        },

        getSelectedOption: function () {
            if (this.selectedOption) {
                return this.selectedOption;
            }

            if (!this.isOpened()) {
                return false;
            }

            return this.el.find('[name="address-verification"]:checked').val();
        },

        updateButtons: function () {
            var selected = this.getSelectedOption(),
                parent = $('.sw-validation-popup'),
                show = '.place-order';

            if ($('[data-role="opc-continue"]:visible').length) {
                show = '.proceed';
            }

            if (selected === 'edit') {
                show = '.edit-address';
            }

            parent.find('.modal-footer button').hide();
            parent.find(show).show();
        }
    }

    function editAddress() {
        popup.close();

        $('.form-shipping-address')[0].scrollIntoView({
            behavior: 'smooth'
        });

        setTimeout(function () {
            $('.form-shipping-address input:visible')[0].focus();
        }, 400);
    }

    function proceed() {
        try {
            fillAddress(JSON.parse(popup.getSelectedOption()));
        } catch (e) {
            // selected option is not verified address
        }

        // warkaround to read selected option a few moments later.
        // @see shipping-payload-extender-mixin.js
        popup.persistSelectedOption();
        popup.close();

        $([
            '[data-role="opc-continue"]:visible',
            '.actions-toolbar .checkout:visible',
        ].join(',')).first().click();
    }

    function findRegionId(address) {
        return _.findKey(countryData()[address.country_id].regions, function (region) {
            return region.code === address.region_code;
        });
    }

    function fillAddress(address) {
        var mapping = {
            'street1': '[name="street[0]"],#street_1',
            'street2': '[name="street[1]"],#street_2',
            'street3': '[name="street[2]"],#street_3',
            'street4': '[name="street[3]"],#street_4'
        };

        if (!address.country_id) {
            address.country_id = $('[name="country_id"]', '.form-shipping-address').val();
        }

        if (address.region_code && !address.region_id) {
            address.region_id = findRegionId(address);
        }

        _.each(address, function (value, key) {
            var selector = mapping[key] ? mapping[key] : '[name="' + key + '"]',
                el = $(selector, '.form-shipping-address');

            // eslint-disable-next-line eqeqeq
            if (!el.length || el.val() == value) {
                return;
            }

            if (el.is('select') && !el.find('option[value=' + value + ']').length) {
                return;
            }

            el.val(value).trigger('change');
        });
    }

    return {
        getSelectedOption: popup.getSelectedOption.bind(popup),

        processResult: function (data) {
            if (!_.isEmpty(data.autocorrect)) {
                fillAddress(data.autocorrect);
                proceed();
            } else {
                popup.open(data);
            }
        }
    };
});
