<?php

namespace Swissup\AddressValidation\Model\Service;

use Magento\Quote\Api\Data\AddressInterface;
use Swissup\AddressValidation\Model\Result;
use Swissup\AddressValidation\Model\ServiceContext;

class Usps extends AbstractService
{
    private $httpClientFactory;

    private $autocorrect = [];

    public function __construct(
        ServiceContext $context,
        \Magento\Framework\HTTP\ZendClientFactory $httpClientFactory
    ) {
        parent::__construct($context);
        $this->httpClientFactory = $httpClientFactory;
    }

    public function canValidate(AddressInterface $address): bool
    {
        if ($address->getExtensionAttributes()->getSwissupValidation() === 'skip') {
            return false;
        }

        if ($address->getCountryId() !== 'US') {
            return false;
        }

        return true;
    }

    /**
     * @throws \Exception  Exception is throws when service was not able to
     *                     validate address for some reason.
     */
    protected function _validate(AddressInterface $address): Result
    {
        $this->preValidate($address);

        $xml = $this->generateRequestString($address);
        $key = $this->getCacheKey($xml);

        if (!$response = $this->loadCache($key)) {
            $url = $this->getConfigValue('carriers/usps/gateway_secure_url');
            $client = $this->httpClientFactory->create();
            $client->setUri($url);
            $client->setConfig(['maxredirects' => 0, 'timeout' => 30]);
            $client->setParameterGet('API', 'Verify');
            $client->setParameterGet('XML', $xml);
            $response = $client->request()->getBody();
            $response = $this->parseResponse($response);
            $this->postValidate($response);
            $this->saveCache($response, $key);
        }

        $originalData = [
            'street1' => $address->getStreet()[0],
            'street2' => $address->getStreet()[1] ?? '',
            'city' => $address->getCity(),
            'region_code' => $address->getRegionCode(),
            'postcode' => $address->getPostcode(),
        ];

        return (new Result())
            ->setIsValid($this->isValid($originalData, $response))
            ->setAutocorrect($this->autocorrect)
            ->setOriginalData($originalData)
            ->setError($response[0]['error'] ?? '')
            ->setSuggestions($response[0]['suggestions'] ?? []);
    }

    private function isValid(array $originalData, array $response): bool
    {
        $this->autocorrect = [];

        if (!empty($response[0]['error'])) {
            return false;
        }

        if (empty($response[0]['suggestions'])) {
            return true;
        }

        $suggestionsCount = count($response[0]['suggestions']);
        foreach ($response[0]['suggestions'] as $suggestion) {
            $diff = array_udiff($originalData, $suggestion, 'strcasecmp');

            if (!$diff) {
                return true; // one of suggestions is the same
            }

            // if entered zip starts from `zip5-` we can use autocorrection
            if ($suggestionsCount === 1 &&
                count($diff) === 1 &&
                isset($diff['postcode'])
            ) {
                $originalZip5 = $originalData['postcode'] . '-';
                if (stripos($suggestion['postcode'], $originalZip5) === 0) {
                    $this->autocorrect['postcode'] = $suggestion['postcode'];
                }
            }
        }

        return false;
    }

    private function generateRequestString(AddressInterface $address): string
    {
        $street = $address->getStreet() ?: [];
        $zip5 = (string) $address->getPostcode();
        $zip4 = '';

        if (strpos($zip5, '-') !== false) {
            list($zip5, $zip4) = explode('-', $zip5);
        }

        $userId = $this->getConfigValue('carriers/usps/userid');
        $result = '<AddressValidateRequest USERID="' . $userId . '">';
        $result .= '<Address ID="0">';

        $mapping = [
            'Address1' => $street[1] ?? '',
            'Address2' => $street[0] ?? '',
            'City' => (string) $address->getCity(),
            'State' => (string) $address->getRegionCode(),
            'Zip5' => $zip5,
            'Zip4' => $zip4,
        ];
        foreach ($mapping as $tagName => $value) {
            $result .= "<{$tagName}><![CDATA[{$value}]]></{$tagName}>";
        }

        $result .= '</Address>';
        $result .= '</AddressValidateRequest>';

        return $result;
    }

    private function parseResponse($text)
    {
        $xml = simplexml_load_string($text);

        if (!$xml) {
            return ['error' => 'Unable to parse USPS response'];
        }

        if ($xml->getName() === 'Error') {
            return [
                'number' => (string) $xml->Number,
                'error' => (string) $xml->Description,
            ];
        }

        if (!$xml->Address) {
            return ['error' => 'Unknown response format. <Address> is missing.'];
        }

        $i = 0;
        $result = [];
        foreach ($xml->Address as $address) {
            if ($address->Error) {
                $result[(string) $address->attributes()->ID]['error'] =
                    trim((string) $address->Error->Description);

                continue;
            }

            $postcode = $address->Zip5 ? (string) $address->Zip5 : '';
            if ($address->Zip4) {
                if ($postcode) {
                    $postcode .= '-';
                }
                $postcode .= (string) $address->Zip4;
            }

            $result[(string) $address->attributes()->ID]['suggestions'][$i++] = [
                'street1' => $address->Address2 ? (string) $address->Address2 : '',
                'street2' => $address->Address1 ? (string) $address->Address1 : '',
                'city' => $address->City ? (string) $address->City : '',
                'region_code' => $address->State ? (string) $address->State : '',
                'postcode' => $postcode,
            ];
        }

        return $result;
    }

    private function preValidate(AddressInterface $address)
    {
        if (!$this->getConfigValue('carriers/usps/userid')) {
            throw new \Exception('USPS userid is missing');
        }
    }

    private function postValidate($response)
    {
        if (isset($response['error'])) {
            throw new \Exception($response['error']);
        }
    }
}
