<?php

/**
 * PAYONE Magento 2 Connector is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * PAYONE Magento 2 Connector is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with PAYONE Magento 2 Connector. If not, see <http://www.gnu.org/licenses/>.
 *
 * PHP version 5
 *
 * @category  Payone
 * @package   Payone_Magento2_Plugin
 * @author    FATCHIP GmbH <support@fatchip.de>
 * @copyright 2003 - 2016 Payone GmbH
 * @license   <http://www.gnu.org/licenses/> GNU Lesser General Public License
 * @link      http://www.payone.de
 */

namespace Payone\Core\Model\Methods;

use Payone\Core\Model\PayoneConfig;
use Magento\Sales\Model\Order;
use Magento\Framework\DataObject;

/**
 * Model for creditcard payment method
 */
class Creditcard extends PayoneMethod
{
    /**
     * Payment method code
     *
     * @var string
     */
    protected $_code = PayoneConfig::METHOD_CREDITCARD;

    /**
     * Info instructions block path
     *
     * @var string
     */
    protected $_infoBlockType = 'Payone\Core\Block\Info\Creditcard';

    /**
     * Clearingtype for PAYONE authorization request
     *
     * @var string
     */
    protected $sClearingtype = 'cc';

    /**
     * Determines if the redirect-parameters have to be added
     * to the authorization-request
     *
     * @var bool
     */
    protected $blNeedsRedirectUrls = true;

    /**
     * Keys that need to be assigned to the additionalinformation fields
     *
     * @var array
     */
    protected $aAssignKeys = [
        'pseudocardpan',
        'truncatedcardpan',
        'cardtype',
        'cardexpiredate',
        'selectedData',
        'cardholder',

    ];

    /**
     * Return parameters specific to this payment type
     *
     * @param  Order $oOrder
     * @return array
     */
    public function getPaymentSpecificParameters(Order $oOrder)
    {
        $aReturn = ['pseudocardpan' => $this->getInfoInstance()->getAdditionalInformation('pseudocardpan')];
        $sSelectedData = $this->getInfoInstance()->getAdditionalInformation('selectedData');
        if (!empty($sSelectedData) && $sSelectedData != 'new') {
            $aReturn['pseudocardpan'] = $this->getInfoInstance()->getAdditionalInformation('selectedData');
        }
        $aReturn['cardholder'] = $this->getInfoInstance()->getAdditionalInformation('cardholder');
        return $aReturn;
    }

    /**
     * Add the checkout-form-data to the checkout session
     *
     * @param  DataObject $data
     * @return $this
     */
    public function assignData(DataObject $data)
    {
        parent::assignData($data);

        $oInfoInstance = $this->getInfoInstance();
        foreach ($this->aAssignKeys as $sKey) {
            $sData = $this->toolkitHelper->getAdditionalDataEntry($data, $sKey);
            if ($sData) {
                $oInfoInstance->setAdditionalInformation($sKey, $sData);
            }
        }

        $aAddData = $data->getAdditionalData();
        if (isset($aAddData['saveData']) && $aAddData['saveData'] == '1') {
            $this->handlePaymentDataStorage($data);
        }

        return $this;
    }

    /**
     * Add value to the payment storage data array
     *
     * @param  array  $aDest
     * @param  array  $aSource
     * @param  string $sDestField
     * @param  string $sSourceField
     * @return void
     */
    protected function addValueToArray(&$aDest, $aSource, $sDestField, $sSourceField)
    {
        if (isset($aSource[$sSourceField])) {
            $aDest[$sDestField] = $aSource[$sSourceField];
        }
    }

    /**
     * Convert DataObject to needed array format
     *
     * @param  DataObject $data
     * @return array
     */
    protected function getPaymentStorageData(DataObject $data)
    {
        $aReturn = parent::getPaymentStorageData($data);
        $aAdditionalData = $data->getAdditionalData();

        if (isset($aAdditionalData['pseudocardpan']) && isset($aAdditionalData['truncatedcardpan'])) {
            $this->addValueToArray($aReturn, $aAdditionalData, 'cardpan', 'pseudocardpan');
            $this->addValueToArray($aReturn, $aAdditionalData, 'masked', 'truncatedcardpan');
            $this->addValueToArray($aReturn, $aAdditionalData, 'cardholder', 'cardholder');
            $this->addValueToArray($aReturn, $aAdditionalData, 'cardtype', 'cardtype');
            $this->addValueToArray($aReturn, $aAdditionalData, 'cardexpiredate', 'cardexpiredate');
        }
        return $aReturn;
    }
}
