<?php
/**
 * Copyright © 2017 MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\SeoRedirects\Model\Redirect\CustomRedirect;

use MageWorx\SeoRedirects\Model\ResourceModel\Redirect\CustomRedirect\CollectionFactory as CustomRedirectCollectionFactory;
use MageWorx\SeoRedirects\Model\Redirect\CustomRedirectFactory;
use MageWorx\SeoRedirects\Model\Redirect\CustomRedirect;
use MageWorx\SeoRedirects\Model\Redirect\Source\CustomRedirect\RedirectTypeRewriteFragment as RedirectTypeRewriteFragmentSource;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Message\ManagerInterface;

abstract class RedirectWizardByDeletedUrlAbstract
{
    /**
     * @var \Magento\Framework\Message\ManagerInterface
     */
    protected $messageManager;

    /**
     * @var CustomRedirectFactory
     */
    protected $customRedirectFactory;

    /**
     * @var CustomRedirectCollectionFactory
     */
    protected $customRedirectCollectionFactory;

    /** @var RedirectTypeRewriteFragmentSource */
    protected $redirectTypeRewriteFragmentSource;

    /**
     * @var \MageWorx\SeoRedirects\Helper\Data
     */
    protected $helperData;

    /**
     * @var array
     */
    protected $loadedCustomRedirects = [];

    /**
     * @var int
     */
    protected $redirectEntityType;

    /**
     * RedirectWizardByDeletedUrlAbstract constructor.
     *
     * @param CustomRedirectCollectionFactory $customRedirectCollectionFactory
     * @param CustomRedirectFactory $customRedirectFactory
     * @param RedirectTypeRewriteFragmentSource $redirectTypeRewriteFragmentSource
     * @param ManagerInterface $messageManager
     * @param \MageWorx\SeoRedirects\Helper\Data $helperData
     */
    public function __construct(
        CustomRedirectCollectionFactory $customRedirectCollectionFactory,
        CustomRedirectFactory $customRedirectFactory,
        RedirectTypeRewriteFragmentSource $redirectTypeRewriteFragmentSource,
        ManagerInterface $messageManager,
        \MageWorx\SeoRedirects\Helper\CustomRedirect\Data $helperData
    ) {
        $this->customRedirectFactory             = $customRedirectFactory;
        $this->customRedirectCollectionFactory   = $customRedirectCollectionFactory;
        $this->redirectTypeRewriteFragmentSource = $redirectTypeRewriteFragmentSource;
        $this->messageManager                    = $messageManager;
        $this->helperData                        = $helperData;
    }

    /**
     * @api
     * @param int $entityId
     * @param \Magento\UrlRewrite\Service\V1\Data\UrlRewrite[] $currentUrlRewrites
     */
    public function process($entityId, $currentUrlRewrites)
    {
        if (!$entityId) {
            return;
        }

        $this->processByDeletedRequestEntity($entityId, $currentUrlRewrites);
        $this->processByDeletedTargetEntity($entityId, $currentUrlRewrites);
    }

    /**
     * @param int $entityId
     * @param \Magento\UrlRewrite\Service\V1\Data\UrlRewrite[] $currentUrlRewrites
     */
    protected function processByDeletedRequestEntity($entityId, $currentUrlRewrites)
    {
        /** @var  $rewrite \Magento\UrlRewrite\Service\V1\Data\UrlRewrite */
        foreach ($currentUrlRewrites as $rewrite) {

            $storeId = $rewrite->getStoreId();
            $key     = $entityId . '_' . $storeId;

            if (empty($this->loadedCustomRedirects[$key])) {
                $requestCollection = $this->customRedirectCollectionFactory->create();
                $requestCollection->addRequestEntityFilter($entityId, $this->redirectEntityType);
                $requestCollection->addStoreFilter($storeId);

                /** @var \MageWorx\SeoRedirects\Model\Redirect\CustomRedirect $oldCustomRedirect */
                $oldCustomRedirect = $requestCollection->getFirstItem();

                $this->loadedCustomRedirects[$key] = $oldCustomRedirect->getId() ? $oldCustomRedirect : false;
            }

            /** @var \MageWorx\SeoRedirects\Model\Redirect\CustomRedirect $oldCustomRedirect |false */
            $oldCustomRedirect = $this->loadedCustomRedirects[$key];

            if ($oldCustomRedirect) {
                $newCustomRedirect = $this->customRedirectFactory->create();
                $newCustomRedirect->setRequestEntityType(CustomRedirect::REDIRECT_TYPE_CUSTOM);
                $newCustomRedirect->setRequestEntityIdentifier($rewrite->getRequestPath());
                $newCustomRedirect->setTargetEntityType($oldCustomRedirect->getTargetEntityType());
                $newCustomRedirect->setTargetEntityIdentifier($oldCustomRedirect->getTargetEntityIdentifier());
                $newCustomRedirect->setIsAutogenerated(true);
                $newCustomRedirect->setStoreId($oldCustomRedirect->getStoreId());
                $newCustomRedirect->setRedirectCode($oldCustomRedirect->getRedirectCode());
                $newCustomRedirect->setStatus(CustomRedirect::STATUS_ENABLED);

                try {
                    $newCustomRedirect->getResource()->save($newCustomRedirect);
                } catch (LocalizedException $e) {
                    $this->messageManager->addNoticeMessage($e->getMessage());
                }
            }
        }

        $redirectsForDelete = array_filter($this->loadedCustomRedirects);

        foreach ($redirectsForDelete as $redirect) {
            $redirect->delete();
        }
    }

    /**
     * @param int $entityId
     * @param \Magento\UrlRewrite\Service\V1\Data\UrlRewrite[] $currentUrlRewrites
     */
    protected function processByDeletedTargetEntity($entityId, $currentUrlRewrites)
    {
        foreach ($currentUrlRewrites as $rewrite) {

            $storeId = $rewrite->getStoreId();

            if ($this->getIsModifyRedirectByTargetEntity($rewrite)) {
                $requestCollection = $this->customRedirectCollectionFactory->create();
                $requestCollection->addTargetEntityFilter($entityId, $this->redirectEntityType);
                $requestCollection->addStoreFilter($storeId);

                /** @var \MageWorx\SeoRedirects\Model\Redirect\CustomRedirect $upgradeableCustomRedirect */
                foreach ($requestCollection as $upgradeableCustomRedirect) {
                    $upgradeableCustomRedirect->setTargetEntityType(CustomRedirect::REDIRECT_TYPE_CUSTOM);
                    $upgradeableCustomRedirect->setTargetEntityIdentifier(
                        $this->prepareTargetUrl($rewrite->getRequestPath(), $rewrite->getEntityId(), $storeId)
                    );
                    $upgradeableCustomRedirect->getResource()->save($upgradeableCustomRedirect);
                }
            }
        }
    }

    /**
     * @param \Magento\UrlRewrite\Service\V1\Data\UrlRewrite $rewrite
     * @return bool
     */
    protected function getIsModifyRedirectByTargetEntity($rewrite)
    {
        if (!$rewrite->getIsAutogenerated()) {
            return false;
        }

        $redirectTypeRewriteFragmentSource = $this->redirectTypeRewriteFragmentSource->toArray();

        if (empty($redirectTypeRewriteFragmentSource[$this->redirectEntityType])) {
            return false;
        }

        if (strpos($rewrite->getTargetPath(), $redirectTypeRewriteFragmentSource[$this->redirectEntityType]) !== 0) {
            return false;
        }

        return true;
    }

    /**
     * @param $url
     * @param int|null $entityId
     * @param int|null $storeId
     * @return mixed
     */
    protected function prepareTargetUrl($url, $entityId = null, $storeId = null)
    {
        return $url;
    }
}
