<?php
/**
 * Copyright © 2015 MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\SeoBase\Model\ResourceModel\Catalog\Product;

use Magento\CatalogUrlRewrite\Model\ProductUrlRewriteGenerator;

class CrossDomain extends \MageWorx\SeoBase\Model\ResourceModel\Catalog\Product
{
    /**
     * @var \MageWorx\SeoBase\Helper\Data
     */
    protected $helperData;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var \Magento\Catalog\Model\Product\Visibility
     */
    protected $productVisibility;

    /**
     * @var \Magento\Catalog\Model\Product\Attribute\Source\Status
     */
    protected $productStatus;

    /**
     * @var \Magento\Eav\Model\ConfigFactory
     */
    protected $eavConfigFactory;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $helperStoreUrl;

    /**
     * @var \MageWorx\SeoBase\Model\ResourceModel\Catalog\Product\FlexibleCanonical
     */
    protected $flexibleCanonicalResource;

    /**
     * @var \Magento\Framework\DataObject\Factory
     */
    protected $objectFactory;

    /**
     * CrossDomain constructor.
     *
     * @param \Magento\Framework\Model\ResourceModel\Db\Context $context
     * @param \MageWorx\SeoBase\Helper\Data $helperData
     * @param \Magento\Catalog\Model\ResourceModel\Product $productResource
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Catalog\Model\Product\Visibility $productVisibility
     * @param \Magento\Catalog\Model\Product\Attribute\Source\Status $productStatus
     * @param \Magento\Eav\Model\ConfigFactory $eavConfigFactory
     * @param \MageWorx\SeoBase\Helper\StoreUrl $helperStoreUrl
     * @param \Magento\Framework\DataObject\Factory $objectFactory
     * @param \MageWorx\SeoAll\Helper\LinkFieldResolver $linkFieldResolver
     * @param FlexibleCanonical $flexibleCanonical
     * @param null $resourcePrefix
     */
    public function __construct(
        \Magento\Framework\Model\ResourceModel\Db\Context $context,
        \MageWorx\SeoBase\Helper\Data $helperData,
        \Magento\Catalog\Model\ResourceModel\Product $productResource,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Catalog\Model\Product\Visibility $productVisibility,
        \Magento\Catalog\Model\Product\Attribute\Source\Status $productStatus,
        \Magento\Eav\Model\ConfigFactory $eavConfigFactory,
        \MageWorx\SeoBase\Helper\StoreUrl $helperStoreUrl,
        \Magento\Framework\DataObject\Factory $objectFactory,
        \MageWorx\SeoAll\Helper\LinkFieldResolver $linkFieldResolver,
        \MageWorx\SeoBase\Model\ResourceModel\Catalog\Product\FlexibleCanonical $flexibleCanonical,
        $resourcePrefix = null
    ) {
        $this->storeManager              = $storeManager;
        $this->objectFactory             = $objectFactory;
        $this->productVisibility         = $productVisibility;
        $this->productStatus             = $productStatus;
        $this->eavConfigFactory          = $eavConfigFactory;
        $this->helperData                = $helperData;
        $this->helperStoreUrl            = $helperStoreUrl;
        $this->flexibleCanonicalResource = $flexibleCanonical;

        parent::__construct($context, $productResource, $linkFieldResolver, $resourcePrefix);
    }

    /**
     * @return void
     */
    protected function _construct()
    {
        $this->_init('catalog_product_entity', 'entity_id');
    }

    /**
     * Get category collection array
     *
     * @param null|string|bool|int|\Magento\Store\Model\Store $storeId
     * @return array|bool
     */
    public function getCrossDomainData($productId, $storeId, $categoryId = null)
    {
        if (is_null($storeId)) {
            return false;
        }

        /* @var $store \Magento\Store\Model\Store */
        $store = $this->storeManager->getStore($storeId);

        if (!$store) {
            return false;
        }
        $adapter = $this->getConnection();;

        $this->select = $adapter->select()->from(
            ['e' => $this->getMainTable()],
            [$this->getIdFieldName()]
        )->joinInner(
            ['w' => $this->getTable('catalog_product_website')],
            'e.entity_id = w.product_id',
            []
        )->joinLeft(
            ['url_rewrite' => $this->getTable('url_rewrite')],
            'e.entity_id = url_rewrite.entity_id AND url_rewrite.is_autogenerated = 1'
            . $adapter->quoteInto(' AND url_rewrite.store_id = ?', $store->getId())
            . $adapter->quoteInto(' AND url_rewrite.entity_type = ?', ProductUrlRewriteGenerator::ENTITY_TYPE),
            ['url' => 'request_path']
        )->joinLeft(
            ['catalog_url_rewrite' => $this->getTable('catalog_url_rewrite_product_category')],
            'url_rewrite.url_rewrite_id = catalog_url_rewrite.url_rewrite_id',
            []
        )
        ->where(
            'e.entity_id = ?',
            $productId
        )
        ->where(
            'w.website_id = ?',
            $store->getWebsiteId()
        );

        $this->flexibleCanonicalResource->addFlexibleConditions($this->select, $storeId);

        $this->addFilter($store->getId(), 'visibility', $this->productVisibility->getVisibleInSiteIds(), 'in');
        $this->addFilter($store->getId(), 'status', $this->productStatus->getVisibleStatusIds(), 'in');

        $query = $adapter->query($this->select);

        $row = $query->fetch();

        if (!is_array($row)) {
            return false;
        }

        return $this->prepareProduct($row, $store->getId(), $categoryId);
    }

    /**
     * Prepare and convert cross domain data to object
     *
     * @param array $productRow
     * @param int $storeId
     * @return \Magento\Framework\Object
     */
    protected function prepareProduct(array $productRow, $storeId)
    {
        if (empty($productRow['url'])) {
            return false;
        }
        $product = $this->objectFactory->create();
        $product->addData($productRow);
        $product['id']       = $productRow[$this->getIdFieldName()];
        $product['store_id'] = $storeId;
        $product->setUrl($this->helperStoreUrl->getUrl($productRow['url'], $storeId, true));

        return $product;
    }
}
