<?php
/**
 * Copyright © 2016 MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\SeoBase\Model\ResourceModel\Catalog\Category;

use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\CatalogUrlRewrite\Model\CategoryUrlRewriteGenerator;
use Magento\Store\Model\Store;

/**
 * SEO Base resource category hreflang URLs
 */
class Hreflangs extends \MageWorx\SeoBase\Model\ResourceModel\Catalog\Category
{

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var \MageWorx\SeoBase\Helper\StoreUrl
     */
    protected $helperStoreUrl;

    /**
     * @var \MageWorx\SeoBase\Helper\Data
     */
    protected $helperData;

    /**
     * @param \Magento\Framework\Model\ResourceModel\Db\Context $context
     * @param \Magento\Catalog\Model\ResourceModel\Category $categoryResource
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \MageWorx\SeoBase\Helper\StoreUrl $helperStoreUrl
     * @param \MageWorx\SeoBase\Helper\Data $helperData
     * @param string|null $resourcePrefix
     */
    public function __construct(
        \Magento\Framework\Model\ResourceModel\Db\Context $context,
        \Magento\Catalog\Model\ResourceModel\Category $categoryResource,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \MageWorx\SeoBase\Helper\StoreUrl $helperStoreUrl,
        \MageWorx\SeoBase\Helper\Data $helperData,
        \MageWorx\SeoAll\Helper\LinkFieldResolver $linkFieldResolver,
        $resourcePrefix = null
    ) {
        $this->storeManager   = $storeManager;
        $this->helperStoreUrl = $helperStoreUrl;
        $this->helperData     = $helperData;
        parent::__construct($context, $categoryResource, $linkFieldResolver, $resourcePrefix);
    }

    /**
     * Retrieve array hreflang URLs:
     * [
     *      (int)itemId => [
     *          'identifier'   => (string)item URL identifier (URL key),
     *          'hreflangUrls' => [
     *              (int)storeId => (string)item store URL
     *          ]
     *       ]
     * ]
     *
     * @param array $storeIds
     * @param array $categoryIds
     * @param bool $isUrlPathsNeeded
     * @return array
     */
    public function getHreflangsData($storeIds, array $categoryIds, $isUrlPathsNeeded = false)
    {
        $adapter         = $this->getConnection();
        $attributeStatus = $this->getAttribute('is_active');
        $linkField       = $this->linkFieldResolver->getLinkField(CategoryInterface::class, 'entity_id');

        $this->select = $adapter->select()->from(
            ['e' => $this->getMainTable()],
            [$this->getIdFieldName(), 'parent_id', 'path', 'position', 'level', 'children_count']
        )->joinLeft(
            ['url_rewrite' => $this->getTable('url_rewrite')],
            'e.entity_id = url_rewrite.entity_id AND url_rewrite.is_autogenerated = 1'
            . $adapter->quoteInto(' AND url_rewrite.store_id IN(?)', $storeIds)
            . $adapter->quoteInto(' AND url_rewrite.entity_type = ?', CategoryUrlRewriteGenerator::ENTITY_TYPE),
            ['store_id', 'request_path']
        )->joinLeft(
            ['status_tab' => $attributeStatus['table']],
            $adapter->quoteInto('status_tab.attribute_id = ?', $attributeStatus['attribute_id'])
            . " AND e.$linkField = status_tab.$linkField AND url_rewrite.store_id = status_tab.store_id",

            []
        )->joinLeft(
            ['status_tab_2' => $attributeStatus['table']],
            $adapter->quoteInto('status_tab_2.attribute_id = ?', $attributeStatus['attribute_id'])
            . " AND e.$linkField = status_tab_2.$linkField"
            . $adapter->quoteInto(' AND status_tab_2.store_id = ?', Store::DEFAULT_STORE_ID),
            []
        )->where(
            'e.entity_id IN (?)',
            $categoryIds
        )->where(
            'CASE WHEN status_tab.value IS NULL THEN status_tab_2.value = ? ELSE status_tab.value = ? END',
            1
        );

        $query = $adapter->query($this->select);
        $rows  = $query->fetchAll();

        if (!is_array($rows)) {
            return false;
        }

        $data = [];
        foreach ($rows as $row) {
            if (array_key_exists($row['entity_id'], $data)) {
                $hreflangUrls = $data[$row['entity_id']]['hreflangUrls'];
            } else {
                $data[$row['entity_id']] = [];
                $hreflangUrls            = [];
            }

            if ($isUrlPathsNeeded) {
                $url = $row['request_path'];
                $hreflangUrls[$row['store_id']] = $url;
            } else {
                $url = $this->helperStoreUrl->getUrl($row['request_path'], $row['store_id'], true);
                if (!in_array($url, $hreflangUrls)) {
                    $hreflangUrls[$row['store_id']] = $url;
                }
            }

            $data[$row['entity_id']] = ['requestPath' => $row['request_path'], 'hreflangUrls' => $hreflangUrls];
        }

        return $data;
    }
}
