<?php
/**
 * Copyright © 2015 MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\SeoBase\Model\ResourceModel\Catalog\Category;

use Magento\CatalogUrlRewrite\Model\CategoryUrlRewriteGenerator;

/**
 * SEO Base resource cross domain category collection model
 */
class CrossDomain extends \MageWorx\SeoBase\Model\ResourceModel\Catalog\Category
{

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var \MageWorx\SeoBase\Helper\StoreUrl
     */
    protected $helperStoreUrl;

    /**
     * @var \MageWorx\SeoBase\Helper\Data
     */
    protected $helperData;

    /**
     * @var \Magento\Framework\DataObject\Factory
     */
    protected $dataObjectFactory;

    /**
     * @param \Magento\Framework\Model\ResourceModel\Db\Context $context
     * @param \Magento\Catalog\Model\ResourceModel\Category $categoryResource
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \MageWorx\SeoBase\Helper\StoreUrl $helperStoreUrl
     * @param \MageWorx\SeoBase\Helper\Data $helperData
     * @param string|null $resourcePrefix
     */
    public function __construct(
        \Magento\Framework\Model\ResourceModel\Db\Context $context,
        \Magento\Catalog\Model\ResourceModel\Category $categoryResource,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \MageWorx\SeoBase\Helper\StoreUrl $helperStoreUrl,
        \MageWorx\SeoBase\Helper\Data $helperData,
        \Magento\Framework\DataObject\Factory $dataObjectFactory,
        \MageWorx\SeoAll\Helper\LinkFieldResolver $linkFieldResolver,
        $resourcePrefix = null
    ) {
        $this->storeManager      = $storeManager;
        $this->dataObjectFactory = $dataObjectFactory;
        $this->helperStoreUrl    = $helperStoreUrl;
        $this->helperData        = $helperData;
        parent::__construct($context, $categoryResource, $linkFieldResolver, $resourcePrefix);
    }

    /**
     * Retrieve category cross domain data
     * @param int $categoryId
     * @param null|string|bool|int|\Magento\Store\Model\Store $storeId
     * @return array|bool
     */
    public function getCrossDomainData($categoryId, $storeId)
    {
        /* @var $store \Magento\Store\Model\Store */
        $store = $this->storeManager->getStore($storeId);
        if (!$store) {
            return false;
        }

        $adapter = $this->getConnection();

        $this->select = $adapter->select()->from(
            ['e' => $this->getMainTable()],
            [$this->getIdFieldName(), 'parent_id', 'path', 'position', 'level', 'children_count']
        )->joinLeft(
            ['url_rewrite' => $this->getTable('url_rewrite')],
            'e.entity_id = url_rewrite.entity_id AND url_rewrite.is_autogenerated = 1'
            . $adapter->quoteInto(' AND url_rewrite.store_id = ?', $store->getId())
            . $adapter->quoteInto(' AND url_rewrite.entity_type = ?', CategoryUrlRewriteGenerator::ENTITY_TYPE),
            ['url' => 'request_path']
        )->where(
            'e.entity_id = (?)',
            $categoryId
        );

        $this->addFilter($store->getId(), 'is_active', 1);

        $query = $adapter->query($this->select);
        $row = $query->fetch();

        if (!is_array($row)) {
            return false;
        }
        return $this->prepareCategory($row, $store->getId());
    }

    /**
     * Prepare and convert cross domain data to object
     *
     * @param array $categoryRow
     * @param null|string|bool|int|\Magento\Store\Model\Store $storeId
     * @return \Magento\Framework\Object
     */
    protected function prepareCategory(array $categoryRow, $storeId)
    {
        $category = $this->dataObjectFactory->create();
        $category->addData($categoryRow);
        $category->setId($categoryRow[$this->getIdFieldName()]);

        if (!empty($categoryRow['url'])) {
            $category->setUrl($this->helperStoreUrl->getUrl($categoryRow['url'], $storeId, true));
        } else {
            $category->setUrl(
                $this->helperStoreUrl->getUrl('catalog/category/view/id/' . $category->getId(), $storeId)
            );
        }
        return $category;
    }
}
