<?php
/**
 * Copyright © 2015 MageWorx. All rights reserved.
 * See LICENSE.txt for license details.
 */

namespace MageWorx\HtmlSitemap\Model\ResourceModel\Catalog;

use Magento\CatalogUrlRewrite\Model\CategoryUrlRewriteGenerator;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Catalog\Model\ResourceModel\Category as CategoryResource;
use Magento\Framework\Model\ResourceModel\Db\Context;
use MageWorx\HtmlSitemap\Helper\Data as SitemapHelper;
use MageWorx\HtmlSitemap\Helper\StoreUrl as StoreUrlHelper;
use MageWorx\HtmlSitemap\Model\Source;
use Magento\Catalog\Api\Data\CategoryInterface;

/**
 * HTML Sitemap resource catalog collection model
 */
class Category extends \Magento\Framework\Model\ResourceModel\Db\AbstractDb
{
    /**
     * Collection Zend Db select
     *
     * @var \Magento\Framework\DB\Select
     */
    protected $select;

    /**
     * Attribute cache
     *
     * @var array
     */
    protected $_attributesCache = [];

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var \Magento\Catalog\Model\ResourceModel\Category
     */
    protected $categoryResource;

    /**
     * Sitemap config data
     *
     * @var \MageWorx\HtmlSitemap\Helper\Data
     */
    protected $sitemapHelper;

    /**
     * @var \MageWorx\HtmlSitemap\Helper\StoreUrl
     */
    protected $storeUrlHelper;

    /**
     * @var \MageWorx\SeoAll\Helper\LinkFieldResolver
     */
    protected $linkFieldResolver;

    /**
     * @param \Magento\Framework\Model\ResourceModel\Db\Context $context
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Catalog\Model\ResourceModel\Category $categoryResource
     * @param \MageWorx\HtmlSitemap\Helper\Data $sitemapHelper
     * @param \MageWorx\HtmlSitemap\Helper\StoreUrl $storeUrlHelper
     * @param string|null $resourcePrefix
     */
    public function __construct(
        Context $context,
        StoreManagerInterface $storeManager,
        CategoryResource $categoryResource,
        SitemapHelper $sitemapHelper,
        StoreUrlHelper $storeUrlHelper,
        \MageWorx\SeoAll\Helper\LinkFieldResolver $linkFieldResolver,
        $resourcePrefix = null
    ) {
        parent::__construct($context, $resourcePrefix);
        $this->storeManager = $storeManager;
        $this->categoryResource = $categoryResource;
        $this->sitemapHelper = $sitemapHelper;
        $this->storeUrlHelper = $storeUrlHelper;
        $this->linkFieldResolver = $linkFieldResolver;
    }

    /**
     * @return void
     */
    protected function _construct()
    {
        $this->_init('catalog_category_entity', 'entity_id');
    }

    /**
     * Get category collection array
     *
     * @param null|string|bool|int|\Magento\Store\Model\Store $storeId
     * @return array|bool
     */
    public function getCollection($storeId = null)
    {
        /* @var $store \Magento\Store\Model\Store */
        $store = $this->storeManager->getStore($storeId);
        if (!$store) {
            return false;
        }

        $categories = [];
        $adapter = $this->getConnection();

        $this->select = $adapter->select()->from(
            $this->getMainTable()
        )->where(
            $this->getIdFieldName() . '=?',
            $store->getRootCategoryId()
        );
        $categoryRow = $adapter->fetchRow($this->select);

        if (!$categoryRow) {
            return false;
        }

        $this->select = $adapter->select()->from(
            ['e' => $this->getMainTable()],
            [$this->getIdFieldName(), 'parent_id', 'path', 'position', 'level', 'children_count']
        )->joinLeft(
            ['url_rewrite' => $this->getTable('url_rewrite')],
            'e.entity_id = url_rewrite.entity_id AND url_rewrite.is_autogenerated = 1'
            . $adapter->quoteInto(' AND url_rewrite.store_id = ?', $store->getId())
            . $adapter->quoteInto(' AND url_rewrite.entity_type = ?', CategoryUrlRewriteGenerator::ENTITY_TYPE),
            ['url' => 'request_path']
        )->where(
            'e.path LIKE ?',
            $categoryRow['path'] . '/%'
        );

        $this->joinAttribute($store->getId(), 'name', true);
        if ($this->sitemapHelper->isShowProducts() && $this->sitemapHelper->isUseCategoryDisplayMode()) {
            $this->joinAttribute($store->getId(), 'display_mode', true);
        }

        $this->addFilter($store->getId(), 'is_active', 1);
        $this->addFilter($store->getId(), 'in_html_sitemap', 1);

        $maxLevel = $this->sitemapHelper->getCategoryMaxDepth();
        if ($maxLevel) {
            $levels = [];
            for ($i=0; $i<=$maxLevel; $i++) {
                $levels[] = $i;
            }
            if (count($levels)) {
                $this->addFilter($store->getId(), 'level', $levels, 'in');
            }
        }

        $order = ['level'];
        if ($this->sitemapHelper->getCatProdSortOrder() == Source\SortOrder::SORT_BY_NAME) {
            $order[] = 'name';
        } elseif ($this->sitemapHelper->getCatProdSortOrder() == Source\SortOrder::SORT_BY_POSITION) {
            $order[] = 'position';
        }
        $this->select->order($order, \Magento\Framework\DB\Select::SQL_ASC);

        $query = $adapter->query($this->select);
        while ($row = $query->fetch()) {
            $category = $this->prepareCategory($row, $storeId);
            $categories[$category->getId()] = $category;
        }

        return $categories;
    }

    /**
     * Prepare category
     *
     * @param array $categoryRow
     * @param int $storeId
     * @return \Magento\Framework\DataObject
     */
    protected function prepareCategory(array $categoryRow, $storeId)
    {
        $category = new \Magento\Framework\DataObject();
        $category->addData($categoryRow);
        $category->setId($categoryRow[$this->getIdFieldName()]);

        if (!empty($categoryRow['url'])) {
            $category->setUrl($this->storeUrlHelper->getUrl($categoryRow['url'], $storeId));
        } else {
            $category->setUrl($this->storeUrlHelper->getUrl('catalog/category/view/id/' . $category->getId(), $storeId));
        }

        return $category;
    }

    /**
     * Add attribute to filter
     *
     * @param int $storeId
     * @param string $attributeCode
     * @param mixed $value
     * @param string $type
     * @return \Magento\Framework\DB\Select|bool
     */
    protected function addFilter($storeId, $attributeCode, $value, $type = '=')
    {
        $linkField = $this->linkFieldResolver->getLinkField(CategoryInterface::class, 'entity_id');

        if (!$this->select instanceof \Magento\Framework\DB\Select) {
            return false;
        }

        if (!isset($this->_attributesCache[$attributeCode])) {
            $attribute = $this->categoryResource->getAttribute($attributeCode);

            $this->_attributesCache[$attributeCode] = [
                'entity_type_id' => $attribute->getEntityTypeId(),
                'attribute_id'   => $attribute->getId(),
                'table'          => $attribute->getBackend()->getTable(),
                'is_global'      => $attribute->getIsGlobal(),
                'backend_type'   => $attribute->getBackendType(),
            ];
        }
        $attribute = $this->_attributesCache[$attributeCode];

        switch ($type) {
            case '=':
                $conditionRule = '=?';
                break;
            case 'in':
                $conditionRule = ' IN(?)';
                break;
            default:
                return false;
        }

        if ($attribute['backend_type'] == 'static') {
            $this->select->where('e.' . $attributeCode . $conditionRule, $value);
        } else {
            $this->select->join(
                ['t1_' . $attributeCode => $attribute['table']],
                'e.' . $linkField . ' = t1_' . $attributeCode . '.' . $linkField .
                ' AND t1_' . $attributeCode . '.store_id = 0',
                []
            )->where(
                't1_' . $attributeCode . '.attribute_id=?',
                $attribute['attribute_id']
            );

            if ($attribute['is_global']) {
                $this->select->where('t1_' . $attributeCode . '.value' . $conditionRule, $value);
            } else {
                $ifCase = $this->select->getAdapter()->getCheckSql(
                    't2_' . $attributeCode . '.value_id > 0',
                    't2_' . $attributeCode . '.value',
                    't1_' . $attributeCode . '.value'
                );
                $this->select->joinLeft(
                    ['t2_' . $attributeCode => $attribute['table']],
                    $this->getConnection()->quoteInto(
                        't1_' .
                        $attributeCode .
                        '.' . $linkField . ' = t2_' .
                        $attributeCode .
                        '.' . $linkField . ' AND t1_' .
                        $attributeCode .
                        '.attribute_id = t2_' .
                        $attributeCode .
                        '.attribute_id AND t2_' .
                        $attributeCode .
                        '.store_id=?',
                        $storeId
                    ),
                    []
                )->where(
                    '(' . $ifCase . ')' . $conditionRule,
                    $value
                );
            }
        }

        return $this->select;
    }

    /**
     * Join attribute by code
     *
     * @param int $storeId
     * @param string $attributeCode
     * @param string $addToResult
     * @return void
     */
    protected function joinAttribute($storeId, $attributeCode, $addToResult = false)
    {
        $linkField = $this->linkFieldResolver->getLinkField(CategoryInterface::class, 'entity_id');

        $adapter = $this->getConnection();
        $attribute = $this->_getAttribute($attributeCode);
        $this->select->joinLeft(
            ['t1_' . $attributeCode => $attribute['table']],
            'e.' . $linkField . ' = t1_' . $attributeCode . '.' . $linkField . ' AND ' . $adapter->quoteInto(
                ' t1_' . $attributeCode . '.store_id = ?',
                \Magento\Store\Model\Store::DEFAULT_STORE_ID
            ) . $adapter->quoteInto(
                ' AND t1_' . $attributeCode . '.attribute_id = ?',
                $attribute['attribute_id']
            ),
            ($addToResult && $attribute['is_global']) ? [$attributeCode => 't1_' . $attributeCode . '.value'] : []
        );

        if (!$attribute['is_global']) {
            $this->select->joinLeft(
                ['t2_' . $attributeCode => $attribute['table']],
                $this->getConnection()->quoteInto(
                    't1_' .
                    $attributeCode .
                    '.' . $linkField . ' = t2_' .
                    $attributeCode .
                    '.' . $linkField . ' AND t1_' .
                    $attributeCode .
                    '.attribute_id = t2_' .
                    $attributeCode .
                    '.attribute_id AND t2_' .
                    $attributeCode .
                    '.store_id = ?',
                    $storeId
                ),
                $addToResult ? [
                    $attributeCode => $this->select->getAdapter()->getCheckSql(
                        't2_' . $attributeCode . '.value_id > 0',
                        't2_' . $attributeCode . '.value',
                        't1_' . $attributeCode . '.value'
                    )] : []
            );
        }
    }

    /**
     * Get attribute data by attribute code
     *
     * @param string $attributeCode
     * @return array
     */
    protected function _getAttribute($attributeCode)
    {
        if (!isset($this->_attributesCache[$attributeCode])) {
            $attribute = $this->categoryResource->getAttribute($attributeCode);

            $this->_attributesCache[$attributeCode] = [
                'entity_type_id' => $attribute->getEntityTypeId(),
                'attribute_id' => $attribute->getId(),
                'table' => $attribute->getBackend()->getTable(),
                'is_global' => $attribute->getIsGlobal() ==
                \Magento\Catalog\Model\ResourceModel\Eav\Attribute::SCOPE_GLOBAL,
                'backend_type' => $attribute->getBackendType(),
            ];
        }
        return $this->_attributesCache[$attributeCode];
    }
}
