<?php
/**
 * @copyright Copyright (c) 2016 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\Totals\Plugin;

use Fooman\Totals\Model\QuoteAddressTotalFactory;
use Fooman\Totals\Model\QuoteAddressTotalManagement;
use Magento\Framework\DB\TransactionFactory;
use Magento\Quote\Model\Quote;

class QuoteSave
{
    /**
     * @var QuoteAddressTotalFactory
     */
    private $quoteAddressTotalFactory;

    /**
     * @var TransactionFactory
     */
    private $transactionFactory;

    /**
     * @var QuoteAddressTotalManagement
     */
    private $quoteAddressTotalManagement;

    public function __construct(
        QuoteAddressTotalFactory $quoteAddressTotalFactory,
        TransactionFactory $transactionFactory,
        QuoteAddressTotalManagement $quoteAddressTotalManagement
    ) {
        $this->quoteAddressTotalFactory = $quoteAddressTotalFactory;
        $this->transactionFactory = $transactionFactory;
        $this->quoteAddressTotalManagement = $quoteAddressTotalManagement;
    }

    /**
     * @param Quote $subject
     * @param \Closure                   $proceed
     *
     * @return Quote
     * @throws \Exception
     */
    public function aroundSave(
        Quote $subject,
        \Closure $proceed
    ) {
        $proceed();
        $this->saveQuoteAddressTotals($subject);
        return $subject;
    }

    /**
     * @param Quote $cart
     *
     * @throws \Exception
     */
    public function saveQuoteAddressTotals(Quote $cart)
    {
        $saveTransaction = false;
        $transaction = $this->transactionFactory->create();
        foreach ($cart->getAllAddresses() as $address) {
            $this->quoteAddressTotalManagement->deleteByCodeAndQuoteAddressId(
                $address->getId(),
                $cart->getId()
            );
            $extensionAttributes = $address->getExtensionAttributes();
            if (!$extensionAttributes) {
                continue;
            }
            $quoteAddressTotalGroup = $extensionAttributes->getFoomanTotalGroup();
            if (!$quoteAddressTotalGroup) {
                continue;
            }

            $foomanQuoteTotals = $quoteAddressTotalGroup->getItems();
            if (!empty($foomanQuoteTotals)) {
                $saveTransaction = true;
                foreach ($foomanQuoteTotals as $totalItem) {
                    /** @var \Fooman\Totals\Api\Data\QuoteAddressTotalInterface $totalItem */
                    $quoteAddressTotals = $this->quoteAddressTotalManagement
                        ->getByTypeIdAndAddressId(
                            $totalItem->getTypeId(),
                            $address->getId()
                        );

                    if (!empty($quoteAddressTotals)) {
                        $quoteAddressTotal = array_shift($quoteAddressTotals);
                    } else {
                        $quoteAddressTotal = $this->quoteAddressTotalFactory->create();
                    }

                    $quoteAddressTotal->setAmount($totalItem->getAmount());
                    $quoteAddressTotal->setBaseAmount($totalItem->getBaseAmount());
                    $quoteAddressTotal->setBasePrice($totalItem->getBasePrice());
                    $quoteAddressTotal->setTaxAmount($totalItem->getTaxAmount());
                    $quoteAddressTotal->setBaseTaxAmount($totalItem->getBaseTaxAmount());
                    $quoteAddressTotal->setLabel($totalItem->getLabel());
                    $quoteAddressTotal->setTypeId($totalItem->getTypeId());
                    $quoteAddressTotal->setCode($totalItem->getCode());
                    $quoteAddressTotal->setQuoteId($cart->getId());
                    $quoteAddressTotal->setQuoteAddressId($address->getId());
                    $transaction->addObject($quoteAddressTotal);
                }
            }
        }
        if ($saveTransaction) {
            $transaction->save();
        }
    }
}
