<?php
/**
 * @copyright Copyright (c) 2016 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\Totals\Plugin;

use Fooman\Totals\Model\GroupFactory;
use Fooman\Totals\Model\OrderTotalFactory;
use Fooman\Totals\Model\OrderTotalManagement;
use Magento\Framework\DB\TransactionFactory;
use Magento\Sales\Api\Data\OrderExtensionFactory;
use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\Data\OrderSearchResultInterface;
use Magento\Sales\Api\OrderRepositoryInterface;

class OrderRepository
{
    /**
     * @var OrderTotalFactory
     */
    private $orderTotalFactory;

    /**
     * @var OrderTotalManagement
     */
    private $orderTotalManagement;

    /**
     * @var OrderExtensionFactory
     */
    private $orderExtensionFactory;

    /**
     * @var GroupFactory
     */
    private $orderTotalGroupFactory;

    /**
     * @var TransactionFactory
     */
    private $transactionFactory;

    /**
     * @param OrderTotalFactory        $orderTotalFactory
     * @param OrderTotalManagement     $orderTotalManagement
     * @param OrderExtensionFactory $orderExtensionFactory
     * @param GroupFactory             $orderTotalGroupFactory
     * @param TransactionFactory      $transactionFactory
     */
    public function __construct(
        OrderTotalFactory $orderTotalFactory,
        OrderTotalManagement $orderTotalManagement,
        OrderExtensionFactory $orderExtensionFactory,
        GroupFactory $orderTotalGroupFactory,
        TransactionFactory $transactionFactory
    ) {
        $this->orderTotalFactory = $orderTotalFactory;
        $this->orderTotalManagement = $orderTotalManagement;
        $this->orderExtensionFactory = $orderExtensionFactory;
        $this->orderTotalGroupFactory = $orderTotalGroupFactory;
        $this->transactionFactory = $transactionFactory;
    }

    /**
     * @param  OrderRepositoryInterface $subject
     * @param  OrderInterface      $order
     *
     * @return OrderInterface      $order
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     * @throws \Exception
     */
    public function afterSave(
        OrderRepositoryInterface $subject,
        OrderInterface $order
    ) {
        $this->saveOrderTotals($order);
        return $order;
    }

    /**
     * @param  OrderInterface $order
     *
     * @throws \Exception
     */
    private function saveOrderTotals(OrderInterface $order)
    {
        $extensionAttributes = $order->getExtensionAttributes();
        if (!$extensionAttributes) {
            return;
        }
        $foomanOrderTotalGroup = $extensionAttributes->getFoomanTotalGroup();
        if (!$foomanOrderTotalGroup) {
            return;
        }

        $foomanOrderTotals = $foomanOrderTotalGroup->getItems();
        if (!empty($foomanOrderTotals)) {
            $transaction = $this->transactionFactory->create();
            foreach ($foomanOrderTotals as $foomanOrderTotalItem) {
                $orderTotals = $this->orderTotalManagement->getByTypeIdAndOrderId(
                    $foomanOrderTotalItem->getTypeId(),
                    $order->getEntityId()
                );

                if (!empty($orderTotals)) {
                    $orderTotal = array_shift($orderTotals);
                } else {
                    $orderTotal = $this->orderTotalFactory->create();
                }

                $orderTotal->setAmount($foomanOrderTotalItem->getAmount());
                $orderTotal->setBaseAmount($foomanOrderTotalItem->getBaseAmount());
                $orderTotal->setTaxAmount($foomanOrderTotalItem->getTaxAmount());
                $orderTotal->setBaseTaxAmount($foomanOrderTotalItem->getBaseTaxAmount());
                $orderTotal->setAmountInvoiced($foomanOrderTotalItem->getAmountInvoiced());
                $orderTotal->setBaseAmountInvoiced($foomanOrderTotalItem->getBaseAmountInvoiced());
                $orderTotal->setAmountRefunded($foomanOrderTotalItem->getAmountRefunded());
                $orderTotal->setBaseAmountRefunded($foomanOrderTotalItem->getBaseAmountRefunded());
                $orderTotal->setLabel($foomanOrderTotalItem->getLabel());
                $orderTotal->setTypeId($foomanOrderTotalItem->getTypeId());
                $orderTotal->setCode($foomanOrderTotalItem->getCode());
                $orderTotal->setOrderId($order->getEntityId());
                $transaction->addObject($orderTotal);
            }
            $transaction->save();
        }
    }

    /**
     * @param OrderRepositoryInterface $subject
     * @param OrderInterface      $order
     *
     * @return OrderInterface
     */
    public function afterGet(
        OrderRepositoryInterface $subject,
        OrderInterface $order
    ) {
        $this->applyExtensionAttributes($order);
        return $order;
    }

    /**
     * @param OrderRepositoryInterface        $subject
     * @param OrderSearchResultInterface $result
     *
     * @return OrderSearchResultInterface
     */
    public function afterGetList(
        OrderRepositoryInterface $subject,
        OrderSearchResultInterface $result
    ) {

        $orders = $result->getItems();
        if (!empty($orders)) {
            foreach ($orders as $order) {
                $this->applyExtensionAttributes($order);
            }
        }

        return $result;
    }

    /**
     * @param OrderInterface $order
     *
     * @return void
     */
    private function applyExtensionAttributes(OrderInterface $order)
    {
        $extensionAttributes = $order->getExtensionAttributes();
        if (!$extensionAttributes) {
            $extensionAttributes = $this->orderExtensionFactory->create();
        }

        $foomanTotalGroup = $extensionAttributes->getFoomanTotalGroup();
        if (!$foomanTotalGroup) {
            $foomanTotalGroup = $this->orderTotalGroupFactory->create();
        }

        //totals have already been added via collection load
        if (count($foomanTotalGroup->getItems())) {
            return;
        }

        $orderTotals = $this->orderTotalManagement->getByOrderId(
            $order->getEntityId()
        );

        if (!empty($orderTotals)) {
            foreach ($orderTotals as $orderTotal) {
                $foomanTotalGroup->addItem($orderTotal);
            }
        }
        $extensionAttributes->setFoomanTotalGroup($foomanTotalGroup);

        $order->setExtensionAttributes($extensionAttributes);
    }
}
