<?php
/**
 * @copyright Copyright (c) 2016 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\Totals\Plugin;

use Fooman\Totals\Model\GroupFactory;
use Fooman\Totals\Model\InvoiceTotalManagement;
use Magento\Sales\Api\Data\InvoiceExtensionFactory;
use Magento\Sales\Api\Data\InvoiceInterface;
use Magento\Sales\Api\Data\InvoiceSearchResultInterface;
use Magento\Sales\Api\InvoiceRepositoryInterface;

class InvoiceRepository
{
    /**
     * @var InvoiceTotalManagement
     */
    private $invoiceTotalManagement;

    /**
     * @var InvoiceExtensionFactory
     */
    private $invoiceExtensionFactory;

    /**
     * @var GroupFactory
     */
    private $invoiceTotalGroupFactory;

    /**
     * @param InvoiceTotalManagement     $invoiceTotalManagement
     * @param InvoiceExtensionFactory $invoiceExtensionFactory
     * @param GroupFactory               $invoiceTotalGroupFactory
     */
    public function __construct(
        InvoiceTotalManagement $invoiceTotalManagement,
        InvoiceExtensionFactory $invoiceExtensionFactory,
        GroupFactory $invoiceTotalGroupFactory
    ) {
        $this->invoiceTotalManagement = $invoiceTotalManagement;
        $this->invoiceExtensionFactory = $invoiceExtensionFactory;
        $this->invoiceTotalGroupFactory = $invoiceTotalGroupFactory;
    }

    /**
     * @param InvoiceRepositoryInterface $subject
     * @param InvoiceInterface      $invoice
     *
     * @return InvoiceInterface      $invoice
     */
    public function afterGet(
        InvoiceRepositoryInterface $subject,
        InvoiceInterface $invoice
    ) {
        $this->applyExtensionAttributes($invoice);
        return $invoice;
    }

    /**
     * @param InvoiceRepositoryInterface $subject
     * @param InvoiceSearchResultInterface $result
     *
     * @return InvoiceSearchResultInterface
     */
    public function afterGetList(
        InvoiceRepositoryInterface $subject,
        InvoiceSearchResultInterface $result
    ) {

        $invoices = $result->getItems();
        if (!empty($invoices)) {
            foreach ($invoices as $invoice) {
                $this->applyExtensionAttributes($invoice);
            }
        }

        return $result;
    }

    /**
     * @param InvoiceInterface $invoice
     *
     * @return void
     */
    private function applyExtensionAttributes(InvoiceInterface $invoice)
    {
        $extensionAttributes = $invoice->getExtensionAttributes();
        if (!$extensionAttributes) {
            $extensionAttributes = $this->invoiceExtensionFactory->create();
        }

        $foomanTotalGroup = $extensionAttributes->getFoomanTotalGroup();
        if (!$foomanTotalGroup) {
            $foomanTotalGroup = $this->invoiceTotalGroupFactory->create();
        }

        //totals have already been added via collection load
        if (count($foomanTotalGroup->getItems())) {
            return;
        }

        $invoiceTotals = $this->invoiceTotalManagement->getByInvoiceId(
            $invoice->getEntityId()
        );

        if (!empty($invoiceTotals)) {
            foreach ($invoiceTotals as $invoiceTotal) {
                $foomanTotalGroup->addItem($invoiceTotal);
            }
        }

        $extensionAttributes->setFoomanTotalGroup($foomanTotalGroup);
        $invoice->setExtensionAttributes($extensionAttributes);
    }
}
