<?php
/**
 * @copyright Copyright (c) 2016 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\Totals\Plugin;

use Fooman\Totals\Api\Data\QuoteAddressTotalInterface;
use Fooman\Totals\Api\Data\TotalGroupInterface;
use Fooman\Totals\Model\QuoteAddressTotalFactory;
use Fooman\Totals\Model\QuoteAddressTotalManagement;
use Magento\Framework\DB\TransactionFactory;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Quote\Model\Quote\Address;

class CartSave
{
    /**
     * @var QuoteAddressTotalFactory
     */
    private $quoteAddressTotalFactory;

    /**
     * @var TransactionFactory
     */
    private $transactionFactory;

    /**
     * @var QuoteAddressTotalManagement
     */
    private $quoteAddressTotalManagement;

    /**
     * @param QuoteAddressTotalFactory                          $quoteAddressTotalFactory
     * @param TransactionFactory                               $transactionFactory
     * @param QuoteAddressTotalManagement                       $quoteAddressTotalManagement
     */
    public function __construct(
        QuoteAddressTotalFactory $quoteAddressTotalFactory,
        TransactionFactory $transactionFactory,
        QuoteAddressTotalManagement $quoteAddressTotalManagement
    ) {
        $this->quoteAddressTotalFactory = $quoteAddressTotalFactory;
        $this->transactionFactory = $transactionFactory;
        $this->quoteAddressTotalManagement = $quoteAddressTotalManagement;
    }

    /**
     * @param  CartRepositoryInterface $subject
     * @param \Closure                                    $proceed
     * @param  CartInterface      $cart
     *
     * @return CartInterface       $cart
     *
     * @throws \Exception
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function aroundSave(
        CartRepositoryInterface $subject,
        \Closure $proceed,
        CartInterface $cart
    ) {
        $proceed($cart);
        $this->saveQuoteAddressTotals($cart);
        return $cart;
    }

    /**
     * @param CartInterface $cart
     *
     * @throws \Exception
     */
    public function saveQuoteAddressTotals(CartInterface $cart)
    {
        $saveTransaction = false;
        $transaction = $this->transactionFactory->create();
        /** @var Address $address */
        foreach ($cart->getAllAddresses() as $address) {
            $extensionAttributes = $address->getExtensionAttributes();
            if (!$extensionAttributes) {
                continue;
            }

            /** @var TotalGroupInterface $quoteAddressTotalGroup */
            $quoteAddressTotalGroup = $extensionAttributes->getFoomanTotalGroup();
            if (!$quoteAddressTotalGroup) {
                continue;
            }

            foreach ($quoteAddressTotalGroup->getItems() as $totalItem) {
                /** @var QuoteAddressTotalInterface $totalItem */
                $quoteAddressTotals = $this->quoteAddressTotalManagement
                    ->getByTypeIdAndAddressId(
                        $totalItem->getTypeId(),
                        $address->getId()
                    );

                if (!empty($quoteAddressTotals)) {
                    $quoteAddressTotal = array_shift($quoteAddressTotals);
                } else {
                    $quoteAddressTotal = $this->quoteAddressTotalFactory->create();
                }
                $quoteAddressTotal->setAmount($totalItem->getAmount());
                $quoteAddressTotal->setBaseAmount($totalItem->getBaseAmount());
                $quoteAddressTotal->setTaxAmount($totalItem->getTaxAmount());
                $quoteAddressTotal->setBaseTaxAmount($totalItem->getBaseTaxAmount());
                $quoteAddressTotal->setBasePrice($totalItem->getBasePrice());
                $quoteAddressTotal->setLabel($totalItem->getLabel());
                $quoteAddressTotal->setTypeId($totalItem->getTypeId());
                $quoteAddressTotal->setCode($totalItem->getCode());
                $quoteAddressTotal->setQuoteId($cart->getId());
                $quoteAddressTotal->setQuoteAddressId($address->getId());
                $transaction->addObject($quoteAddressTotal);
                $saveTransaction = true;
            }
        }
        if ($saveTransaction) {
            $transaction->save();
        }
    }
}
