<?php
/**
 * @copyright Copyright (c) 2016 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\Totals\Observer;

use Fooman\Totals\Model\CreditmemoTotalFactory;
use Fooman\Totals\Model\CreditmemoTotalManagement;
use Fooman\Totals\Model\OrderTotalManagement;
use Magento\Framework\DB\TransactionFactory;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Event\Observer as EventObserver;
use Magento\Sales\Api\Data\CreditmemoInterface;

class CreditmemoTotal implements ObserverInterface
{
    /**
     * @var CreditmemoTotalFactory
     */
    private $creditmemoTotalFactory;

    /**
     * @var CreditmemoTotalManagement
     */
    private $creditmemoTotalManagement;

    /**
     * @var OrderTotalManagement
     */
    private $orderTotalManagement;

    /**
     * @var TransactionFactory
     */
    private $transactionFactory;

    /**
     * @param CreditmemoTotalFactory    $creditmemoTotalFactory
     * @param CreditmemoTotalManagement $creditmemoTotalManagement
     * @param OrderTotalManagement      $orderTotalManagement
     * @param TransactionFactory       $transactionFactory
     */
    public function __construct(
        CreditmemoTotalFactory $creditmemoTotalFactory,
        CreditmemoTotalManagement $creditmemoTotalManagement,
        OrderTotalManagement $orderTotalManagement,
        TransactionFactory $transactionFactory
    ) {
        $this->creditmemoTotalFactory = $creditmemoTotalFactory;
        $this->creditmemoTotalManagement = $creditmemoTotalManagement;
        $this->orderTotalManagement = $orderTotalManagement;
        $this->transactionFactory = $transactionFactory;
    }

    /**
     * @param EventObserver $observer
     *
     * @return void
     * @throws \Exception
     */
    public function execute(EventObserver $observer)
    {
        $this->saveCreditmemoTotals($observer->getCreditmemo());
    }

    /**
     * @param  CreditmemoInterface $creditmemo
     *
     * @throws \Exception
     */
    private function saveCreditmemoTotals(CreditmemoInterface $creditmemo)
    {
        $extensionAttributes = $creditmemo->getExtensionAttributes();
        if (!$extensionAttributes) {
            return;
        }
        $foomanCreditmemoTotalGroup = $extensionAttributes->getFoomanTotalGroup();
        if (!$foomanCreditmemoTotalGroup) {
            return;
        }

        $foomanCreditmemoTotals = $foomanCreditmemoTotalGroup->getItems();
        if (!empty($foomanCreditmemoTotals)) {
            $transaction = $this->transactionFactory->create();
            foreach ($foomanCreditmemoTotals as $foomanCreditmemoTotalItem) {
                $creditmemoTotals = false;
                if ($creditmemo->getEntityId()) {
                    $creditmemoTotals = $this->creditmemoTotalManagement->getByTypeAndCreditmemoId(
                        $foomanCreditmemoTotalItem->getTypeId(),
                        $creditmemo->getEntityId()
                    );
                }

                if (!empty($creditmemoTotals)) {
                    $creditmemoTotal = array_shift($creditmemoTotals);
                } else {
                    $creditmemoTotal = $this->creditmemoTotalFactory->create();
                }

                $creditmemoTotal->setAmount($foomanCreditmemoTotalItem->getAmount());
                $creditmemoTotal->setBaseAmount($foomanCreditmemoTotalItem->getBaseAmount());
                $creditmemoTotal->setTaxAmount($foomanCreditmemoTotalItem->getTaxAmount());
                $creditmemoTotal->setBaseTaxAmount($foomanCreditmemoTotalItem->getBaseTaxAmount());
                $creditmemoTotal->setLabel($foomanCreditmemoTotalItem->getLabel());
                $creditmemoTotal->setTypeId($foomanCreditmemoTotalItem->getTypeId());
                $creditmemoTotal->setCode($foomanCreditmemoTotalItem->getCode());
                $creditmemoTotal->setOrderId($creditmemo->getOrderId());
                $creditmemoTotal->setCreditmemoId($creditmemo->getEntityId());
                $transaction->addObject($creditmemoTotal);

                //Update corresponding order total
                $orderTotal = $this->getCorrespondingOrderTotal($foomanCreditmemoTotalItem, $creditmemo);
                $orderTotal->setAmountRefunded(
                    $orderTotal->getAmountRefunded() + $foomanCreditmemoTotalItem->getAmount()
                );
                $orderTotal->setBaseAmountRefunded(
                    $orderTotal->getBaseAmountRefunded() + $foomanCreditmemoTotalItem->getBaseAmount()
                );
                $transaction->addObject($orderTotal);
            }
            $transaction->save();
        }
    }

    public function getCorrespondingOrderTotal($creditmemoTotal, $creditmemo)
    {
        $orderTotals = $this->orderTotalManagement->getByTypeIdAndOrderId(
            $creditmemoTotal->getTypeId(),
            $creditmemo->getOrderId()
        );
        return array_shift($orderTotals);
    }
}
