<?php
/**
 * @copyright Copyright (c) 2019 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\SurchargeShipping\Model;

use Fooman\Surcharge\Model\SurchargeCalculation;
use Magento\Quote\Api\Data\ShippingAssignmentInterface;

class WeightType implements \Fooman\Surcharge\Api\Data\TypeInterface
{
    /**
     * @var string
     */
    private $type = 'weight';

    /**
     * @var \Fooman\Surcharge\Helper\SurchargeConfig
     */
    private $surchargeConfigHelper;

    /**
     * @var \Fooman\Surcharge\Model\SurchargeCalculationFactory
     */
    private $surchargeCalculationFactory;

    /**
     * @var \Fooman\Surcharge\Helper\Currency
     */
    private $currencyHelper;

    /**
     * @param \Fooman\Surcharge\Model\SurchargeCalculationFactory $surchargeCalculationFactory
     * @param \Fooman\Surcharge\Helper\SurchargeConfig            $surchargeConfigHelper
     * @param \Fooman\Surcharge\Helper\Currency                   $currencyHelper
     */
    public function __construct(
        \Fooman\Surcharge\Model\SurchargeCalculationFactory $surchargeCalculationFactory,
        \Fooman\Surcharge\Helper\SurchargeConfig $surchargeConfigHelper,
        \Fooman\Surcharge\Helper\Currency $currencyHelper
    ) {
        $this->surchargeCalculationFactory = $surchargeCalculationFactory;
        $this->surchargeConfigHelper = $surchargeConfigHelper;
        $this->currencyHelper = $currencyHelper;
    }

    /**
     * @param \Fooman\Surcharge\Api\SurchargeInterface $surcharge
     * @param \Magento\Quote\Api\Data\CartInterface    $quote
     * @param ShippingAssignmentInterface              $shippingAssignment
     *
     * @return \Fooman\Totals\Model\QuoteAddressTotal[]
     */
    public function calculate(
        \Fooman\Surcharge\Api\SurchargeInterface $surcharge,
        \Magento\Quote\Api\Data\CartInterface $quote,
        ShippingAssignmentInterface $shippingAssignment
    ) {
        $config = $this->surchargeConfigHelper->getConfig($surcharge);

        $weight = $config->getWeight();
        if (!$weight) {
            return [];
        }

        /** @var SurchargeCalculation $surchargeCalc */
        $surchargeCalc = $this->surchargeCalculationFactory
            ->create(['quote' => $quote, 'surcharge' => $surcharge, 'assignment' => $shippingAssignment]);

        if (!$surchargeCalc->shouldSurchargeApply($quote, $config, $surchargeCalc->getCurrentSubTotal())) {
            return [];
        }

        if ($this->surchargeApplies($shippingAssignment, $config)) {
            $amount = $this->calculateWeightSurcharge($shippingAssignment, $config, $surchargeCalc, $surcharge);
            return [$surchargeCalc->processTotals(false, $amount, true)];
        }

        return [];
    }

    public function calculateWeightSurcharge($shippingAssignment, $config, $surchargeCalculation, $surcharge)
    {
        $weight = 0;
        foreach ($shippingAssignment->getItems() as $item) {
            if ($surchargeCalculation->shouldItemCountTowardsTotal($item, $surcharge)) {
                $weight += $item->getRowWeight();
            }
        }
        if ($weight == 0) {
            return 0;
        }
        $multiplier = ceil($weight / $config->getWeight());
        $surchargeAmount = $multiplier * $config->getFixed();
        if ($config->getMaxFixed() && $surchargeAmount > $config->getMaxFixed()) {
            $surchargeAmount = $config->getMaxFixed();
        }
        return $surchargeAmount;
    }

    /**
     * @return string
     */
    public function getLabel()
    {
        return __('Weight Surcharge');
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @param ShippingAssignmentInterface $shippingAssignment
     * @param                             $config
     *
     * @return bool
     */
    public function surchargeApplies(ShippingAssignmentInterface $shippingAssignment, $config)
    {
        if (!$config->getApplyMethodFilter()) {
            return true;
        }

        $shippingMethods = $config->getShipping();
        if (is_string($shippingMethods)) {
            $shippingMethods = [$shippingMethods];
        }

        $currentShippingMethod = $shippingAssignment->getShipping()->getMethod();
        return in_array($currentShippingMethod, $shippingMethods);
    }
}
