<?php
/**
 * @copyright Copyright (c) 2019 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\SurchargeShipping\Model;

use Fooman\Surcharge\Model\SurchargeCalculation;
use Magento\Quote\Api\Data\ShippingAssignmentInterface;

/**
 * @author     Kristof Ringleff
 * @package    Fooman_SurchargeShipping
 * @copyright  Copyright (c) 2016 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
class ShippingType implements \Fooman\Surcharge\Api\Data\TypeInterface
{
    /**
     * @var string
     */
    private $type = 'shipping';

    /**
     * @var \Fooman\Surcharge\Helper\SurchargeConfig
     */
    private $surchargeConfigHelper;

    /**
     * @var \Fooman\Surcharge\Model\SurchargeCalculationFactory
     */
    private $surchargeCalculationFactory;

    /**
     * @param \Fooman\Surcharge\Model\SurchargeCalculationFactory $surchargeCalculationFactory
     * @param \Fooman\Surcharge\Helper\SurchargeConfig            $surchargeConfigHelper
     */
    public function __construct(
        \Fooman\Surcharge\Model\SurchargeCalculationFactory $surchargeCalculationFactory,
        \Fooman\Surcharge\Helper\SurchargeConfig $surchargeConfigHelper
    ) {
        $this->surchargeCalculationFactory = $surchargeCalculationFactory;
        $this->surchargeConfigHelper = $surchargeConfigHelper;
    }

    /**
     * @param \Fooman\Surcharge\Api\SurchargeInterface $surcharge
     * @param \Magento\Quote\Api\Data\CartInterface    $quote
     * @param ShippingAssignmentInterface              $shippingAssignment
     *
     * @return \Fooman\Totals\Model\QuoteAddressTotal[]
     */
    public function calculate(
        \Fooman\Surcharge\Api\SurchargeInterface $surcharge,
        \Magento\Quote\Api\Data\CartInterface $quote,
        ShippingAssignmentInterface $shippingAssignment
    ) {
        $config = $this->surchargeConfigHelper->getConfig($surcharge);

        $shippingMethods = $config->getShipping();
        if (!$shippingMethods) {
            return [];
        }

        if (is_string($shippingMethods)) {
            $shippingMethods = [$shippingMethods];
        }

        /** @var SurchargeCalculation $surchargeCalc */
        $surchargeCalc = $this->surchargeCalculationFactory
            ->create(['quote' => $quote, 'surcharge' => $surcharge, 'assignment' => $shippingAssignment]);

        if (!$surchargeCalc->shouldSurchargeApply($quote, $config, $surchargeCalc->getCurrentSubTotal())) {
            return [];
        }

        if ($this->surchargeApplies($shippingAssignment, $shippingMethods)) {
            return [$surchargeCalc->processTotals()];
        }

        return [];
    }

    /**
     * @return string
     */
    public function getLabel()
    {
        return __('Shipping Surcharge');
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @param ShippingAssignmentInterface $shippingAssignment
     * @param array                       $shippingMethods
     *
     * @return bool
     */
    public function surchargeApplies(ShippingAssignmentInterface $shippingAssignment, $shippingMethods)
    {
        $currentShippingMethod = $shippingAssignment->getShipping()->getMethod();
        return in_array($currentShippingMethod, $shippingMethods);
    }
}
