<?php
/**
 * @copyright Copyright (c) 2009 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\SurchargePayment\Plugin;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Checkout\Api\Data\PaymentDetailsInterface;
use Magento\Framework\Pricing\Helper\Data as PricingHelper;
use Fooman\Surcharge\Helper\Surcharge as SurchargeHelper;
use Fooman\SurchargePayment\Model\SurchargeAdjustedPaymentMethodFactory;
use Magento\Store\Model\ScopeInterface;

class SurchargePreview
{
    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var SurchargeAdjustedPaymentMethodFactory
     */
    private $surchargeAdjustedPaymentMethodFactory;

    /**
     * @var SurchargeHelper
     */
    private $surchargeHelper;

    /**
     * @var PricingHelper
     */
    private $pricingHelper;

    /**
     * @param ScopeConfigInterface $scopeConfig
     * @param SurchargeAdjustedPaymentMethodFactory $surchargeAdjustedPaymentMethodFactory
     * @param SurchargeHelper $surchargeHelper
     * @param PricingHelper $pricingHelper
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        SurchargeAdjustedPaymentMethodFactory $surchargeAdjustedPaymentMethodFactory,
        SurchargeHelper $surchargeHelper,
        PricingHelper $pricingHelper
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->surchargeAdjustedPaymentMethodFactory = $surchargeAdjustedPaymentMethodFactory;
        $this->surchargeHelper = $surchargeHelper;
        $this->pricingHelper = $pricingHelper;
    }

    /**
     * @param PaymentDetailsInterface $subject
     * @param \Magento\Quote\Api\Data\PaymentMethodInterface[] $result
     * @return \Magento\Quote\Api\Data\PaymentMethodInterface[]
     */
    public function afterGetPaymentMethods(
        PaymentDetailsInterface $subject,
        $result
    ) {
        if (!isset($result[0])) {
            return $result;
        }

        $originalQuote = $result[0]->getInfoInstance()->getQuote();

        if (!$originalQuote) {
            return $result;
        }

        $paymentMethodsForDisplay = $result;

        if ($this->scopeConfig->isSetFlag(
            'sales/surcharge/titleadjust',
            ScopeInterface::SCOPE_STORE,
            $originalQuote->getStoreId()
        )) {
            $paymentMethods = $result;
            $paymentMethodsForDisplay = [];

            foreach ($paymentMethods as $paymentMethod) {
                try {
                    $methodCode = $paymentMethod->getCode();
                    $quoteClone = clone $originalQuote;
                    $quoteClone->setId(null);
                    $quoteClone->setPaymentMethod($methodCode);
                    $quoteClone->getPayment()->importData(
                        [
                            'method' => $methodCode,
                            'additional_data'=> [
                                'fooman_payment_surcharge_preview' => 'true',
                                'buckaroo_skip_validation' => 'true'
                            ]
                        ]
                    );
                    $quoteClone->setTotalsCollectedFlag(false);
                    $quoteClone->collectTotals();
                } catch (\Exception $e) {
                    $paymentMethodsForDisplay[] = $paymentMethod;
                    continue;
                }

                if ($quoteClone->isVirtual()) {
                    $address = $quoteClone->getBillingAddress();
                } else {
                    $address = $quoteClone->getShippingAddress();
                }

                $appliedSurcharges = [];

                if ($foomanTotalGroup = $address->getExtensionAttributes()->getFoomanTotalGroup()) {
                    foreach ($foomanTotalGroup->getItems() as $item) {
                        $appliedSurcharges[] = [
                            'amount' => $item->getAmount(),
                            'surcharge' => $this->surchargeHelper->getSurchargeByTypeId($item->getTypeId())
                        ];
                    }
                } else {
                    $paymentMethodsForDisplay[] = $paymentMethod;
                    continue;
                }

                $descriptions = [];
                $totalAmount = 0;
                $adjustedTitle = $paymentMethod->getTitle();

                foreach ($appliedSurcharges as $appliedSurcharge) {
                    if ($appliedSurcharge['amount'] != 0) {
                        $descriptions[] = $appliedSurcharge['surcharge']->getDescription();
                    }
                    $totalAmount += $appliedSurcharge['amount'];
                }

                if ($totalAmount != 0) {
                    $formattedAmount = $this->pricingHelper->currency(
                        $totalAmount,
                        true,
                        false
                    );
                    $adjustedTitle = $this->replaceTitle(
                        $paymentMethod->getTitle(),
                        $formattedAmount,
                        implode(', ', $descriptions)
                    );
                }

                $paymentMethodsForDisplay[] = $this->surchargeAdjustedPaymentMethodFactory
                    ->create(['data' => ['code' => $methodCode, 'title' => $adjustedTitle]]);
            }
        }

        return $paymentMethodsForDisplay;
    }

    /**
     * @param string $title
     * @param string $formattedAmount
     * @param string $description
     * @return string
     */
    private function replaceTitle($title, $formattedAmount, $description)
    {
        $search = ['{TITLE}', '{AMOUNT}', '{DESCRIPTION}'];
        $replace = [$title, $formattedAmount, $description];
        $format = $this->scopeConfig->getValue('sales/surcharge/titleformat');
        $result = str_replace($search, $replace, $format);

        return $result;
    }
}
