<?php
/**
 * @copyright Copyright (c) 2009 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\SurchargeMinimumAmount\Model;

class EnforcedAmountType implements \Fooman\Surcharge\Api\Data\TypeInterface
{
    /**
     * @var string
     */
    private $type = 'enforced_amount';

    /**
     * @var \Fooman\Surcharge\Helper\SurchargeConfig
     */
    private $surchargeConfigHelper;

    /**
     * @var \Fooman\Surcharge\Model\SurchargeCalculationFactory
     */
    private $surchargeCalculationFactory;

    /**
     * @var \Fooman\Surcharge\Helper\Currency
     */
    private $currencyHelper;

    /**
     * @var \Fooman\Totals\Model\QuoteAddressTotalManagement
     */
    private $quoteAddressTotalManagement;

    /**
     * @param \Fooman\Surcharge\Model\SurchargeCalculationFactory $surchargeCalculationFactory
     * @param \Fooman\Surcharge\Helper\SurchargeConfig            $surchargeConfigHelper
     * @param \Fooman\Surcharge\Helper\Currency                   $currencyHelper
     * @param \Fooman\Totals\Model\QuoteAddressTotalManagement    $quoteAddressTotalManagement
     */
    public function __construct(
        \Fooman\Surcharge\Model\SurchargeCalculationFactory $surchargeCalculationFactory,
        \Fooman\Surcharge\Helper\SurchargeConfig $surchargeConfigHelper,
        \Fooman\Surcharge\Helper\Currency $currencyHelper,
        \Fooman\Totals\Model\QuoteAddressTotalManagement $quoteAddressTotalManagement
    ) {
        $this->surchargeCalculationFactory = $surchargeCalculationFactory;
        $this->surchargeConfigHelper = $surchargeConfigHelper;
        $this->currencyHelper = $currencyHelper;
        $this->quoteAddressTotalManagement = $quoteAddressTotalManagement;
    }

    /**
     * @param  \Fooman\Surcharge\Api\SurchargeInterface $surcharge
     * @param  \Magento\Quote\Api\Data\CartInterface    $quote
     *
     * @return \Fooman\Totals\Api\Data\QuoteAddressTotalInterface[]
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function calculate(
        \Fooman\Surcharge\Api\SurchargeInterface $surcharge,
        \Magento\Quote\Api\Data\CartInterface $quote,
        \Magento\Quote\Api\Data\ShippingAssignmentInterface $shippingAssignment
    ) {
        $config = $this->surchargeConfigHelper->getConfig($surcharge);
        $surchargeCalculation = $this->surchargeCalculationFactory
            ->create(['quote' => $quote, 'surcharge' => $surcharge, 'assignment' => $shippingAssignment]);

        $currentSubTotal = $surchargeCalculation->getCurrentSubTotal();

        if ($currentSubTotal < $this->getEnforcedAmount($config, $quote)
            && $surchargeCalculation->shouldSurchargeApply($quote, $config, $currentSubTotal)
        ) {
            $total = $surchargeCalculation->processTotals(true);
            $this->enforceAmount($quote, $total, $currentSubTotal, $config);
            $total->setLabel($surcharge->getDescription());
            return [$total];
        }

        $this->quoteAddressTotalManagement->deleteByTypeIdAndQuoteId($surcharge->getTypeId(), $quote->getId());
        return [];
    }

    public function getEnforcedAmount($config, $quote)
    {
        return $config->getEnforcedAmount();
    }

    /**
     * @return \Magento\Framework\Phrase
     */
    public function getLabel()
    {
        return __('Minimum Order Fee');
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    private function enforceAmount($quote, $total, $currentSubTotal, $config)
    {
        $total->setBaseAmount($this->currencyHelper->round($config->getEnforcedAmount() - $currentSubTotal));
        $total->setAmount($this->currencyHelper->convertToQuoteCurrency($quote, $total->getBaseAmount()));
    }
}
