<?php
/**
 * @copyright Copyright (c) 2016 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\Surcharge\Plugin;

use Fooman\Surcharge\Helper\Currency;
use Fooman\Totals\Model\CreditmemoTotalManagement;
use Magento\Tax\Block\Sales\Order\Tax;
use Magento\Tax\Model\Config;

class FixSubtotalInclDisplay
{
    /**
     * @var CreditmemoTotalManagement
     */
    private $creditmemoTotalManagement;

    /**
     * @var Config
     */
    private $taxConfig;

    /**
     * @var Currency
     */
    private $currencyHelper;

    public function __construct(
        Config $taxConfig,
        Currency $currencyHelper,
        CreditmemoTotalManagement $creditmemoTotalManagement
    ) {
        $this->taxConfig = $taxConfig;
        $this->currencyHelper = $currencyHelper;
        $this->creditmemoTotalManagement = $creditmemoTotalManagement;
    }

    /**
     * Magento's creditmemos overstate tax for tax inclusive subtotals when subtotal is zero
     * This plugin fixes the subtotal incl tax display line. The database values are correct and this issue
     * is a display issue only due to Magento's assumption tax amount = tax on products + tax on shipping
     *
     * @param Tax $subject
     * @param $result
     * @return mixed
     */
    public function afterInitTotals(
        Tax $subject,
        $result
    ) {
        $source = $subject->getSource();
        if ($subject->getFoomanAppliedTaxFix() != 1
            && $source instanceof \Magento\Sales\Model\Order\Creditmemo
            && $source->getSubtotal() == 0
        ) {
            $storeId = $source->getStoreId();
            $parent = $subject->getParentBlock();
            $overstatedTaxes = $this->getOverstatedTaxAmount($source);

            if (!$this->taxConfig->displaySalesSubtotalBoth($storeId) &&
                !$this->taxConfig->displaySalesSubtotalInclTax($storeId)
            ) {
                return $result;
            }
            if ($this->taxConfig->displaySalesSubtotalBoth($storeId)) {
                $subtotalInclTotal = $parent->getTotal('subtotal_incl');
            } elseif ($this->taxConfig->displaySalesSubtotalInclTax($storeId)) {
                $subtotalInclTotal = $parent->getTotal('subtotal');
            }
            $subtotalInclTotal->setValue(
                $this->currencyHelper->round($subtotalInclTotal->getValue() - $overstatedTaxes['tax_amount'])
            );
            $subtotalInclTotal->setBaseValue(
                $this->currencyHelper->round($subtotalInclTotal->getBaseValue() - $overstatedTaxes['base_tax_amount'])
            );

            $subject->setFoomanAppliedTaxFix(1);
        }
        return $result;
    }

    private function getOverstatedTaxAmount(\Magento\Sales\Model\Order\Creditmemo $creditmemo)
    {
        $totalSurchargeTaxAmount = 0;
        $baseTotalSurchargeTaxAmount = 0;
        $creditMemoTotals = $this->getTotalsFromExtAttr($creditmemo);
        if (empty($creditMemoTotals)) {
            //unfortunately the frontend does not load the creditmemo via repository in all cases
            //so our extension attributes are not set
            $creditMemoTotals = $this->getTotalsFromDb($creditmemo);
        }

        if (!empty($creditMemoTotals)) {
            foreach ($creditMemoTotals as $total) {
                $totalSurchargeTaxAmount += $total->getTaxAmount();
                $baseTotalSurchargeTaxAmount += $total->getBaseTaxAmount();
            }
        }

        return [
            'tax_amount' => $totalSurchargeTaxAmount,
            'base_tax_amount' => $baseTotalSurchargeTaxAmount
        ];
    }

    private function getTotalsFromExtAttr($salesObject)
    {
        $returnTotals = [];
        $extensionAttributes = $salesObject->getExtensionAttributes();
        if (!$extensionAttributes) {
            return $returnTotals;
        }
        $foomanTotalGroup = $extensionAttributes->getFoomanTotalGroup();
        if (!$foomanTotalGroup) {
            return $returnTotals;
        }
        $orderTotals = $foomanTotalGroup->getItems();
        if (empty($orderTotals)) {
            return $returnTotals;
        }

        foreach ($orderTotals as $orderTotal) {
            if ($orderTotal->getCode() == \Fooman\Surcharge\Model\Surcharge::CODE) {
                $returnTotals[] = $orderTotal;
            }
        }

        return $returnTotals;
    }

    private function getTotalsFromDb(\Magento\Sales\Model\Order\Creditmemo $creditmemo)
    {
        return $this->creditmemoTotalManagement->getByCodeAndCreditmemoId('fooman_surcharge', $creditmemo->getId());
    }
}
