<?php
/**
 * @copyright Copyright (c) 2016 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\Surcharge\Plugin;

use Fooman\Surcharge\Model\Surcharge;
use Fooman\Totals\Model\GroupFactory;
use Fooman\Totals\Model\OrderTotalManagement;
use Fooman\Totals\Model\CreditmemoTotalManagement;
use Magento\Tax\Helper\Data;

class FixCreditmemoTaxRateDisplay
{
    /**
     * @var OrderTotalManagement
     */
    private $orderTotalManagement;

    /**
     * @var CreditmemoTotalManagement
     */
    private $creditmemoTotalManagement;

    /**
     * @var GroupFactory
     */
    private $groupFactory;

    public function __construct(
        OrderTotalManagement $orderTotalManagement,
        CreditmemoTotalManagement $creditmemoTotalManagement,
        GroupFactory         $groupFactory
    ) {
        $this->orderTotalManagement = $orderTotalManagement;
        $this->creditmemoTotalManagement = $creditmemoTotalManagement;
        $this->groupFactory = $groupFactory;
    }

    /**
     * Magento's creditmemos overstate tax, their calculation automatically takes up all existing tax
     * irrespective of origin
     *
     * This plugin fixes the detailed tax summary display
     *
     * @param Data $taxHelper
     * @param $result
     * @param $source
     * @return mixed
     */
    public function afterGetCalculatedTaxes(
        Data $taxHelper,
        $result,
        $source
    ) {
        if (!$source instanceof \Magento\Sales\Model\Order\Creditmemo) {
            return $result;
        }
        if ($source->getFoomanRequiresTaxFix() || $this->requiresCreditmemoFix($source)) {
            $source->setFoomanRequiresTaxFix(0);
            $overstatedTaxes = $this->getOverstatedTaxAmount($source);
            //Need to revisit for multiple taxes
            if (count($result) === 1) {
                $taxRateKey = array_key_first($result);
                $result[$taxRateKey]['tax_amount'] -= $overstatedTaxes['tax_amount'];
                $result[$taxRateKey]['base_tax_amount'] -= $overstatedTaxes['base_tax_amount'];
            }
        }
        return $result;
    }

    private function requiresCreditmemoFix($creditmemo)
    {
        $creditMemoTotals = $this->getCreditMemoSurcharges($creditmemo);
        $totalAmount = 0;
        if (empty($creditMemoTotals)) {
            $orderTotalGroup = $this->getOrderSurcharges($creditmemo->getOrder());
            return count($orderTotalGroup->getItems()) > 0;
        }
        foreach ($creditMemoTotals as $total) {
            $totalAmount += $total->getAmount();
        }

        return $totalAmount == 0;
    }

    private function getOverstatedTaxAmount(\Magento\Sales\Model\Order\Creditmemo $creditmemo)
    {
        $totalSurchargeTaxAmount = 0;
        $baseTotalSurchargeTaxAmount = 0;
        $creditMemoTotals = $this->getTotalsFromExtAttr($creditmemo);
        $orderTotalGroup = $this->getOrderSurcharges($creditmemo->getOrder());

        if (!empty($creditMemoTotals)) {
            foreach ($creditMemoTotals as $total) {
                $orderTotal = $orderTotalGroup->getByTypeId($total->getTypeId());
                if ($orderTotal) {
                    $missingTax = $this->getMissingTaxAmount($orderTotal);
                    $totalSurchargeTaxAmount += $missingTax['tax_amount'] - $total->getTaxAmount();
                    $baseTotalSurchargeTaxAmount += $missingTax['base_tax_amount'] - $total->getBaseTaxAmount();
                }
            }
        } else {
            foreach ($orderTotalGroup->getItems() as $orderTotal) {
                $missingTax = $this->getMissingTaxAmount($orderTotal);
                $totalSurchargeTaxAmount += $missingTax['tax_amount'];
                $baseTotalSurchargeTaxAmount += $missingTax['base_tax_amount'];
            }
        }
        return [
            'tax_amount' => $totalSurchargeTaxAmount,
            'base_tax_amount' => $baseTotalSurchargeTaxAmount
        ];
    }

    private function getOrderSurcharges($order)
    {
        $group = $this->groupFactory->create();
        $totals = $this->orderTotalManagement->getByCodeAndOrderId(Surcharge::CODE, $order->getId());
        if (!empty($totals)) {
            foreach ($totals as $total) {
                $group->addItem($total);
            }
        }
        return $group;
    }

    private function getMissingTaxAmount($orderTotal)
    {
        return [
            'tax_amount' => $orderTotal->getTaxAmount(),
            'base_tax_amount' => $orderTotal->getBaseTaxAmount()
        ];
    }

    private function getCreditMemoSurcharges($creditmemo)
    {
        return $this->creditmemoTotalManagement->getByCodeAndCreditmemoId(Surcharge::CODE, $creditmemo->getId());
    }

    private function getTotalsFromExtAttr($creditmemo)
    {
        $returnTotals = [];
        $extensionAttributes = $creditmemo->getExtensionAttributes();
        if (!$extensionAttributes) {
            return $this->getCreditMemoSurcharges($creditmemo);
        }
        $foomanTotalGroup = $extensionAttributes->getFoomanTotalGroup();
        if (!$foomanTotalGroup) {
            return $this->getCreditMemoSurcharges($creditmemo);
        }
        $orderTotals = $foomanTotalGroup->getItems();
        if (empty($orderTotals)) {
            return $returnTotals;
        }

        foreach ($orderTotals as $orderTotal) {
            if ($orderTotal->getCode() == \Fooman\Surcharge\Model\Surcharge::CODE) {
                $returnTotals[] = $orderTotal;
            }
        }

        return $returnTotals;
    }
}
