<?php
/**
 * @copyright Copyright (c) 2016 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\Surcharge\Model;

use Fooman\Surcharge\Helper\SurchargeConfig;
use Fooman\Surcharge\Model\ResourceModel\Surcharge\Collection;
use Fooman\Surcharge\Model\ResourceModel\Surcharge\CollectionFactory;
use Magento\Checkout\Model\ConfigProviderInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

class SurchargeConfigProvider implements ConfigProviderInterface
{

    const XML_PATH_DISPLAY_CART_SURCHARGE = 'tax/cart_display/fooman_surcharge';
    const XML_PATH_DISPLAY_CART_SURCHARGE_ZERO = 'tax/cart_display/fooman_surcharge_zero';
    const XML_PATH_EXPLANATION_AS_TOOLTIP = 'sales/surcharge/explanations_as_tooltip';

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var SurchargeConfig
     */
    private $surchargeConfigHelper;

    /**
     * @var Collection
     */
    private $collection;

    public function __construct(
        ScopeConfigInterface $scopeConfig,
        CollectionFactory $collectionFactory,
        SurchargeConfig $surchargeConfigHelper,
        StoreManagerInterface $storeManager
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->collectionFactory = $collectionFactory;
        $this->storeManager = $storeManager;
        $this->surchargeConfigHelper = $surchargeConfigHelper;
    }

    public function getConfig()
    {
        $config = [
            'foomanSurchargeConfig' => [
                'active' => $this->isActiveForStore(),
                'isDisplayedTaxInclusive' => $this->isDisplayedTaxInclusive(),
                'isDisplayedTaxExclusive' => $this->isDisplayedTaxExclusive(),
                'isDisplayedBoth' => $this->isDisplayedBoth(),
                'isZeroDisplayed' => $this->isZeroDisplayed(),
                'requiresBillingRefresh' => $this->requiresBillingRefresh(),
                'isDisplayedAsTooltip' => $this->isDisplayedAsTooltip()
            ]
        ];
        $config = array_merge_recursive($config, $this->getSurchargeDetails());

        return $config;
    }

    public function isDisplayedTaxInclusive()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_DISPLAY_CART_SURCHARGE,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        ) == \Magento\Tax\Model\Config::DISPLAY_TYPE_INCLUDING_TAX;
    }

    public function isDisplayedAsTooltip()
    {
        return $this->scopeConfig->isSetFlag(
            self::XML_PATH_EXPLANATION_AS_TOOLTIP,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        );
    }

    public function isDisplayedTaxExclusive()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_DISPLAY_CART_SURCHARGE,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        ) == \Magento\Tax\Model\Config::DISPLAY_TYPE_EXCLUDING_TAX;
    }

    public function isDisplayedBoth()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_DISPLAY_CART_SURCHARGE,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        ) == \Magento\Tax\Model\Config::DISPLAY_TYPE_BOTH;
    }

    public function isZeroDisplayed()
    {
        return $this->scopeConfig->isSetFlag(
            self::XML_PATH_DISPLAY_CART_SURCHARGE_ZERO,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return int
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function isActiveForStore()
    {
        try {
            $collection = $this->getSurcharges();
            return ($collection->getSize() > 0) ? 1 : 0;
        } catch (\Exception $e) {
            return 0;
        }
    }

    /**
     * @return int
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function requiresBillingRefresh()
    {
        try {
            $collection = $this->getSurcharges();
            if ($collection->getSize() === 0) {
                return 0;
            }
            foreach ($collection as $surcharge) {
                $surchargeConfig = $this->surchargeConfigHelper->getConfig($surcharge);
                if ($surchargeConfig->getApplyRegionFilter()
                    && $surchargeConfig->getRegionFilterAddressType() === SurchargeRestrictor::ADDRESS_TYPE_BILL) {
                    return 1;
                }
            }
            return 0;
        } catch (\Exception $e) {
            return 0;
        }
    }

    /**
     * @return \Fooman\Surcharge\Model\ResourceModel\Surcharge\Collection
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function getSurcharges()
    {
        if ($this->collection === null) {
            $currentStoreId = $this->storeManager->getStore()->getId();
            $this->collection = $this->collectionFactory->create();
            $this->collection
                ->addFieldToFilter('store_id', [$currentStoreId, Store::DEFAULT_STORE_ID])
                ->addFieldToFilter('is_active', true);
        }
        return $this->collection;
    }

    private function getSurchargeDetails()
    {
        $surcharges = [];
        $collection = $this->getSurcharges();
        if ($collection->getSize() === 0) {
            return $surcharges;
        }
        foreach ($collection as $surcharge) {
            $surcharges['foomanSurchargeConfig'][$surcharge->getTypeId()] = [
                'explanation' => $surcharge->getExplanation()
            ];
        }
        return $surcharges;
    }
}
