<?php
/**
 * @copyright Copyright (c) 2009 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\ProductFees\Model;

use Fooman\ProductFees\Model\ResourceModel\SurchargeModifier\CollectionFactory;
use Magento\Framework\Exception\CouldNotSaveException;

class SurchargeModifierRepository
{

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var ResourceModel\SurchargeModifier
     */
    private $surchargeModifierResource;

    /**
     * @var SurchargeModifierFactory
     */
    private $surchargeModifierFactory;

    public function __construct(
        CollectionFactory $collectionFactory,
        ResourceModel\SurchargeModifier $surchargeModifierResource,
        SurchargeModifierFactory $surchargeModifierFactory
    ) {
        $this->collectionFactory = $collectionFactory;
        $this->surchargeModifierResource = $surchargeModifierResource;
        $this->surchargeModifierFactory = $surchargeModifierFactory;
    }

    public function getListAsArray($productId)
    {
        $collection = $this->getListByProductId($productId);
        return $collection->toArray()['items'];
    }

    public function save($data, $productId)
    {
        $processed = [];
        $existingIds = $this->getListByProductId($productId)->getAllIds();
        if (null !== $data && is_array($data)) {
            foreach ($data as $entry) {
                if (isset($entry['linked_surcharge_id'])) {
                    if (isset($processed[$entry['linked_surcharge_id']])) {
                        throw new CouldNotSaveException(__(
                            'Surcharge with ID %1 is used more than once. Please create separate surcharges instead.',
                            (int)$entry['linked_surcharge_id']
                        ));
                    }
                    $surchargeModifier = $this->surchargeModifierFactory->create();
                    $surchargeModifier->setLinkedProductId((int)$productId);
                    $surchargeModifier->setLinkedSurchargeId((int)$entry['linked_surcharge_id']);
                    $surchargeModifier->setType((int)$entry['type']);
                    $surchargeModifier->setAmount($entry['amount']);
                    if ($this->shouldSetId($productId, $entry)) {
                        $surchargeModifier->setId($entry['id']);
                    }

                    $this->surchargeModifierResource->save($surchargeModifier);
                    $existingKey = array_search($surchargeModifier->getId(), $existingIds);
                    if ($existingKey !== false) {
                        unset($existingIds[$existingKey]);
                    }
                    $processed[$entry['linked_surcharge_id']] = true;
                }
            }
        }
        if ($existingIds) {
            $this->surchargeModifierResource->deleteByIds($existingIds);
        }
    }

    private function shouldSetId($productId, $entry)
    {
        if (!isset($entry['id'])) {
            return false;
        }
        return (int)$entry['linked_product_id'] === (int)$productId;
    }

    public function getListByProductId($productId)
    {
        $collection = $this->collectionFactory->create();
        $collection->addFieldToFilter('linked_product_id', $productId);
        return $collection;
    }

    public function getBySurchargeIdAndProductId($surchargeId, $productId)
    {
        $collection = $this->getListByProductId($productId);
        $collection->addFieldToFilter('linked_surcharge_id', $surchargeId);
        $collection->setPageSize(1);
        return $collection->getFirstItem();
    }

    public function checkIfExcluded($surchargeId, $productId)
    {
        $collection = $this->getListByProductId($productId);
        $collection->addFieldToFilter('linked_surcharge_id', $surchargeId);
        $collection->addFieldToFilter('type', ProductType::MOD_EXCLUDE);
        $collection->setPageSize(1);
        return $collection->getSize() == 1;
    }

    public function checkIfIncluded($surchargeId, $productId)
    {
        $collection = $this->getListByProductId($productId);
        $collection->addFieldToFilter('linked_surcharge_id', $surchargeId);
        $collection->addFieldToFilter('type', ProductType::MOD_INCLUDE);
        $collection->setPageSize(1);
        return $collection->getSize() == 1;
    }

    public function checkIfAppliesAfterCancellations($surchargeId, $quote)
    {
        $allItems = $quote->getAllItems();
        if (!$allItems) {
            return true;
        }

        $cancelAllPossible = true;
        foreach ($allItems as $item) {
            $shouldCancelItem = $this->checkIfItemCancels($surchargeId, $item->getProductId());
            if (!$shouldCancelItem->getId()) {
                $cancelAllPossible = false;
                continue;
            }
            // Strict comparison doesn't work here
            if ($shouldCancelItem->getType() == ProductType::MOD_CANCEL_ONE) {
                return false;
            }
        }
        //if we reach here with a true value all items have had a MOD_CANCEL_ALL modifier
        //send back false to indicate the surcharge should not get applied
        return !$cancelAllPossible;
    }

    public function checkIfItemCancels($surchargeId, $productId)
    {
        $collection = $this->getListByProductId($productId);
        $collection->addFieldToFilter('linked_surcharge_id', $surchargeId);
        $collection->addFieldToFilter(
            'type',
            [
                'in' =>
                    [
                        ProductType::MOD_CANCEL_ONE,
                        ProductType::MOD_CANCEL_ALL
                    ]
            ]
        );
        $collection->setPageSize(1);
        return $collection->getFirstItem();
    }
}
