<?php
/**
 * @copyright Copyright (c) 2009 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\ProductFees\Model;

use Fooman\ProductFees\Model\Config\Source\Type;
use Fooman\Surcharge\Helper\SurchargeConfig;
use Fooman\Surcharge\Model\SurchargeCalculationFactory;

class ProductType implements \Fooman\Surcharge\Api\Data\TypeInterface
{

    const PER_ORDER = 'per_order';
    const PER_PRODUCT = 'per_product';
    const PER_ITEM = 'per_item';

    const MOD_DIRECT = 1;
    const MOD_MIN = 2;
    const MOD_EXCLUDE = 3;
    const MOD_INCLUDE = 4;
    const MOD_CANCEL_ONE = 5;
    const MOD_CANCEL_ALL = 6;

    /**
     * @var string
     */
    private $type = 'product';

    /**
     * @var SurchargeModifierRepository
     */
    private $modifierRepository;

    /**
     * @var SurchargeConfig
     */
    private $surchargeConfigHelper;

    /**
     * @var SurchargeCalculationFactory
     */
    private $surchargeCalculationFactory;

    /**
     * @param SurchargeModifierRepository $modifierRepository
     * @param SurchargeConfig $surchargeConfigHelper
     * @param SurchargeCalculationFactory $surchargeCalculationFactory
     */
    public function __construct(
        SurchargeModifierRepository $modifierRepository,
        SurchargeConfig $surchargeConfigHelper,
        SurchargeCalculationFactory $surchargeCalculationFactory
    ) {
        $this->modifierRepository = $modifierRepository;
        $this->surchargeConfigHelper = $surchargeConfigHelper;
        $this->surchargeCalculationFactory = $surchargeCalculationFactory;
    }

    /**
     * @param \Fooman\Surcharge\Api\SurchargeInterface            $surcharge
     * @param \Magento\Quote\Api\Data\CartInterface               $quote
     * @param \Magento\Quote\Api\Data\ShippingAssignmentInterface $shippingAssignment
     *
     * @return \Fooman\Totals\Model\QuoteAddressTotal[]
     */
    public function calculate(
        \Fooman\Surcharge\Api\SurchargeInterface $surcharge,
        \Magento\Quote\Api\Data\CartInterface $quote,
        \Magento\Quote\Api\Data\ShippingAssignmentInterface $shippingAssignment
    ) {
        $config = $this->surchargeConfigHelper->getConfig($surcharge);
        $surchargeCalc = $this->surchargeCalculationFactory
            ->create(['quote' => $quote, 'surcharge' => $surcharge, 'assignment' => $shippingAssignment]);

        if (!$surchargeCalc->shouldSurchargeApply($quote, $config, $surchargeCalc->getCurrentSubTotal())) {
            return [];
        }

        $surchargeAmount = 0;
        $onceApplied = false;
        $multiplier = 1;

        $items = $shippingAssignment->getItems();
        if (!empty($items)) {
            foreach ($items as $quoteItem) {
                $productSurcharge = $this->modifierRepository->getBySurchargeIdAndProductId(
                    $surcharge->getId(),
                    $quoteItem->getProductId()
                );

                if ($productSurcharge->getId()) {
                    $parentMultiplier = 1;
                    if ($quoteItem->getParentItem()) {
                        $parentMultiplier = $quoteItem->getParentItem()->getQty();
                    }
                    if ($config->getType() === self::PER_ITEM) {
                        $multiplier = $quoteItem->getQty() * $parentMultiplier;
                    }
                    if ($config->getMaxItemMultiplier() && $multiplier > $config->getMaxItemMultiplier()) {
                        $multiplier = $config->getMaxItemMultiplier();
                    }
                    if ($config->getType() === self::PER_ORDER) {
                        $surchargeAmount = max(
                            $surchargeAmount,
                            $productSurcharge->getAmount() * $multiplier
                        );
                    } else {
                        $surchargeAmount += $productSurcharge->getAmount() * $multiplier;
                    }
                    $onceApplied = true;
                }
            }

            if ($onceApplied) {
                if ($config->getMaxFixed() && $surchargeAmount > $config->getMaxFixed()) {
                    $surchargeAmount = $config->getMaxFixed();
                }
                return [$surchargeCalc->processTotals(false, $surchargeAmount, true)];
            }
        }
        return [];
    }

    /**
     * @return string
     */
    public function getLabel()
    {
        return __('Product Fee');
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }
}
