<?php
/**
 * @copyright Copyright (c) 2009 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\ProductFees\Model;

use Fooman\ProductFees\Model\Config\Source\Type;
use Fooman\Surcharge\Helper\SurchargeConfig;
use Fooman\Surcharge\Model\SurchargeCalculationFactory;
use Fooman\Totals\Model\QuoteAddressTotalFactory;

class MinSpendType implements \Fooman\Surcharge\Api\Data\TypeInterface
{

    /**
     * @var string
     */
    private $type = 'product';

    /**
     * @var SurchargeModifierRepository
     */
    private $modifierRepository;

    /**
     * @var SurchargeConfig
     */
    private $surchargeConfigHelper;

    /**
     * @var SurchargeCalculationFactory
     */
    private $surchargeCalculationFactory;

    /**
     * @param SurchargeModifierRepository $productResource
     * @param QuoteAddressTotalFactory $quoteAddressTotalFactory
     * @param SurchargeConfig $surchargeConfigHelper
     */
    public function __construct(
        SurchargeModifierRepository $modifierRepository,
        SurchargeConfig $surchargeConfigHelper,
        SurchargeCalculationFactory $surchargeCalculationFactory
    ) {
        $this->modifierRepository = $modifierRepository;
        $this->surchargeConfigHelper = $surchargeConfigHelper;
        $this->surchargeCalculationFactory = $surchargeCalculationFactory;
    }

    /**
     * @param \Fooman\Surcharge\Api\SurchargeInterface            $surcharge
     * @param \Magento\Quote\Api\Data\CartInterface               $quote
     * @param \Magento\Quote\Api\Data\ShippingAssignmentInterface $shippingAssignment
     *
     * @return \Fooman\Totals\Model\QuoteAddressTotal[]
     */
    public function calculate(
        \Fooman\Surcharge\Api\SurchargeInterface $surcharge,
        \Magento\Quote\Api\Data\CartInterface $quote,
        \Magento\Quote\Api\Data\ShippingAssignmentInterface $shippingAssignment
    ) {
        $config = $this->surchargeConfigHelper->getConfig($surcharge);
        $surchargeCalc = $this->surchargeCalculationFactory
            ->create(['quote' => $quote, 'surcharge' => $surcharge, 'assignment' => $shippingAssignment]);

        if (!$surchargeCalc->shouldSurchargeApply($quote, $config, $surchargeCalc->getCurrentSubTotal())) {
            return [];
        }

        $surchargeAmount = $config->getMaxFixed();
        $onceApplied = false;

        $items = $shippingAssignment->getShipping()->getAddress()->getAllItems();
        if (!empty($items)) {
            foreach ($items as $quoteItem) {
                $included = $this->modifierRepository->checkIfIncluded(
                    $surcharge->getId(),
                    $quoteItem->getProductId()
                );

                if ($included) {
                    $onceApplied = true;
                    $surchargeAmount -= $this->getAmountForItem($quoteItem);
                }
            }

            if ($onceApplied && $surchargeAmount > 0) {
                return [$surchargeCalc->processTotals(false, $surchargeAmount, true)];
            }
        }
        return [];
    }

    public function getAmountForItem($quoteItem)
    {
        if (!$quoteItem->getParentItemId()) {
            return $quoteItem->getBaseRowTotal();
        }
        $parentItem = $quoteItem->getParentItem();
        if (!$parentItem) {
            return 0;
        }
        if ($parentItem->getProductType() == 'configurable') {
            return $parentItem->getBaseRowTotal();
        }
        if ($parentItem->getProductType() == 'bundle') {
            return $quoteItem->getBaseRowTotal();
        }
        return 0;
    }

    /**
     * @return string
     */
    public function getLabel()
    {
        return __('Minimum Product Spend');
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }
}
