<?php

namespace Fooman\OrderFees\Model;

use Fooman\Surcharge\Helper\SurchargeConfig;
use Fooman\Surcharge\Model\SurchargeCalculationFactory;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Api\CartRepositoryInterface;
use Fooman\OrderFees\Api\OrderFeesManagementInterface;
use Fooman\Surcharge\Helper\Surcharge as SurchargeHelper;
use Magento\Quote\Api\Data\CartExtensionInterfaceFactory;
use Magento\Quote\Model\Quote;
use Fooman\Surcharge\Model\ResourceModel\Surcharge\CollectionFactory;

/**
 * Order fees management object.
 */
class OrderFeesManagement implements OrderFeesManagementInterface
{
    /**
     * @var CartRepositoryInterface
     */
    private $quoteRepository;

    /**
     * @var SurchargeHelper
     */
    private $surchargeHelper;

    /**
     * @var EnabledOrderFeesFactory
     */
    private $enabledOrderFeesFactory;

    /**
     * @var CartExtensionInterfaceFactory
     */
    private $cartExtensionInterfaceFactory;

    /**
     * @var CollectionFactory
     */
    private $surchargeCollectionFactory;

    /**
     * @var OrderFeeType
     */
    private $orderFeeType;

    /**
     * @var SurchargeCalculationFactory
     */
    private $surchargeCalculationFactory;

    /**
     * @var SurchargeConfig
     */
    private $surchargeConfigHelper;

    /**
     * @var AvailableFeeFactory
     */
    private $availableFeeFactory;

    /**
     * @var FeeFactory
     */
    private $feeFactory;

    /**
     * @param CartRepositoryInterface $quoteRepository
     * @param SurchargeHelper $surchargeHelper
     * @param EnabledOrderFeesFactory $enabledOrderFeesFactory
     * @param CartExtensionInterfaceFactory $cartExtensionInterfaceFactory
     * @param CollectionFactory $surchargeCollectionFactory
     * @param OrderFeeType $orderFeeType
     * @param SurchargeCalculationFactory $surchargeCalculationFactory
     * @param SurchargeConfig $surchargeConfigHelper
     * @param AvailableFeeFactory $availableFeeFactory
     * @param FeeFactory $feeFactory
     */
    public function __construct(
        CartRepositoryInterface $quoteRepository,
        SurchargeHelper $surchargeHelper,
        EnabledOrderFeesFactory $enabledOrderFeesFactory,
        CartExtensionInterfaceFactory $cartExtensionInterfaceFactory,
        CollectionFactory $surchargeCollectionFactory,
        OrderFeeType $orderFeeType,
        SurchargeCalculationFactory $surchargeCalculationFactory,
        SurchargeConfig $surchargeConfigHelper,
        AvailableFeeFactory $availableFeeFactory,
        FeeFactory $feeFactory
    ) {
        $this->quoteRepository = $quoteRepository;
        $this->surchargeHelper = $surchargeHelper;
        $this->enabledOrderFeesFactory = $enabledOrderFeesFactory;
        $this->cartExtensionInterfaceFactory = $cartExtensionInterfaceFactory;
        $this->surchargeCollectionFactory = $surchargeCollectionFactory;
        $this->orderFeeType = $orderFeeType;
        $this->surchargeCalculationFactory = $surchargeCalculationFactory;
        $this->surchargeConfigHelper = $surchargeConfigHelper;
        $this->availableFeeFactory = $availableFeeFactory;
        $this->feeFactory = $feeFactory;
    }

    /**
     * @inheritDoc
     */
    public function get($cartId)
    {
        /** @var Quote $quote */
        $quote = $this->quoteRepository->getActive($cartId);
        $cartExtension = $quote->getExtensionAttributes();
        if (null === $cartExtension) {
            return [];
        }
        $enabledFees = $cartExtension->getFoomanEnabledOrderFees();
        if (!$enabledFees) {
            return [];
        }
        $result = [];
        foreach ($enabledFees->getItems() as $typeId => $active) {
            if ($active) {
                $fee = $this->feeFactory->create();
                $result[] = $fee->setTypeId($typeId);
            }
        }
        return $result;
    }

    /**
     * @inheritDoc
     */
    public function getAvailable($cartId)
    {
        $result = [];
        $quote = $this->quoteRepository->getActive($cartId);
        $enabledSurcharges = $this->getEnabledOrderFeesSurcharges($quote->getStoreId());
        foreach ($enabledSurcharges as $enabledSurcharge) {
            $surchargeConfig = $this->surchargeConfigHelper->getConfig($enabledSurcharge);
            if ($this->isSurchargeApplicable($enabledSurcharge, $quote, $surchargeConfig)) {
                $result[] = $this->availableFeeFactory->create(
                    [
                        'typeId' => $enabledSurcharge->getTypeId(),
                        'label' => $enabledSurcharge->getDescription(),
                        'checked' => $this->isOrderFeeApplied($enabledSurcharge, $quote),
                        'introText' => $surchargeConfig->getIntroText()
                    ]
                );
            }
        }
        return $result;
    }

    private function isOrderFeeApplied($surcharge, $quote)
    {
        return $this->orderFeeType->surchargeApplies($quote, $surcharge);
    }

    private function isSurchargeApplicable($surcharge, $quote, $surchargeConfig)
    {
        if (!$surchargeConfig->getIsOptional()) {
            return false;
        }

        $surchargeCalculation = $this->surchargeCalculationFactory
            ->create(['quote' => $quote, 'surcharge' => $surcharge]);

        return $surchargeCalculation->shouldSurchargeApply(
            $quote,
            $surchargeConfig,
            $surchargeCalculation->getCurrentSubTotal()
        );
    }

    /**
     * @return Fooman\Surcharge\Model\ResourceModel\Surcharge\Collection
     */
    private function getEnabledOrderFeesSurcharges($storeId)
    {
        $collection = $this->surchargeCollectionFactory->create();
        $collection->addFieldToFilter('type', $this->orderFeeType->getType());
        $collection->addFieldToFilter('is_active', 1);
        $collection->addFieldToFilter('store_id', [$storeId, \Magento\Store\Model\Store::DEFAULT_STORE_ID]);

        return $collection;
    }

    /**
     * @inheritDoc
     */
    public function set($cartId, $enabledOrderFees)
    {
        /** @var Quote $quote */
        $quote = $this->quoteRepository->getActive($cartId);
        if (!$quote->getItemsCount()) {
            throw new NoSuchEntityException(__('The "%1" Cart doesn\'t contain products.', $cartId));
        }
        if (!$quote->getStoreId()) {
            throw new NoSuchEntityException(__("Cart isn't assigned to correct store"));
        }

        /** @var EnabledOrderFees $enabledFees */
        $enabledFees = $this->enabledOrderFeesFactory->create();
        $enabledOrderFeesTypeIds = explode(',', $enabledOrderFees);
        foreach ($enabledOrderFeesTypeIds as $enabledOrderFeesTypeId) {
            $surcharge = $this->surchargeHelper->getSurchargeByTypeId($enabledOrderFeesTypeId);
            $enabledFees->enable($surcharge);
        }

        $cartExtension = $quote->getExtensionAttributes();
        if (null === $cartExtension) {
            $cartExtension = $this->cartExtensionInterfaceFactory->create();
        }
        $cartExtension->setFoomanEnabledOrderFees($enabledFees);
        $quote->setExtensionAttributes($cartExtension);

        try {
            $this->quoteRepository->save($quote->collectTotals());
        } catch (LocalizedException $e) {
            throw new CouldNotSaveException(__("The order fee couldn't be applied: " .$e->getMessage()), $e);
        } catch (\Exception $e) {
            throw new CouldNotSaveException(
                __("The order fee couldn't be applied. Verify the order fee and try again."),
                $e
            );
        }
        return true;
    }

    /**
     * @inheritDoc
     */
    public function remove($cartId)
    {
        /** @var Quote $quote */
        $quote = $this->quoteRepository->getActive($cartId);
        if (!$quote->getItemsCount()) {
            throw new NoSuchEntityException(__('The "%1" Cart doesn\'t contain products.', $cartId));
        }

        $cartExtension = $quote->getExtensionAttributes();
        if (null === $cartExtension) {
            $cartExtension = $this->cartExtensionInterfaceFactory->create();
        }
        /** @var EnabledOrderFees $enabledFees */
        $enabledFees = $this->enabledOrderFeesFactory->create();
        $cartExtension->setFoomanEnabledOrderFees($enabledFees);
        $quote->setExtensionAttributes($cartExtension);

        try {
            $this->quoteRepository->save($quote->collectTotals());
        } catch (\Exception $e) {
            throw new CouldNotDeleteException(
                __("The enabled order fee couldn't be deleted. Verify the order fee and try again.")
            );
        }

        return true;
    }
}