<?php
/**
 * @copyright Copyright (c) 2009 Fooman Limited (http://www.fooman.co.nz)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace Fooman\OrderFees\Model;

use Fooman\Surcharge\Api\Data\TypeInterface;
use Fooman\Surcharge\Api\SurchargeInterface;
use Fooman\Surcharge\Helper\SurchargeConfig;
use Fooman\Surcharge\Model\SurchargeCalculationFactory;
use Magento\Quote\Api\Data\CartInterface;
use Magento\Quote\Api\Data\ShippingAssignmentInterface;

class OrderFeeType implements TypeInterface
{
    /**
     * @var string
     */
    private $type = 'orderfee';

    /**
     * @var SurchargeCalculationFactory
     */
    private $surchargeCalculationFactory;

    /**
     * @var SurchargeConfig
     */
    private $surchargeConfigHelper;

    /**
     * @param SurchargeCalculationFactory $surchargeCalculationFactory
     * @param SurchargeConfig $surchargeConfigHelper
     */
    public function __construct(
        SurchargeCalculationFactory $surchargeCalculationFactory,
        SurchargeConfig $surchargeConfigHelper
    ) {
        $this->surchargeCalculationFactory = $surchargeCalculationFactory;
        $this->surchargeConfigHelper = $surchargeConfigHelper;
    }

    /**
     * @param SurchargeInterface $surcharge
     * @param CartInterface $quote
     * @param ShippingAssignmentInterface $shippingAssignment
     * @return array
     */
    public function calculate(
        SurchargeInterface $surcharge,
        CartInterface $quote,
        ShippingAssignmentInterface $shippingAssignment
    ) {
        $surchargeCalculation = $this->surchargeCalculationFactory
            ->create(['quote' => $quote, 'surcharge' => $surcharge, 'assignment' => $shippingAssignment]);

        if ($this->surchargeApplies($quote, $surcharge)) {
            return [$surchargeCalculation->processTotals()];
        }

        return [];
    }

    /**
     * @return string
     */
    public function getLabel()
    {
        return __('Order Fee');
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @param CartInterface $quote
     * @param $surcharge
     */
    public function surchargeApplies(CartInterface $quote, $surcharge)
    {
        $config = $this->surchargeConfigHelper->getConfig($surcharge);
        if (!$config->getIsOptional()) {
            return true;
        }
        $cartExtension = $quote->getExtensionAttributes();

        if ($cartExtension) {
            /** @var EnabledOrderFees $enabled */
            $enabled = $cartExtension->getFoomanEnabledOrderFees();
            if ($enabled) {
                return $enabled->isActive($surcharge);
            }
        }

        return false;
    }
}
