<?php
declare(strict_types=1);

namespace Xtwo\wishlistApi\Model;

use Magento\Wishlist\Model\WishlistFactory;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Framework\Event\ManagerInterface as EventManager;
use Magento\Framework\Exception\NoSuchEntityException;
use Xtwo\AttributeOption\Helper\Image as ImageHelper;
use Magento\Store\Model\StoreManagerInterface;
use Magento\CatalogInventory\Api\StockRegistryInterface;
use Xtwo\CustomApis\Helper\Data as CustomApisHelper;

class WishlistManagement implements \Xtwo\wishlistApi\Api\WishlistManagementInterface
{
    private $wishlistFactory;
    private $productRepository;
    private $eventManager;
    protected $imageHelper;
    protected $stockRegistry;
    protected $customApisHelper;
	protected $storeManager;

    public function __construct(
        WishlistFactory $wishlistFactory,
        ProductRepositoryInterface $productRepository,
        EventManager $eventManager,
        ImageHelper $imageHelper,
        StockRegistryInterface $stockRegistry,
		StoreManagerInterface $storeManager,
		\Magento\Tax\Model\Calculation $taxCalculation,
        CustomApisHelper $customApisHelper
    ) {
        $this->wishlistFactory = $wishlistFactory;
        $this->productRepository = $productRepository;
        $this->eventManager = $eventManager;
        $this->imageHelper = $imageHelper;
        $this->stockRegistry = $stockRegistry;
		$this->storeManager = $storeManager;
		$this->taxCalculation = $taxCalculation;
        $this->customApisHelper = $customApisHelper;
    }

    public function get(int $customerId)
    {
        $wishlist = $this->wishlistFactory->create()->loadByCustomerId($customerId);
        if (!$wishlist->getId()) {
            throw new NoSuchEntityException(__('Customer does not yet have a wishlist', null, 1));
        }

        $filteredItems = [];
        foreach ($wishlist->getItemCollection() as $item) {
            $product = $item->getProduct();
            preg_match('/href="(.*?)"/', $product->getAttributeText('manufacturer'), $matches);
            $layerImageOptionId = $product->getLayerImages();
            $layerImagesPath = $this->getLayerImagesPath($layerImageOptionId);
            $manufacturerImagePath = $this->getManufacturerImagePath($product);
            $stockItem = $this->stockRegistry->getStockItem($product->getId(), $product->getStore()->getWebsiteId());
			
			// Calculate price with VAT
			$taxClassId = $product->getTaxClassId();
			$store      = $this->storeManager->getStore();
			$request = $this->taxCalculation->getRateRequest(null, null, null, $store);
			$taxRate = $this->taxCalculation->getRate($request->setProductClassId($taxClassId));
			
			$price = $product->getPrice();
			$price = round((float)$price, 2);
			$basePrice = $price / 1.19;
			$priceWithTax = $basePrice + ($basePrice * ($taxRate / 100));
			
			$msrpPrice = $product->getMsrp();
			$msrpBasePrice = $msrpPrice / 1.19;
			$msrpPriceWithTax = $msrpBasePrice + ($msrpBasePrice * ($taxRate / 100));
			
            $filteredItems[] = [
                'id' => $product->getId(),
                'name' => $product->getName(),
                'image' => $product->getData('image'),
                'sku' => $product->getSku(),
                'price' => (string)$priceWithTax,
                'delivery_time' => $product->getAttributeText('delivery_time'),
                'product_url' => $product->getUrlKey(),
                'msrp' => (string)$msrpPriceWithTax,
                'layer_images' => $layerImagesPath,
                'manufacturer' => [
                                    strip_tags($product->getAttributeText('manufacturer')),
                                    isset($matches[1]) ? str_replace('.html', '', $matches[1]) : '',
                                    $manufacturerImagePath
                                 ],
                'stock' => $stockItem->getQty(),
                'is_in_stock' => $stockItem->getIsInStock(),
                'is_in_wishlist' => true,
                'average_rating' => $this->customApisHelper->getAverageRating($product->getId()),
                'show_addtocart_button' => $this->customApisHelper->getButtonStatus($product)
            ];
        }

        $wishlist['items'] = $filteredItems;
        return $wishlist;
    }

    public function add(int $customerId, $item)
    {
        $productId = $item->getProduct();
        if ($productId === null) {
            throw new \Magento\Framework\Exception\LocalizedException(__("Invalid product. Please select a valid product."));
        }

        $product = $this->productRepository->getById($productId);

        $wishlist = $this->wishlistFactory->create()->loadByCustomerId($customerId, true);
        $wishlist->addNewItem($product);
        $wishlist->save();
        return true;
    }

    public function delete(int $customerId, int $productId): bool
    {
        $wishlist = $this->wishlistFactory->create()->loadByCustomerId($customerId);
        if (!$wishlist->getId()) {
            throw new NoSuchEntityException(__('Customer does not yet have a wishlist', null, 1));
        }

        $item = $wishlist->getItemCollection()->getItemByColumnValue('product_id', $productId);
        if (!$item) {
            throw new NoSuchEntityException(__('Item with product ID %1 not found in the wishlist', $productId), null, 1);
        }

        $item->delete();
        return true;
    }
    private function getLayerImagesPath($optionId)
    {
        if ($this->imageHelper->getLayerImage($optionId)) {
            $imagePath = 'attributeoption/image/' . $this->imageHelper->getLayerImage($optionId);
            return $imagePath;
        }else{
            return '';
        }
    }
    private function getManufacturerImagePath($data)
    {
        if ($this->imageHelper->getManufacturerImage($data)) {
            $imagePath = 'attributeoption/image/' . $this->imageHelper->getManufacturerImage($data);
            return $imagePath;
        } else {
            return '';
        }
    }
}
