<?php
/**
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_Automationshell
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 */

namespace Xtwo\UpdatePrices\Model\ResourceModel\Product;

use Xtwo\UpdatePrices\Api\PriceInterface;

/**
 * Special price resource.
 */
class Price implements PriceInterface
{
    /**
     * Price storage table.
     *
     * @var string
     */
    private $priceTable = 'catalog_product_entity_decimal';
    
    /**
     * @var \Magento\Catalog\Model\ResourceModel\Attribute
     */
    private $attributeResource;
    
    /**
     * @var \Magento\Catalog\Api\ProductAttributeRepositoryInterface
     */
    private $attributeRepository;
    
    /**
     * @var \Magento\Catalog\Model\ProductIdLocatorInterface
     */
    private $productIdLocator;
    
    /**
     * Metadata pool.
     *
     * @var \Magento\Framework\EntityManager\MetadataPool
     */
    private $metadataPool;
    
    /**
     * Special Price attribute ID.
     *
     * @var int
     */
    private $priceAttributeId;
    
    /**
     * @param \Magento\Catalog\Model\ResourceModel\Attribute $attributeResource
     * @param \Magento\Catalog\Api\ProductAttributeRepositoryInterface $attributeRepository
     * @param \Magento\Catalog\Model\ProductIdLocatorInterface $productIdLocator
     * @param \Magento\Framework\EntityManager\MetadataPool $metadataPool
     */
    public function __construct(
        \Magento\Catalog\Model\ResourceModel\Attribute $attributeResource,
        \Magento\Catalog\Api\ProductAttributeRepositoryInterface $attributeRepository,
        \Magento\Catalog\Model\ProductIdLocatorInterface $productIdLocator,
        \Magento\Framework\EntityManager\MetadataPool $metadataPool
    ) {
        $this->attributeResource = $attributeResource;
        $this->attributeRepository = $attributeRepository;
        $this->productIdLocator = $productIdLocator;
        $this->metadataPool = $metadataPool;
    }
    
    /**
     * {@inheritdoc}
     */
    public function get($storeId, $entityIds, $limitPerChunk, $lastEntityId)
    {
        $priceTable = $this->attributeResource->getTable($this->priceTable);
        $linkField = $this->getEntityLinkField();
        $select = $this->attributeResource->getConnection()
            ->select()
            ->from(
                $priceTable,
                [
                    $linkField,
                    'value',
                ]
            )
            ->where($priceTable . '.store_id = ?', $storeId)
            ->where($priceTable . '.attribute_id = ?', $this->getPriceAttributeId());
        if (!empty($entityIds)) {
            $select->where($priceTable . '.' . $linkField . ' IN (?)', $entityIds);
        } else {
            $select->where($priceTable. '.' . $linkField . '> ' . $lastEntityId);
            $select->limit($limitPerChunk);
        }
        $select->order('entity_id ASC');
        
        return $this->attributeResource->getConnection()->fetchAll($select);
    }
    
    /**
     * {@inheritdoc}
     */
    public function delete($storeId, $productsIdsSamePricesAsDefault)
    {
        $connection = $this->attributeResource->getConnection();
        $connection->beginTransaction();
        try {
            $this->attributeResource->getConnection()->delete(
                $this->attributeResource->getTable($this->priceTable),
                [
                    'attribute_id = ?' => $this->getPriceAttributeId(),
                    'store_id = ?' => $storeId,
                    $this->getEntityLinkField() . ' IN (?)' => $productsIdsSamePricesAsDefault
                ]
            );
            $connection->commit();
        } catch (\Exception $exception) {
            $connection->rollBack();
            throw new \Magento\Framework\Exception\CouldNotDeleteException(
                __('Could not delete Prices'),
                $exception
            );
        }
    }

    /**
     * Get special price attribute ID.
     *
     * @return int|null
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function getPriceAttributeId()
    {
        if (!$this->priceAttributeId) {
            $this->priceAttributeId = $this->attributeRepository->get('price')->getAttributeId();
        }
        
        return $this->priceAttributeId;
    }

    /**
     * Get link field.
     *
     * @return string
     * @throws \Exception
     */
    private function getEntityLinkField()
    {
        return $this->metadataPool->getMetadata(\Magento\Catalog\Api\Data\ProductInterface::class)
            ->getLinkField();
    }
}
