<?php
/**
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_Automationshell
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 */

namespace Xtwo\UpdatePrices\Console\Command;

use Xtwo\UpdatePrices\Model\ResourceModel\Product\Price;
use Magento\Catalog\Model\ResourceModel\Product\Action;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\Catalog\Model\ResourceModel\Product;

/**
 * Class UpdatePricesCommand
 *
 * @package Core\UpdatePrices\Console\Command
 */
class UpdatePricesCommand extends Command
{
    const LIMIT_PER_CHUNK = 5000;
    const LIMIT_ROUND = 80;

    /**
     * Names of input arguments or options
     */
    const INPUT_KEY_OPTION_CALCULATE = 'calculate';
    const INPUT_KEY_OPTION_STORE = 'store';
    const INPUT_KEY_OPTION_SURCHARGE = 'surcharge';
    const INPUT_KEY_OPTION_DEBUG_PRODUCT_ID = 'debugProductId';
    const INPUT_KEY_OPTION_SKU = 'sku';

    protected $lastEntityId = 0;

    /** @var array container for result */
    protected $processingInfoData = [];

    protected $debugInfoForProduct = [];
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private $storeManager;
    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    private $scopeConfig;
    /**
     * @var \Psr\Log\LoggerInterface
     */
    private $logger;
    /**
     * @var mixed
     */
    private $multiplier;
    /**
     * @var \Magento\Catalog\Model\ResourceModel\Product
     */
    private $productResource;
    /**
     * @var \Core\UpdatePrices\Console\Command\Price
     */
    private $productPriceResource;
    /**
     * @var OutputInterface
     */
    private $output;
    /**
     * @var \Magento\Catalog\Model\ResourceModel\Product\CollectionFactory
     */
    private $productCollectionFactory;
    /**
     * @var \Magento\Eav\Model\ResourceModel\Entity\Attribute\Set\CollectionFactory
     */
    private $attributeSetCollectionFactory;
    /**
     * @var string
     */
    private $surcharge;
    /**
     * @var \Magento\Catalog\Model\ResourceModel\Product\Action
     */
    private $productResourceAction;
    /**
     * @var \Symfony\Component\Console\Input\InputInterface
     */
    private $input;
    /**
     * @var bool|string|string[]|null
     */
    private $debugProductId;
    /** @var array debug low price info */
    private $lowPriceArray = [];

    /**
     * UpdatePricesCommand constructor.
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Psr\Log\LoggerInterface $logger
     * @param \Magento\Catalog\Model\ResourceModel\Product $productResource
     * @param \Xtwo\UpdatePrices\Model\ResourceModel\Product\Price $productPriceResource
     * @param \Magento\Catalog\Model\ResourceModel\Product\CollectionFactory $productCollectionFactory
     * @param \Magento\Eav\Model\ResourceModel\Entity\Attribute\Set\CollectionFactory $attributeSetCollectionFactory
     * @param \Magento\Catalog\Model\ResourceModel\Product\Action $productResourceAction
     */
    public function __construct(
        StoreManagerInterface $storeManager,
        ScopeConfigInterface $scopeConfig,
        LoggerInterface $logger,
        Product $productResource,
        Price $productPriceResource,
        CollectionFactory $productCollectionFactory,
        \Magento\Eav\Model\ResourceModel\Entity\Attribute\Set\CollectionFactory $attributeSetCollectionFactory,
        Action $productResourceAction
    )
    {
        parent::__construct();

        $this->storeManager = $storeManager;
        $this->scopeConfig = $scopeConfig;
        $this->logger = $logger;
        $this->productResource = $productResource;
        $this->productPriceResource = $productPriceResource;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->attributeSetCollectionFactory = $attributeSetCollectionFactory;
        $this->productResourceAction = $productResourceAction;
    }

    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this->setName('catalog:product:prices:update');
        $this->setDescription('Updates products prices.');

        $this->addOption(
            self::INPUT_KEY_OPTION_CALCULATE,
            'c',
            InputOption::VALUE_NONE,
            ''
        );
        $this->addOption(
            self::INPUT_KEY_OPTION_STORE,
            's',
            InputOption::VALUE_OPTIONAL
        );
        $this->addOption(
            self::INPUT_KEY_OPTION_SURCHARGE,
            'sur',
            InputOption::VALUE_OPTIONAL
        );
        $this->addOption(
            self::INPUT_KEY_OPTION_DEBUG_PRODUCT_ID,
            'debug',
            InputOption::VALUE_OPTIONAL
        );
        $this->addOption(
            self::INPUT_KEY_OPTION_SKU,
            'sku',
            InputOption::VALUE_OPTIONAL
        );

        $helpText = <<<TEXT
 Usage:  catalog:product:prices:update [--calculate] [--store] [--surcharge]

    calculate  [--store] [--surcharge]
    Examples:
    --calculate [--store xtwostore_de|xtwostore_com|xtwostore_fr|xtwostore_cn]
    Without the store option, all stores will be processed

    --calculate --surcharge 9.90
    Add a static surcharge on all calculated prices. The given surcharge has priority before configured (+) one.

    --sku - if only predefined list of products has to be updated, then SKUs comma-separated.

    [--debugProductId <ID>]
    Example:   --debugProductId 245778
    To show additional debug for this product (Developer only)
TEXT;
        $this->addUsage($helpText);

        parent::configure();
    }

    /**
     * @param InputInterface $input
     * @param OutputInterface $output
     *
     * @return void
     * @throws \Exception
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $store = $input->getOption(self::INPUT_KEY_OPTION_STORE);
        $calculate = $input->getOption(self::INPUT_KEY_OPTION_CALCULATE);
        $sku = $input->getOption(self::INPUT_KEY_OPTION_SKU);
        $this->debugProductId = $input->getOption(self::INPUT_KEY_OPTION_DEBUG_PRODUCT_ID);
        $this->output = $output;
        $this->input = $input;

        if ($calculate !== true) {
            $output->writeln('<error>No calculate option was provided.</error>');
            return;
        }

        if (!empty($store)) {
            $store = $this->storeManager->getStore($store);
            if (!$store->getId()) {
                $output->writeln('<error>Wrong store code.</error>');
                return;
            }
        }
        $stores = $store
            ? array($store)
            : $this->storeManager->getStores(true, true);

        try {
            $this->setSurcharge();

            foreach ($stores as $store) {
                if (!$store->getIsActive()) {
                    continue;
                }

                if (!$this->isActiveForStore($store)) {
                    $this->logAndOutput(
                        sprintf("Skipped for store '%s' as inactive or invalid multiplier\n",
                            $store->getCode()
                        )
                    );

                    continue;
                }

                $round = 0;
                $countPricesSameAsDefault = 0;
                $this->logAndOutput(sprintf('Starting store "%s"', $store->getCode()));
                while (true) {
                    // get default prices (in chunks)
                    $defaultPrices = $this->getPrices($sku);

                    if (empty($defaultPrices) || $round > self::LIMIT_ROUND) {
                        echo "Finalizing\n";
                        break;
                    }

                    // calculate prices with store/attributeset/brand/serie/sku - related multiplier
                    $calculatedPrices = $this->calculateProductPrices($defaultPrices, $store);

                    // if prices are same in default and store, drop
                    $countPricesSameAsDefault += $this->resetStorePricesToDefaultIfEqualToDefault(
                        $defaultPrices,
                        $calculatedPrices,
                        $store->getId()
                    );

                    // get current prices for the store (to get differences)
                    $storePrices = $this->getPrices($sku, $store);

                    // identify the calculated price differences to the current for the store
                    $pricesToBeChanged = $this->filterPriceChanges(
                        $calculatedPrices,
                        $storePrices,
                        $defaultPrices
                    );

                    // save the differences for the store
                    $this->saveNewPricesForStore($pricesToBeChanged, $store);

                    if ($this->isSkuReduced()) {
                        break;
                    }
                    // next chunk
                    $keys = array_keys($defaultPrices);
                    $highest = end($keys);
                    $this->lastEntityId = $highest;

                    $round++;
                }

                $this->logAndOutput(sprintf('Finished "%s"', $store->getCode()));
                $statsText = <<<TEXT
                \nTotal individual on SKU: %s\nTotal individual on Series: %s\nTotal individual on Brand: %s\nTotal individual on AttributSet: %s\nTotal individual on LowPrice: %s\nTotal reset to default: %s\n
TEXT;
                $stats = sprintf($statsText,
                    !empty($this->processingInfoData['sku']) ? (int)$this->processingInfoData['sku'] : 0,
                    !empty($this->processingInfoData['serie']) ? (int)$this->processingInfoData['serie'] : 0,
                    !empty($this->processingInfoData['brand']) ? (int)$this->processingInfoData['brand'] : 0,
                    !empty($this->processingInfoData['set']) ? (int)$this->processingInfoData['set'] : 0,
                    !empty($this->processingInfoData['lowprice']) ? (int)$this->processingInfoData['lowprice'] : 0,
                    (int)$countPricesSameAsDefault
                );

                $this->logAndOutput($stats);

                $this->lastEntityId = 0; // Reset for next store
            }
            if (!empty($this->debugInfoForProduct)) {
                $msg = implode(', ', $this->debugInfoForProduct);
                $this->logAndOutput(sprintf("\nDebug P#%s: %s", $this->debugProductId, $msg));
            }

        } catch (\Exception $exception) {
            $this->logAndOutput(
                sprintf('Stopped processing, Reason Exception: %s', $exception->getMessage()),
                'error'
            );
        }
    }

    /**
     * @param array $pricesToBeChanged
     * @param \Magento\Store\Api\Data\StoreInterface $store
     * @throws \Exception
     */
    protected function saveNewPricesForStore($pricesToBeChanged, $store)
    {
        if (empty($pricesToBeChanged)) {
            $this->logAndOutput('No prices to update');
            return;
        }

        $this->logAndOutput(sprintf('Updating %s prices for %s ', count($pricesToBeChanged), $store->getCode()));

        foreach ($pricesToBeChanged as $productId => $price) {
            $this->productResourceAction->updateAttributes([$productId], ['price' => $price], $store->getId());
        }
    }

    /**
     * @param array $newPrices the calculated prices
     * @param array $storePrices the already overwritten prices of store x
     * @param array $defaultPrices the default prices from store 0
     * @return array
     */
    protected function filterPriceChanges($newPrices, $storePrices, $defaultPrices)
    {
        $diff = [];
        foreach ($newPrices as $productId => $price) {
            $storePrice = array_key_exists($productId, $storePrices) ? $storePrices[$productId] : null;
            if ($this->isDebugProduct($productId)) {
                $this->debugInfoForProduct[] = 'new-price=' . $price;
                $this->debugInfoForProduct[] = 'store-price=' . $storePrice;
                $this->debugInfoForProduct[] = 'default-price=' . (array_key_exists($productId, $defaultPrices)
                        ? $defaultPrices[$productId]
                        : '-');
            }
            if (is_null($storePrice)) {
                $storePrice = $defaultPrices[$productId];
            }
            if ($price > 0 && (!$storePrice || $price != $storePrice)) {
                $diff[$productId] = $price;
            }
        }
        return $diff;
    }

    /**
     * @return bool
     */
    protected function isSkuReduced()
    {
        $skuList = trim($this->input->getOption(self::INPUT_KEY_OPTION_SKU));
        return empty($skuList) ? false : true;
    }

    /**
     * @param array $defaultPrices
     * @param array $calculatedPrices
     * @param int $storeId
     * @return int|void count of deleted rows with same prices as default
     * @throws \Magento\Framework\Exception\CouldNotDeleteException
     */
    public function resetStorePricesToDefaultIfEqualToDefault($defaultPrices, $calculatedPrices, $storeId)
    {
        // to be saved only
        if (!$storeId) {
            return 0;
        }

        $productsIdsSamePricesAsDefault = [];
        foreach ($calculatedPrices as $productId => $newPrice) {
            if (isset($defaultPrices[$productId]) && $defaultPrices[$productId] == $newPrice) {
                $productsIdsSamePricesAsDefault[] = (int)$productId;
            }
        }

        $result = 0;
        if (count($productsIdsSamePricesAsDefault)) {
            $result = $this->productPriceResource->delete($storeId, $productsIdsSamePricesAsDefault);

            if ($result) {
                $this->logAndOutput(sprintf(' Reset %s prices ', $result));
            }
        }

        return $result;
    }

    /**
     * Returns the prices in chunks
     *
     * @param \Magento\Store\Api\Data\StoreInterface $store
     * @param string $sku
     * @return array
     */
    protected function getPrices($sku, $store = null)
    {
        $storeId = $store ? $store->getId() : 0;
        $entityIds = $this->getOptionalEntityIds();
        if ($entityIds === false) {
            $this->logAndOutput(sprintf('No products found for %s', $sku));
            return [];
        }
        $data = $this->productPriceResource->get(
            $storeId,
            $entityIds,
            self::LIMIT_PER_CHUNK,
            $this->lastEntityId
        );

        $resultData = [];
        foreach ($data as $item) {
            $resultData[$item['entity_id']] = $item['value'];
        }

        if (is_null($store) && count($resultData)) {
            $this->logAndOutput(
                sprintf('Processing chunk of %s prices, start@%s ',
                    count($resultData), $this->lastEntityId)
            );
        }
        return $resultData;
    }

    /**
     * If sku value is provided as option for the command, the corresponding product entity ids are returned
     *
     * @return array|bool
     */
    protected function getOptionalEntityIds()
    {
        $skuList = trim($this->input->getOption(self::INPUT_KEY_OPTION_SKU));
        if (!empty($skuList)) {
            $skus = explode(',', $skuList);
            $data = $this->productResource->getProductsIdsBySkus($skus);
            return empty($data) ? false : $data;
        }
        return [];
    }

    /**
     * Returns optional static surcharge to be added to price
     *
     * @return int|string|string[]
     * @throws \Exception
     */
    private function setSurcharge()
    {
        $this->surcharge = trim($this->input->getOption(self::INPUT_KEY_OPTION_SURCHARGE));
        if (empty($this->surcharge)) {
            return 0;
        }
        $value = str_replace(',', '.', $this->surcharge);
        if (!is_numeric($value)) {
            throw new \Exception('Surcharge is invalid: ' . $this->surcharge);
        }
    }

    /**
     * Logging and outputing information
     *
     * @param string $message
     * @param string $type
     */
    private function logAndOutput($message, $type = 'info')
    {
        $this->output->writeln('<' . $type . '>' . $message . '</' . $type . '>');
        $this->logger->info($message);
    }

    /**
     * Checking multiplier activity settings pro store
     *
     * @param \Magento\Store\Api\Data\StoreInterface $store
     * @return bool
     */
    protected function isActiveForStore($store)
    {
        $isActive = $this->scopeConfig->isSetFlag(
            'catalog/store_prices/active',
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
            $store->getId()
        );
        if (!$isActive) {
            return false;
        }
        $this->multiplier = $this->scopeConfig->getValue(
            'catalog/store_prices/multiplier',
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
            $store->getId()
        );

        if ((empty($this->multiplier))
            || (!empty($this->multiplier) && !is_numeric($this->multiplier))) {
            return false;
        }
        return true;
    }

    /**
     * Calculating of product prices
     *
     * @param array $defaultPrices
     * @param \Magento\Store\Api\Data\StoreInterface $store
     * @return array
     * @throws \Exception
     */
    protected function calculateProductPrices($defaultPrices, $store)
    {
        $result = [];

        $multiplierSettings = $this->getProductPriceSetting($store);
        $multiplierFactorStore = $this->multiplier; //@TODO check
        $this->logAndOutput(sprintf(
                ' Base Multiplier:%s ',
                $multiplierFactorStore ? $multiplierFactorStore : '-'
            )
        );
        $collection = $this->productCollectionFactory->create();
        $collection->addAttributeToSelect(['series', 'attribute_set_id', 'shipping_forwarding'])
            ->addIdFilter(array_keys($defaultPrices))
            ->addAttributeToFilter([
                ['attribute' => 'manufacturer', 'notnull' => true],
                ['attribute' => 'manufacturer', 'null' => true],
            ]);

        foreach ($collection as $product) {

            $individualMultiplier = null;
            $productId = $product->getId();
            $sku = strtolower($product->getSku());
            $series = strtolower($product->getAttributeText('series'));
            $manufacturer = strtolower(
                $product->getResource()
                ->getAttribute('manufacturer')
                ->setStoreId(0)
                ->getFrontend()
                ->getValue($product)
            );

            $isForwarding = (bool)$product->getData('shipping_forwarding');
            if ($this->isDebugProduct($productId)) {
                $this->debugInfoForProduct[] = 'Found multiplier in sku=' . ((isset($multiplierSettings['sku']) &&
                        isset($multiplierSettings['sku'][$sku])) ? $multiplierSettings['sku'][$sku] : 'none');
            }

            $multiplierArray = [
                'sku' => $sku,
                'serie' => $series,
                'brand' => $manufacturer
            ];
            $multiplierDefinedBy = '';
            foreach ($multiplierArray as $multiplierKey => $multiplierValue) {

                if (isset($multiplierSettings[$multiplierKey])
                    && isset($multiplierSettings[$multiplierKey][$multiplierValue])) {
                    $individualMultiplier = $multiplierSettings[$multiplierKey][$multiplierValue];
                    if ($individualMultiplier['multiplier'] != 1) {
                        $this->addStatistics($multiplierKey, 1);
                    }
                    $multiplierDefinedBy = $multiplierKey;
                    break;
                }
            }

            if (empty($multiplierDefinedBy) && isset($multiplierSettings['set']) &&
                ($attrSetKey = $this->getAttributeSetKeyForProduct($product, $multiplierSettings['set']))) {
                $individualMultiplier = $multiplierSettings['set'][$attrSetKey];
                if ($individualMultiplier['multiplier'] != 1) {
                    $this->addStatistics('set', 1);
                }
                $multiplierDefinedBy = 'set';

            } elseif (empty($multiplierDefinedBy) && isset($multiplierSettings['lowprice']) &&
                ($lowPriceKey = $this->getLowPriceKeyForProduct(
                    $product,
                    $defaultPrices,
                    $multiplierSettings['lowprice'])
                )) {
                $individualMultiplier = $multiplierSettings['lowprice'][$lowPriceKey];
                if ($individualMultiplier['multiplier'] != 1) {
                    $this->lowPriceArray[$productId] = $individualMultiplier['multiplier'];
                    $this->addStatistics('lowprice', 1);
                }
                $multiplierDefinedBy = 'lowprice';
            }

            if (isset($defaultPrices[$productId]) && $defaultPrices[$productId]) {
                // First priority: individual multiplier
                if ($individualMultiplier) {
                    $newPrice = $defaultPrices[$productId];
                    if ($multiplier = $individualMultiplier['multiplier']) {
                        $newPrice = $newPrice * $multiplier;
                    }
                    $add = $this->surcharge;
                    if (!$add) {
                        if (($parcel = $individualMultiplier['addParcel']) && !$isForwarding) {
                            $add = $parcel;
                        } elseif (($forwarding = $individualMultiplier['addForwarding']) && $isForwarding) {
                            $add = $forwarding;
                        } elseif ($generalAdd = $individualMultiplier['add']) {
                            $add = $generalAdd;
                        }
                    } //@TODO refactor

                    if ($add) {
                        $newPrice = $newPrice + $add;
                    }
                    if ($sub = $individualMultiplier['sub']) {
                        $newPrice = $newPrice - $sub;
                    }

                    $result[$productId] = round($newPrice, 2);

                } else if ($multiplierFactorStore) {
                    $newPrice = $defaultPrices[$productId] * $multiplierFactorStore;
                    $result[$productId] = round($newPrice + $this->surcharge);

                } else {
                    $newPrice = $defaultPrices[$productId];
                    $result[$productId] = $this->surcharge
                        ? round($newPrice + $this->surcharge)
                        : $newPrice;
                }
            }

            if ($this->isDebugProduct($productId)) {
                $this->debugInfoForProduct[] = 'calculated=' . ($result[$productId]);
                if ($individualMultiplier) {
                    $this->debugInfoForProduct[] = 'multiplierDefinedBy=' . $multiplierDefinedBy . '/'
                        . $individualMultiplier;
                } else if ($multiplierFactorStore) {
                    $this->debugInfoForProduct[] = 'multiplierDefinedBy=store-default-factor/' . $multiplierFactorStore;
                }
            }
        }

        return $result;
    }

    /**
     * Returns the (configured) multiplier-setting key for a line, if the line matches to the given product.
     *
     * @param \Magento\Catalog\Model\Product $product
     * @param array $multiplierSettings
     * @return bool|int|string
     * @throws \Exception
     */
    protected function getAttributeSetKeyForProduct($product, $multiplierSettings)
    {
        foreach ($multiplierSettings as $attrSetKey => $multiplier) {
            $brand = null;
            $attrSetName = $attrSetKey;
            if (strpos($attrSetKey, '|') !== false) {
                $data = explode('|', $attrSetKey);
                $brand = trim($data[0]);
                $attrSetName = trim($data[1]);
            }
            $attrSetId = $this->getAttributeSetIdFromName($attrSetName);

            if (empty($attrSetId)) {
                throw new \Exception('Cannot identify attributeset ID by name: ' . $attrSetName);
            }

            $isHit = $product->getAttributeSetId() == $attrSetId;
            if ($isHit && !empty($brand)) {
                $manufacturer = strtolower($product->getAttributeText('manufacturer'));
                $isHit = !empty($manufacturer) && ($brand == $manufacturer);
            }

            if ($isHit) {
                return $attrSetKey;
            }
        }
        return false;
    }

    /**
     * @param string $attributeSetName
     * @return array
     */
    public function getAttributeSetIdFromName($attributeSetName)
    {
        static $attributeSetIds = [];

        if (!isset($attributeSetIds[$attributeSetName])) {
            $attributeCollection = $this->attributeSetCollectionFactory->create();
            $attributeSetIds[$attributeSetName] = $attributeCollection
                ->setEntityTypeFilter(4) //@TODO entity type product const search
                ->addFilter('attribute_set_name', $attributeSetName)
                ->getFirstItem()
                ->getId();
        }
        return $attributeSetIds[$attributeSetName];
    }

    /**
     * @param \Magento\Catalog\Model\Product $product
     * @param array $defaultPrices
     * @param $multiplierSettings
     * @return bool|string
     */
    protected function getLowPriceKeyForProduct($product, $defaultPrices, $multiplierSettings)
    {
        foreach ($multiplierSettings as $priceLevelKey => $multiplier) {
            $priceLevelKeys = explode('-', $priceLevelKey);
            $lowLimit = (float)$priceLevelKeys[0];
            $highLimit = (float)$priceLevelKeys[1];
            $price = $defaultPrices[$product->getId()];
            $isHit = ($price > 0) && ($price >= $lowLimit && $price <= $highLimit);
            if ($isHit) {
                return $priceLevelKey;
            }
        }
        return false;
    }

    /**
     * @param string $key
     * @param string $value
     */
    protected function addStatistics($key, $value)
    {
        if (array_key_exists($key, $this->processingInfoData)) {
            $current = $this->processingInfoData[$key];
            $this->processingInfoData[$key] = $current + (int)$value;
        } else {
            $this->processingInfoData[$key] = (int)$value;
        }
    }

    /**
     * Returns the backend configuration as an array
     *
     * @param \Magento\Store\Api\Data\StoreInterface $store
     * @return array
     * @throws \Exception on any invalid configuration
     */
    protected function getProductPriceSetting($store)
    {
        static $multiplierSettings = [];
        $storeId = $store->getId();

        if (!isset($multiplierSettings[$storeId])) {
            $settingsArr = [];

            $types = ['sku', 'serie', 'brand', 'set', 'lowprice'];
            foreach ($types as $type) {
                $multiplierSettingSku = explode("\n",
                    $this->scopeConfig->getValue(
                        'catalog/store_prices/multiplier_' . $type,
                        \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
                        $storeId
                    )
                );

                foreach ($multiplierSettingSku as $settingString) {
                    $setting = trim($settingString);
                    if ($type != 'set') {
                        $setting = strtolower($setting);
                    }

                    if (!empty($setting[0]) && $setting[0] == '/') {
                        continue;
                    }
                    if (!empty($setting)) {
                        $data = explode('#', trim($setting));
                        if (count($data) == 2) {
                            $element = $data[0];
                            if (($type == 'lowprice') && (strpos($element, '-') === false)) {
                                throw new \Exception(
                                    sprintf('%s configuration line not valid: %s', $type, $setting)
                                );
                            }

                            if (strpos($data[1], '&') > 0) {
                                $operators = explode('&', $data[1]);
                                foreach ($operators as $operator) {
                                    if (empty($operator)) {
                                        continue;
                                    }
                                    $result = $this->processOperator($operator);
                                }
                            } else {
                                $operator = $data[1];
                                if (empty($operator)) {
                                    throw new \Exception(
                                        'Invalid multiplier config: ' . $setting
                                    );
                                }
                                $result = $this->processOperator($operator);
                            }

                            if (!$result['multiplier'] && !$result['add'] && !$result['sub'] && !$result['addParcel']
                                && !$result['addForwarding']) {
                                throw new Exception(
                                    sprintf('%s invalid price configuration: %s', $type, $setting)
                                );
                            }

                            $settingsArr[$type][$element] = $result;
                        } else {
                            throw new \Exception(
                                sprintf('%s configuration line not valid: %s', $type, $setting)
                            );
                        }
                    }
                }
            }

            $message = <<<MESSAGE
            Registered setting for store "%s" is valid, sku %s, series %s, brand %s, attributeSet %s, lowPrice %s
MESSAGE;
            $this->logAndOutput(
                sprintf(
                    $message,
                    $store->getCode(),
                    !empty($settingsArr['sku']) ? count($settingsArr['sku']) : 0,
                    !empty($settingsArr['serie']) ? count($settingsArr['serie']) : 0,
                    !empty($settingsArr['brand']) ? count($settingsArr['brand']) : 0,
                    !empty($settingsArr['set']) ? count($settingsArr['set']) : 0,
                    !empty($settingsArr['lowprice']) ? count($settingsArr['lowprice']) : 0
                )
            );

            $multiplierSettings[$storeId] = $settingsArr;
        }

        return $multiplierSettings[$storeId];
    }

    /**
     * Transforming operator values to string constants
     *
     * @param string $operator
     * @return array
     */
    protected function processOperator($operator)
    {
        $result = [
            'add' => 0,
            'sub' => 0,
            'addParcel' => 0,
            'addForwarding' => 0,
            'multiplier' => 0
        ];
        if ($operator[0] == '+') {
            $result['add'] = substr(trim($operator), 1);
        } elseif ($operator[0] == '-') {
            $result['sub'] = substr(trim($operator), 1);
        } elseif ($operator[0] == 'p') {
            $result['addParcel'] = substr(trim($operator), 1);
        } elseif ($operator[0] == 'f') {
            $result['addForwarding'] = substr(trim($operator), 1);
        } else {
            $result['multiplier'] = trim($operator);
        }
        foreach ($result as $key => $value) {
            $result[$key] = floatval(str_replace(',', '.', $value));
        }
        return $result;
    }

    /**
     * Checking it's a debug product
     *
     * @param int $productId
     * @return bool
     */
    protected function isDebugProduct($productId)
    {
        return ($this->debugProductId && $this->debugProductId == $productId);
    }

}
