<?php

namespace Xtwo\Storeswitch\Model;

use Xtwo\Storeswitch\Api\CountryDropDownInterface;
use Xtwo\Storeswitch\ViewModel\StoreSwitchModel;
use Magento\Catalog\Model\CategoryFactory;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Catalog\Model\ProductRepository;
use Magento\Catalog\Model\Product;
use Xtwo\CustomApis\Helper\Data as CustomApisHelper;

use function PHPUnit\Framework\matches;

class CountryDropDown implements CountryDropDownInterface
{
    protected $storeSwitch;
    protected $directoryHelper;
    private $_countryFactory;
    protected $categoryFactory;
    protected $productCollectionFactory;
    protected $productRepository;
    protected $productModel;
    protected $countryInformation;
    protected $customApisHelper;
    private $categoryCollectionFactory;
    private $_storeManager;


    public function __construct(
        StoreSwitchModel $storeSwitch,
        \Magento\Directory\Model\CountryFactory $countryFactory,
        CategoryFactory $categoryFactory,
        ProductCollectionFactory $productCollectionFactory,
        ProductRepository $productRepository,
        Product $productModel,
        \Magento\Directory\Api\CountryInformationAcquirerInterface $countryInformation,
        CustomApisHelper $customApisHelper,
        \Magento\Catalog\Model\ResourceModel\Category\CollectionFactory $categoryCollectionFactory,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
    ) {
        $this->storeSwitch = $storeSwitch;
        $this->_countryFactory = $countryFactory;
        $this->categoryFactory = $categoryFactory;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->productRepository = $productRepository;
        $this->productModel = $productModel;
        $this->countryInformation = $countryInformation;
        $this->customApisHelper = $customApisHelper;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
        $this->_storeManager = $storeManager;
    }
    /**
     * @inheritdoc
     */
    public function getCountryDropDownData()
    {
        $websites = $this->storeSwitch->getWebsites();
        $finalResponse = [];

        foreach ($websites as $website) {
            $countryCode = $this->storeSwitch->getStoreCountryCode($website->getDefaultStore());
            $defaultStore = $website->getDefaultStore();
            $countryData = [
                'store_id' => $defaultStore->getId(),
                'country_full_name' => $this->getCountryName($countryCode),
                'country_id' => $countryCode,
                'country_flag' => $this->customApisHelper->getCountryFlagIconUrl($countryCode),
            ];

            $finalResponse[] = $countryData;
        }

        return $finalResponse;
    }

    private function getCountryName($countryCode)
    {
        $locale = 'en_US';
        $countryCollection = $this->_countryFactory->create()->loadByCode($countryCode);
        return $countryCollection->getName($locale);
    }

    protected function getStoreUrl($storeId)
    {
        $storeManager = \Magento\Framework\App\ObjectManager::getInstance()->get(\Magento\Store\Model\StoreManagerInterface::class);
        $store = $storeManager->getStore($storeId);
        $storeUrl = $store->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_WEB);

        return $storeUrl;
    }

    public function getConvertedUrlKey($urlKey, $clickedStoreId)
    {
        if (strpos($urlKey, 'category/') !== false || strpos($urlKey, 'product/') !== false || empty($urlKey)) {
            $modifiedUrl = str_replace("category/", "", $urlKey);
            $parts = explode("/", $modifiedUrl);
            $part1 = $parts[0] ?? '';
            $part2 = $parts[1] ?? '';

            $clickedStoreUrl = $this->getStoreUrl($clickedStoreId);

            $part1CatId = $this->getCategoryIdByName($part1, $clickedStoreId);
            $part2CatId = $this->getCategoryIdByName($part2, $clickedStoreId);

            $catFinalurl = $this->buildCategoryFinalUrl($part1CatId, $part2CatId, $clickedStoreId);

            if (!empty($catFinalurl)) {
                return $clickedStoreUrl.'category/'.$catFinalurl;
            } else {
                $modifiedProductUrl = str_replace("product/", "", $urlKey);
                $finalProductUrl = $this->getFinalProductUrl($modifiedProductUrl, $clickedStoreId);
                if (!empty($finalProductUrl)) {
                    return $clickedStoreUrl . 'product/' . $finalProductUrl;
                } else {
                    return $clickedStoreUrl . $urlKey;
                }
            }
        }elseif (empty($urlKey)) {
            $clickedStoreUrl = $this->getStoreUrl($clickedStoreId);
            return $clickedStoreUrl;
        } else {
            $urlsObj = [
                // service
                [
                    '1' => "service/versand",
                    '2' => "service/shipping",
                    '3' => "service/les-frais-de-livraison",
					'5' => "service/versand",
                ],
                [
                    '1' => "service/lieferzeit",
                    '2' => "service/delivery-time",
                    '3' => "service/delais-de-livraison",
                ],
                [
                    '1' => "service/lieferzeit",
                    '2' => "service/payment",
                    '3' => "service/paiement",
                ],
                [
                    '1' => "service/garantie",
                    '2' => "service/warranty",
                    '3' => "service/garanties",
                ],
                //about
                [
                    '1' => "unternehmen/ueber-xtwo",
                    '2' => "aboutus/about-xtwo",
                    '3' => "entreprise/qui-sommes-nous",
                ],
                [
                    '1' => "unternehmen/geschaeftskunden",
                    '2' => "aboutus/business-customer",
                    '3' => "entreprise/client-professionnel",
                ],
                [
                    '1' => "unternehmen/ausstellung",
                    '2' => "aboutus/showroom",
                    '3' => "entreprise/showroom",
                ],
                [
                    '1' => "unternehmen/kontakt",
                    '2' => "aboutus/contact",
                    '3' => "entreprise/contactez-nous",
                ],
                //legal
                [
                    '1' => "rechtliches/agb",
                    '2' => "legal/terms-and-conditions",
                    '3' => "mentions-juridiques/nos-conditions-generales-de-affaires",
                ],
                [
                    '1' => "rechtliches/widerrufsbelehrung",
                    '2' => "legal/return-policy",
                    '3' => "mentions-juridiques/droit-de-retractation",
                ],
                [
                    '1' => "rechtliches/datenschutz",
                    '2' => "legal/privacy-policy",
                    '3' => "mentions-juridiques/confidentialite-des-donnees",
                ],
                [
                    '1' => "rechtliches/impressum",
                    '2' => "legal/imprint",
                    '3' => "mentions-juridiques/confidentialite-des-donnees",
                ],
                //Resources
                [
                    '1' => "quellen/faq",
                    '2' => "resources/faq",
                    '3' => "information/faq",
                ],
            ];
            $matchedArray = [];

            foreach ($urlsObj as $item) {
                if (in_array($urlKey, $item)) {
                    $matchedArray = $item;
                    break;
                }
            }

            if (count($matchedArray) <= 4 && !empty($matchedArray)) {
                if (!empty($matchedArray[$clickedStoreId])) {
                    return $this->getStoreUrl($clickedStoreId).$matchedArray[$clickedStoreId];
                } else {
                    return $this->getStoreUrl($clickedStoreId).$matchedArray[2];
                }
            }else{
                return $this->getStoreUrl($clickedStoreId).$urlKey;
            }

        }
    }

    private function getCategoryIdByName($categoryName, $storeId)
    {
        return $this->categoryCollectionFactory
            ->create()
            ->addAttributeToSelect('*')
            ->addAttributeToFilter('url_key', $categoryName)
            ->setPageSize(1)
            ->setCurPage(1)
            ->setStoreId($storeId)
            ->getFirstItem()
            ->getId();
    }

    private function buildCategoryFinalUrl($part1CatId, $part2CatId, $clickedStoreId)
    {
        $catFinalUrl = '';

        if ($part1CatId) {
            $part1Category = $this->categoryFactory->create()->setStoreId($clickedStoreId)->load($part1CatId);
            $part1UrlKey = $part1Category->getUrlKey();
            $catFinalUrl .= $part1UrlKey;

            if ($part2CatId) {
                $part2Category = $this->categoryFactory->create()->setStoreId($clickedStoreId)->load($part2CatId);
                $part2UrlKey = $part2Category->getUrlKey();
                $catFinalUrl .= '/' . $part2UrlKey;
            }
        }

        return $catFinalUrl;
    }

    protected function getCountryFlagIconUrl($countryCode)
    {
        $flagIconPath = 'wysiwyg/smartwave/porto/flags/' . $countryCode . '.png';
        if (file_exists($flagIconPath)) {
            return $flagIconPath;
        } else {
            return 'default/flag/icon.png';
        }
    }

    private function getFinalProductUrl($modifiedProductUrl, $clickedStoreId)
    {
        $productCollection = $this->productCollectionFactory->create()
            ->addAttributeToSelect('*')
            ->addAttributeToFilter('url_key', ['eq' => $modifiedProductUrl]);

        foreach ($productCollection as $data) {
            $product = $this->productRepository->getById($data->getId(), false, $clickedStoreId);
            $finalProductUrl = $product->getUrlKey();
            if (!empty($finalProductUrl)) {
                return $finalProductUrl;
            }
        }

        return '';
    }

}