<?php

namespace Xtwo\Setup\Setup;

use Exception;
use Magento\Config\Model\ResourceModel\Config;
use Magento\Customer\Api\GroupRepositoryInterface;
use Magento\Customer\Api\Data\GroupInterfaceFactory as CustomerGroupFactory;
use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\App\State;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Setup\ModuleContextInterface;
use Magento\Framework\Setup\ModuleDataSetupInterface;
use Magento\Framework\Setup\UpgradeDataInterface;
use Magento\Tax\Api\Data\TaxClassInterfaceFactory;
use Magento\Tax\Api\TaxClassRepositoryInterface;
use Magento\Tax\Api\TaxRateRepositoryInterface;
use Magento\Tax\Api\TaxRuleRepositoryInterface;
use Magento\Tax\Helper\Data as TaxHelperData;
use Magento\Tax\Model\Calculation\RateFactory;
use Magento\Tax\Model\Calculation\RuleFactory;
use Magento\Directory\Model\RegionFactory;
use Psr\Log\LoggerInterface as Logger;

/**
 * Class UpgradeData
 */
class UpgradeData implements UpgradeDataInterface
{
    /**
     * @var \string[][]
     */
    static $rateGermany = [
        'Germany' => ['DE', '19.0000'],
    ];

    /**
     * code => country id, rate, zip code (or range), region, is range
     * @var array
     */
    static $ratesVATRetail = [
        'Austria' => ['AT', '20.0000'],
        'Belgium' => ['BE', '21.0000'],
        'Bulgaria' => ['BG', '20.0000'],
        'Croatia' => ['HR', '19.0000'],
        'Cyprus' => ['CY', '19.0000'],
        'Czech Republic' => ['CZ', '21.0000'],
        'Denmark' => ['DK', '25.0000'],
        'Estonia' => ['EE', '20.0000'],
        'Finland' => ['FI', '24.0000'],
        'France' => ['FR', '20.0000'],
        'Greece' => ['GR', '24.0000'],
        'Hungary' => ['HU', '27.0000'],
        'Irland' => ['IE', '23.0000'],
        'Italy' => ['IT', '22.0000'],
        'Latvia' => ['LV', '21.0000'],
        'Lithuania' => ['LT', '21.0000'],
        'Luxemburg' => ['LU', '17.0000'],
        'Malta' => ['MT', '18.0000'],
        'Monaco' => ['MC', '20.0000'],
        'Netherlands' => ['NL', '21.0000'],
        'Poland' => ['PL', '23.0000'],
        'Portugal' => ['PT', '23.0000'],
        'Romania' => ['RO', '19.0000'],
        'Slovakia' => ['SK', '20.0000'],
        'Slovenia' => ['SI', '22.0000'],
        'Spain' => ['ES', '21.0000'],
        'Spain Canares (LP)' => ['ES', '0.0000', '35000-35999', 'Las Palmas', true],
        'Spain Canares (SC)' => ['ES', '0.0000', '38000-38999', 'Santa Cruz de Tenerife', true],
        'Sweden' => ['SE', '25.0000']
    ];

    /**
     * @var array
     */
    static $ratesVATNotInRules = [
        'Germany Büsingen' => ['DE', '0.0000', '78266'],
        'Germany Helgoland' => ['DE', '0.0000', '27498'],
        "Italy Campione d'Italia" => ['IT', '0.0000', '22060'],
        'Italy Livigno' => ['IT', '0.0000', '23030'],
        'Spain Ceuta' => ['ES', '0.0000', '51001-51080', 'Ceuta', true],
        'Spain Melilla' => ['ES', '0.0000', '52001', 'Melilla'],
    ];

    /**
     * @var string[]
     */
    static $defaultCustomerGroups = [
        'B2B-VK-VAT' => 'Retail Customer',
        'B2B-VK-NOVAT' => 'Wholesale',
        'B2B-RE-VAT' => 'Retail Customer',
        'B2B-RE-NOVAT' => 'Wholesale',
    ];

    /**
     * @var string
     */
    protected $wholesaleTaxClass;

    /**
     * @var Config
     */
    private $configResourceModel;
    /**
     * @var State
     */
    private $appState;
    /**
     * @var Logger
     */
    private $logger;
    /**
     * @var GroupRepositoryInterface
     */
    private $customerGroupRepository;
    /**
     * @var CustomerGroupFactory
     */
    private $customerGroupFactory;
    /**
     * @var \Magento\Tax\Model\Calculation\RateFactory
     */
    private RateFactory $rateFactory;
    /**
     * @var \Magento\Tax\Model\Calculation\RuleFactory
     */
    private RuleFactory $ruleFactory;
    /**
     * @var \Magento\Tax\Api\TaxRateRepositoryInterface
     */
    private TaxRateRepositoryInterface $rateRepository;
    /**
     * @var \Magento\Tax\Api\TaxRuleRepositoryInterface
     */
    private TaxRuleRepositoryInterface $ruleRepository;
    /**
     * @var \Magento\Tax\Helper\Data
     */
    private TaxHelperData $taxHelper;
    /**
     * @var \Magento\Tax\Api\Data\TaxClassInterfaceFactory
     */
    private TaxClassInterfaceFactory $taxClassFactory;
    /**
     * @var \Magento\Tax\Api\TaxClassRepositoryInterface
     */
    private TaxClassRepositoryInterface $taxClassRepository;
    /**
     * @var \Magento\Framework\Api\FilterBuilder
     */
    private FilterBuilder $filterBuilder;
    /**
     * @var \Magento\Framework\Api\SearchCriteriaBuilder
     */
    private SearchCriteriaBuilder $searchCriteriaBuilder;
    /**
     * @var \Magento\Directory\Model\RegionFactory
     */
    private RegionFactory $regionFactory;

    /**
     * InstallData constructor.
     *
     * @param Config $configResourceModel
     * @param \Magento\Framework\App\State $appState
     * @param Logger $logger
     * @param CustomerGroupFactory $customerGroupFactory
     * @param GroupRepositoryInterface $customerGroupRepository
     * @param \Magento\Tax\Model\Calculation\RateFactory $rateFactory
     * @param \Magento\Tax\Model\Calculation\RuleFactory $ruleFactory
     * @param \Magento\Tax\Api\TaxRateRepositoryInterface $rateRepository
     * @param \Magento\Tax\Api\TaxRuleRepositoryInterface $ruleRepository
     * @param \Magento\Tax\Helper\Data $taxHelper
     * @param \Magento\Tax\Api\Data\TaxClassInterfaceFactory $taxClassFactory
     * @param \Magento\Tax\Api\TaxClassRepositoryInterface $taxClassRepository
     * @param \Magento\Framework\Api\SearchCriteriaBuilder $searchCriteriaBuilder
     * @param \Magento\Framework\Api\FilterBuilder $filterBuilder
     * @param \Magento\Directory\Model\RegionFactory $regionFactory
     */
    public function __construct(
        Config $configResourceModel,
        State $appState,
        Logger $logger,
        CustomerGroupFactory $customerGroupFactory,
        GroupRepositoryInterface $customerGroupRepository,
        RateFactory $rateFactory,
        RuleFactory $ruleFactory,
        TaxRateRepositoryInterface $rateRepository,
        TaxRuleRepositoryInterface $ruleRepository,
        TaxHelperData $taxHelper,
        TaxClassInterfaceFactory $taxClassFactory,
        TaxClassRepositoryInterface $taxClassRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        FilterBuilder $filterBuilder,
        RegionFactory $regionFactory
    ) {
        $this->configResourceModel = $configResourceModel;
        $this->appState = $appState;
        $this->logger = $logger;
        $this->customerGroupRepository = $customerGroupRepository;
        $this->customerGroupFactory = $customerGroupFactory;
        $this->rateFactory = $rateFactory;
        $this->ruleFactory = $ruleFactory;
        $this->rateRepository = $rateRepository;
        $this->ruleRepository = $ruleRepository;
        $this->taxHelper = $taxHelper;
        $this->taxClassFactory = $taxClassFactory;
        $this->taxClassRepository = $taxClassRepository;
        $this->filterBuilder = $filterBuilder;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->regionFactory = $regionFactory;
    }

    /**
     * @param \Magento\Framework\Setup\ModuleDataSetupInterface $setup
     * @param \Magento\Framework\Setup\ModuleContextInterface $context
     *
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function upgrade(
        ModuleDataSetupInterface $setup,
        ModuleContextInterface $context
    ) {
        try {
            $this->appState->getAreaCode();
        } catch (LocalizedException $ex) {
            $this->appState->setAreaCode('adminhtml');
        }

        $setup->startSetup();
        if (version_compare($context->getVersion(), '1.0.1', '<')) {
            try {
                $this->wholesaleTaxClass = $this->setupTaxClass();
                $this->updateWholesaleCustomerGroup();
                $this->createCustomerGroups();
                $this->setupTaxRules();
                $this->setupTaxRates(self::$ratesVATNotInRules);
            } catch (Exception $exception) {
                $this->logger->error($exception->getMessage());
            }
        }

        $setup->endSetup();
    }

    /**
     * @param string $taxClassName
     * @param string $taxClassType
     *
     * @return string
     * @throws \Magento\Framework\Exception\InputException
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function setupTaxClass($taxClassName = 'Wholesale', $taxClassType = 'CUSTOMER')
    {
        $taxClass = $this->taxClassFactory->create();

        $taxClass->setClassName($taxClassName);
        $taxClass->setClasstype($taxClassType);

        return $this->taxClassRepository->save($taxClass);
    }

    /**
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function updateWholesaleCustomerGroup()
    {
        $this->searchCriteriaBuilder->addFilters(
            [
                $this->filterBuilder
                    ->setField('customer_group_code')
                    ->setConditionType('eq')
                    ->setValue('Wholesale')
                    ->create(),
            ]
        );

        $this->searchCriteriaBuilder->setPageSize(100);
        $searchCriteria = $this->searchCriteriaBuilder->create();

        foreach ($this->customerGroupRepository->getList($searchCriteria)->getItems() as $wholesaleGroup) {
            $wholesaleGroup->setTaxClassId($this->wholesaleTaxClass);
            $this->customerGroupRepository->save($wholesaleGroup);
        }
    }

    /**
     *
     */
    protected function createCustomerGroups()
    {
        foreach (self::$defaultCustomerGroups as $groupCode => $taxGroup) {
            $group = $this->customerGroupFactory->create();
            $group
                ->setCode($groupCode)
                ->setTaxClassId(
                    $taxGroup == 'Retail Customer'
                        ? $this->taxHelper->getDefaultCustomerTaxClass()
                        : $this->wholesaleTaxClass
                );

            $this->customerGroupRepository->save($group);
        }
    }

    /**
     * @param array $rateIds
     *
     * @return \Magento\Tax\Api\Data\TaxRateInterface
     * @throws \Magento\Framework\Exception\InputException
     */
    protected function setupTaxRules()
    {
        $rules = [
            'Retail Taxrule' => [
                [$this->taxHelper->getDefaultCustomerTaxClass()],
                self::$ratesVATRetail
                ],
            'Wholesale Taxrule' => [
                [$this->wholesaleTaxClass],
                self::$ratesVATRetail
                ],
            'Retail & Wholesale Germany Taxrule' => [
                [$this->taxHelper->getDefaultCustomerTaxClass(), $this->wholesaleTaxClass],
                self::$rateGermany
            ]
        ];
        foreach ($rules as $code => $ruleInfo) {
            $rule = $this->ruleFactory->create();
            $rule->setCode($code);
            $rule->setPriority(0);
            $rule->setCustomerTaxClassIds($ruleInfo[0]);
            $rule->setProductTaxClassIds([$this->taxHelper->getDefaultProductTaxClass()]);
            $rule->setTaxRateIds($this->setupTaxRates($ruleInfo[1], $code == 'Wholesale Taxrule'));

            $this->ruleRepository->save($rule);
        }
    }

    /**
     * @param array $rates
     * @param bool $isZeroRate
     *
     * @return array
     * @throws \Magento\Framework\Exception\InputException
     */
    protected function setupTaxRates(array $rates, $isZeroRate = false)
    {
        $rateIds = [];
        foreach ($rates as $code => $rateInfo) {
            if ($isZeroRate && $rateInfo[1] == '0.0000') {
                continue;
            }
            $rate = $this->rateFactory->create();

            $rate->setCode($code . (!($isZeroRate) ? '' : ' 0%' ));
            $rate->setTaxCountryId($rateInfo[0]);

            $rate->setZipIsRange(!empty($rateInfo[4]) && $rateInfo[4] ? "1" : "0");
            if (!empty($rateInfo[4]) && $rateInfo[4] && !empty($rateInfo[2])) {
                $zipCodes = explode("-", $rateInfo[2]);
                if (is_array($zipCodes) && !empty($zipCodes[0]) && !empty($zipCodes[1])) {
                    $rate->setZipFrom($zipCodes[0]);
                    $rate->setZipTo($zipCodes[1]);
                    $rate->setTaxPostcode($rateInfo[2]);
                }
            }

            if (!empty($rateInfo[3])) {
                $region = $this->regionFactory->create()->loadByName($rateInfo[3], $rateInfo[0]);
                if ($region->getId() !== null) {
                    $rate->setTaxRegionId($region->getId());
                }
            } else {
                $rate->setTaxRegionId(0);
            }

            $rate->setTaxPostcode("*");
            $rate->setRate(!($isZeroRate) ? $rateInfo[1] : '0.0000');

            $rate = $this->rateRepository->save($rate);
            $rateIds[] = $rate->getId();
        }
        return $rateIds;
    }
}
