<?php
/**
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_SalesMailing
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 */
namespace Xtwo\SalesMailing\Model\Service;

class UrlResolver
{
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var \Magento\Backend\Helper\Data
     */
    protected $backendHelper;

    /**
     * @var \Magento\Framework\UrlInterface
     */
    protected $urlInterface;

    protected $salesMailingHelper;
    protected $hasher;
    public function __construct(
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Backend\Helper\Data $backendHelper,
        \Magento\Framework\UrlInterface $urlInterface,
        \Xtwo\SalesMailing\Model\Service\FileResolver $fileResolver,
        \Xtwo\SalesMailing\Model\Service\Hasher $hasher,
        \Xtwo\SalesMailing\Helper\Data $salesMailingHelper,
        \Magento\Framework\App\State $state,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
    ) {
        $this->storeManager = $storeManager;
        $this->backendHelper = $backendHelper;
        $this->urlInterface = $urlInterface;
        $this->fileResolver = $fileResolver;
        $this->hasher = $hasher;
        $this->salesMailingHelper=$salesMailingHelper;
        $this->_state = $state;
        $this->scopeConfig = $scopeConfig;
    }

    /**
     * Get invoice URL
     *
     * @param \Magento\Sales\Model\Order $order
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getInvoiceUrl(\Magento\Sales\Model\Order $order)
    {
        $documentPaths = $this->fileResolver->getInvoiceDocumentPath($order);
        return $this->resolve($documentPaths, $order, \Xtwo\SalesMailing\Model\Configuration::TRIGGER_TYPE_EXT_INVOICE);
    }

    /**
     * Get shipment URL
     *
     * @param \Magento\Sales\Model\Order $order
     *
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getShipmentUrl(\Magento\Sales\Model\Order $order)
    {
        $documentPaths = $this->fileResolver->getShippingDocumentPath($order);
        return $this->resolve($documentPaths, $order, \Xtwo\SalesMailing\Model\Configuration::TRIGGER_TYPE_EXT_SHIPMENT);
    }

    /**
     * Get order confirmation URL
     *
     * @param \Magento\Sales\Model\Order $order
     *
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getOrderConfirmationUrl(\Magento\Sales\Model\Order $order)
    {
        $documentPaths = $this->fileResolver->getOrderConfirmationDocumentPath($order);
        return $this->resolve($documentPaths, $order, \Xtwo\SalesMailing\Model\Configuration::TRIGGER_TYPE_EXT_ORDERCONFIRMATION);
    }

    /**
     * Get credit memo URL
     *
     * @param \Magento\Sales\Model\Order $order
     *
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getCreditMemoUrl(\Magento\Sales\Model\Order $order)
    {
        $documentPaths = $this->fileResolver->getCreditMemoDocumentPath($order);
        return $this->resolve($documentPaths, $order, \Xtwo\SalesMailing\Model\Configuration::TRIGGER_TYPE_EXT_CREDITMEMO);
    }

    /**
     * Get document path
     *
     * @param $documentPaths
     * @param \Magento\Sales\Model\Order $order
     * @param $type
     * @return array
     */
    private function resolve($documentPaths, $order, $type=null)
    {
        $documentUrls = array();
        $documentCdnEnabled = $this->scopeConfig->getValue( 'aws_s3_connection/configuration/document_cdn', 
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE);
        try {
            foreach ($documentPaths as $k=>$documentPath) {
                if(!$documentCdnEnabled){
                    if (!file_exists($documentPath)) {
                        throw new \Magento\Framework\Exception\LocalizedException(
                            __('File not found.')
                        );
                    }
                }

                $fileName = pathinfo($documentPath)['filename'];
               
                $params = array(
                    'file' => $fileName
                );
                
                $url = ($this->_state->getAreaCode()=='adminhtml') ?
                    $this->backendHelper->getUrl('salesmailing/download/index', $params) :
                    
                    $this->urlInterface->getUrl('salesmailing/download/index', $params);

                // We add the hash as usual parameter as otherwise it causes problems containing a slash with
                $hash = $this->hasher->hash($fileName);
                
                $url .= '?hash='.$hash;
                if($documentCdnEnabled){
                    $url =$documentPath;
                    $key = $k ? $k : $fileName;
                }else{
                    $key = filectime($documentPath) ? filectime($documentPath) : $fileName;
                    if (is_numeric($key)) {
                        $delaySec = $this->getDelayOfDocumentTypeInSeconds($order, $type);
                        $key = $key - $delaySec;
                    }
                }
                if (is_numeric($key)) {
                    $delaySec = $this->getDelayOfDocumentTypeInSeconds($order, $type);
                    $key = $key - $delaySec;
                }
                $documentUrls[$key] = [
                    'file_name' => $fileName.'.pdf',
                    'created_at' => $key,
                    'url' => $url
                ];
            }

        } catch (\Exception $e) {
            $documentUrls = [];
        }

        krsort($documentUrls);
        return $documentUrls;
    }

    /**
     * @param \Magento\Sales\Model\Order $order
     * @param $type
     * @return int
     * @throws Mage_Core_Model_Store_Exception
     */
    private function getDelayOfDocumentTypeInSeconds($order, $type) {
        if (!$type) return 0;
        $delay = 0;
        
        $configurations = $this->salesMailingHelper->getActiveConfigurationsOfSpecificType($type);
        /** @var \Xtwo\SalesMailing\Model\Configuration $configuration */
        foreach ($configurations as $configuration) {
            if ($configuration['is_active'][$order->getStoreId() ]) {
            //if ($configuration->isActive($order->getStoreId())) {
                $delay = $configuration->getDelay();
                if ($delay) $delay = $delay * 3600;
                break;
            }
        }
        return $delay;
    }
}