<?php
namespace Xtwo\ProductListingApi\Plugin;
 
use Magento\Framework\Api\FilterBuilder;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\Filter;
use Magento\Framework\Api\Search\FilterGroup;
use Magento\Catalog\Api\Data\ProductSearchResultsInterface;
use Magento\Catalog\Model\ProductRepository;
use Magento\Eav\Model\Config as EavConfig;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;
use Xtwo\AttributeOption\Helper\Image as ImageHelper;
use Xtwo\CustomApis\Helper\Data as CustomApisHelper;
use Magento\Wishlist\Model\WishlistFactory;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Catalog\Api\Data\ProductExtensionFactory;
use Magento\Framework\App\RequestInterface;
use Magento\Catalog\Model\CategoryFactory;
use Magento\Store\Model\StoreManagerInterface;

class ProductList
{
    private $filterBuilder;
    private $searchCriteriaBuilder;
    private $categoryCollectionFactory;
    private $eavConfig;
    private $imageHelper;
    private $customApisHelper;
    private $wishlistFactory;
    private $customerSession;
    private $extensionFactory;
    private $collectionFactory;
    private $request;
    private $categoryFactory;
    protected $storeManager;
 
    public function __construct(
        \Magento\Catalog\Model\ResourceModel\Product\CollectionFactory $collectionFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        FilterBuilder $filterBuilder,
        EavConfig $eavConfig,
        CategoryCollectionFactory $categoryCollectionFactory,
        ImageHelper $imageHelper,
        CustomApisHelper $customApisHelper,
        WishlistFactory $wishlistFactory,
        CustomerSession $customerSession,
        ProductExtensionFactory $extensionFactory,
        RequestInterface $request,
        CategoryFactory $categoryFactory,
		\Magento\Tax\Model\Calculation $taxCalculation,
        StoreManagerInterface $storeManager

    ) {
        $this->collectionFactory = $collectionFactory;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->filterBuilder = $filterBuilder;
        $this->eavConfig = $eavConfig;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
        $this->imageHelper = $imageHelper;
        $this->customApisHelper = $customApisHelper;
        $this->wishlistFactory = $wishlistFactory;
        $this->customerSession = $customerSession;
        $this->extensionFactory = $extensionFactory;
        $this->request = $request;
        $this->categoryFactory = $categoryFactory;
		$this->taxCalculation = $taxCalculation;
        $this->storeManager = $storeManager;
        
    }
 
    public function beforeGetList(
        \Magento\Catalog\Api\ProductRepositoryInterface $subject,
        \Magento\Framework\Api\SearchCriteriaInterface $searchCriteria
    ) {
        $storeId = $this->storeManager->getStore()->getId();
        $urlKeyFilter = $this->getFilterValue($searchCriteria, 'url_key');
        $subCategoryUrl1 = $this->request->getParam('subCategoryUrl1');
        $subCategoryUrl2 = $this->request->getParam('subCategoryUrl2');
    
        $categoryIds = [];
    
        if ($urlKeyFilter) {
            $mainCategoryUrl = $urlKeyFilter['value'];
            $categoryIds = $this->customApisHelper->getCategoryLevels($mainCategoryUrl,$subCategoryUrl1,$subCategoryUrl2,$storeId);
            if (!empty($categoryIds)) {
                $filter = $this->filterBuilder
                    ->setField('category_id')
                    ->setValue($categoryIds)
                    ->setConditionType('in')
                    ->create();
                $this->addFilterToSearchCriteria($searchCriteria, $filter);
            }
            $this->removeFilterFromSearchCriteria($searchCriteria, 'url_key');
        }
    
    
        // Handle Price Filters
        $priceFrom = $priceTo = null;
        $filterGroups = $searchCriteria->getFilterGroups();
    
        foreach ($filterGroups as $filterGroup) {
            foreach ($filterGroup->getFilters() as $filter) {
                if ($filter->getField() == 'price') {
                    if ($filter->getConditionType() == 'gteq') {
                        $priceFrom = $filter->getValue();
                    } elseif ($filter->getConditionType() == 'lteq') {
                        $priceTo = $filter->getValue();
                    }
                }
            }
        }
    
        if ($priceFrom !== null) {
            $filterFrom = $this->filterBuilder
                ->setField('price')
                ->setValue($priceFrom)
                ->setConditionType('gteq')
                ->create();
            $this->addFilterToSearchCriteria($searchCriteria, $filterFrom);
        }
    
        if ($priceTo !== null) {
            $filterTo = $this->filterBuilder
                ->setField('price')
                ->setValue($priceTo)
                ->setConditionType('lteq')
                ->create();
            $this->addFilterToSearchCriteria($searchCriteria, $filterTo);
        }
    
        return [$searchCriteria];
    }      
 
    public function afterGetList(
        ProductRepository $subject,
        ProductSearchResultsInterface $result,
        SearchCriteriaInterface $searchCriteria
    ) {
        $customerId = $this->getCustomerIdFromRequest();
        $filteredItems = [];
    
        $priceFrom = $priceTo = null;
        foreach ($searchCriteria->getFilterGroups() as $filterGroup) {
            foreach ($filterGroup->getFilters() as $filter) {
                if ($filter->getField() == 'price') {
                    if ($filter->getConditionType() == 'gteq') {
                        $priceFrom = $filter->getValue();
                    } elseif ($filter->getConditionType() == 'lteq') {
                        $priceTo = $filter->getValue();
                    }
                }
            }
        }
		
        foreach ($result->getItems() as $product) {
            if ($product->getTypeId() == 'configurable') {
                continue;
            }
    
            $productPrice = $product->getPrice();
    
            if (($priceFrom && $productPrice < $priceFrom) || ($priceTo && $productPrice > $priceTo)) {
                continue;
            }
    
            $loadedProduct = $product->load($product->getId());
            $isInWishlist = $this->isProductInWishlist($customerId, $loadedProduct);
            $averageRating = $this->customApisHelper->getAverageRating($loadedProduct->getId());
            $showAddToCartButton = $this->customApisHelper->getButtonStatus($loadedProduct);
			
			// Calculate price with VAT
			$taxClassId = $product->getTaxClassId();
			$store      = $this->storeManager->getStore();
			$request = $this->taxCalculation->getRateRequest(null, null, null, $store);
			$taxRate = $this->taxCalculation->getRate($request->setProductClassId($taxClassId));
			
			$price = $product->getPrice();
			$price = round($price, 2);
			$basePrice = $price / 1.19;
			$priceWithTax = $basePrice + ($basePrice * ($taxRate / 100));
			
			$msrpPrice = $product->getMsrp();
			$msrpBasePrice = $msrpPrice / 1.19;
			$msrpPriceWithTax = $msrpBasePrice + ($msrpBasePrice * ($taxRate / 100));
    
            $this->addCustomExtraAttributes($product, $isInWishlist, $averageRating, $showAddToCartButton, $msrpPriceWithTax);
			
			
		
			$productPrice = $product->setPrice($priceWithTax);
			
            $filteredItems[] = $product;
        }
    
        $result->setItems($filteredItems);
    
        return $result;
    }   

    private function findCategoryIdByUrlKey($categories, $urlKey)
    {
        foreach ($categories as $category) {
            if ($category['url_key'] === $urlKey) {
                return $category['id'];
            }
        }
        return null;
    }

    private function getSubcategoriesByParentId($parentCategoryId)
    {
        $categoryCollection = $this->categoryCollectionFactory->create();
        $categoryCollection->addAttributeToSelect('url_key') 
            ->addAttributeToSelect('name') 
            ->addAttributeToFilter('parent_id', $parentCategoryId)
            ->addIsActiveFilter()
            ->setOrder('position', 'ASC'); 

        $subcategories = [];
        foreach ($categoryCollection as $category) {
            $subcategories[] = [
                'id' => $category->getId(),
                'url_key' => $category->getUrlKey(),
            ];
        }

        return $subcategories;
    }
       
    private function addFilterToSearchCriteria(SearchCriteriaInterface $searchCriteria, Filter $filter)
    {
        $filterGroups = $searchCriteria->getFilterGroups();
        $newFilterGroup = new FilterGroup();
        $newFilterGroup->setFilters([$filter]);
 
        $filterGroups[] = $newFilterGroup;
        $searchCriteria->setFilterGroups($filterGroups);
    }
 
    private function removeFilterFromSearchCriteria(SearchCriteriaInterface $searchCriteria, $field)
    {
        $filterGroups = $searchCriteria->getFilterGroups();
 
        foreach ($filterGroups as $filterGroup) {
            $filters = $filterGroup->getFilters();
            foreach ($filters as $key => $filter) {
                if ($filter->getField() == $field) {
                    unset($filters[$key]);
                }
            }
            $filterGroup->setFilters($filters);
        }
 
        $searchCriteria->setFilterGroups($filterGroups);
    }
 
    private function getFilterValue(SearchCriteriaInterface $searchCriteria, $field)
    {
        foreach ($searchCriteria->getFilterGroups() as $filterGroup) {
            foreach ($filterGroup->getFilters() as $filter) {
                if ($filter->getField() == $field) {
                    return [
                        'value' => $filter->getValue(),
                        'conditionType' => $filter->getConditionType()
                    ];
                }
            }
        }
        return null;
    }
 
 
    private function getCategoryIdFromUrlKey($urlKey)
    {
        $categoryCollection = $this->categoryCollectionFactory->create();
        $categoryCollection->addAttributeToFilter('url_key', $urlKey);
        $category = $categoryCollection->getFirstItem();
 
        if ($category && $category->getId()) {
            return $category->getId();
        }
 
        return null;
    }
 
    private function addCustomExtraAttributes(
        \Magento\Catalog\Api\Data\ProductInterface $product,
        $isInWishlist,
        $averageRating,
        $showAddToCartButton,
		$msrpPriceWithTax
    ) {
        $productId = $product->getId();
        $loadedProduct = $product->load($productId);
        preg_match('/href="(.*?)"/', $loadedProduct->getAttributeText('manufacturer'), $matches);
 
        $customAttributes = [
            [
                'attribute_code' => 'delivery_time',
                'value' => $loadedProduct->getAttributeText('delivery_time') ?: ''
            ],
            [
                'attribute_code' => 'msrp',
                'value' => (string)$msrpPriceWithTax ?: ''
            ],
            [
                'attribute_code' => 'manufacturer',
                'value' => [
                    strip_tags($loadedProduct->getAttributeText('manufacturer')),
                    isset($matches[1]) ? str_replace('.html', '', $matches[1]) : '',
                    $this->getManufacturerImagePath($loadedProduct)
                ]
            ],
            [
                'attribute_code' => 'layer_images',
                'value' => $this->getLayerImagesPath($loadedProduct->getLayerImages())
            ],
            [
                'attribute_code' => 'image',
                'value' => $loadedProduct->getData('image')
            ],
            [
                'attribute_code' => 'url_key',
                'value' => $loadedProduct->getData('url_key')
            ],
            [
                'attribute_code' => 'color',
                'value' => $loadedProduct->getAttributeText('color')
            ],
            [
                'attribute_code' => 'series',
                'value' => $loadedProduct->getAttributeText('series')
            ],
            [
                'attribute_code' => 'connection_size',
                'value' => $loadedProduct->getAttributeText('connection_size')
            ],
            [
                'attribute_code' => 'type2',
                'value' => $loadedProduct->getAttributeText('type2')
            ]
        ];
 
        foreach ($customAttributes as $attribute) {
            $product->setCustomAttribute($attribute['attribute_code'], $attribute['value']);
        }
 
        $extensionAttributes = $product->getExtensionAttributes();
        if ($extensionAttributes === null) {
            $extensionAttributes = $this->extensionFactory->create();
        }
        $extensionAttributes->setIsInWishlist($isInWishlist);
        $extensionAttributes->setAverageRating($averageRating);
        $extensionAttributes->setShowAddtocartButton($showAddToCartButton);
        $product->setExtensionAttributes($extensionAttributes);
    }
 
    private function getCustomerIdFromRequest()
    {
        $request = \Magento\Framework\App\ObjectManager::getInstance()->get('Magento\Framework\App\RequestInterface');
        return $request->getParam('customerId', null);
    }
 
    private function isProductInWishlist($customerId, $product)
    {
        if (!$customerId) {
            return false;
        }
 
        $wishlist = $this->wishlistFactory->create()->loadByCustomerId($customerId, true);
        $wishlistItems = $wishlist->getItemCollection();
        foreach ($wishlistItems as $item) {
            if ($item->getProductId() == $product->getId()) {
                return true;
            }
        }
 
        return false;
    }
 
    private function getManufacturerImagePath($data)
    {
        if ($this->imageHelper->getManufacturerImage($data)) {
            $imagePath = 'attributeoption/image/' . $this->imageHelper->getManufacturerImage($data);
            return $imagePath;
        } else {
            return '';
        }
    }
 
    private function getLayerImagesPath($optionId)
    {
        if ($this->imageHelper->getLayerImage($optionId)) {
            $imagePath = 'attributeoption/image/' . $this->imageHelper->getLayerImage($optionId);
            return $imagePath;
        } else {
            return '';
        }
    }
}