<?php
namespace Xtwo\ProductListingApi\Model;

use Xtwo\ProductListingApi\Api\SearchResultLayeredNavigationInterface;
use Magento\Catalog\Model\Layer\Resolver as LayerResolver;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Catalog\Model\CategoryFactory;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Eav\Model\ResourceModel\Entity\Attribute\CollectionFactory as EavAttributeCollectionFactory;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;

class SearchResultLayeredNavigation implements SearchResultLayeredNavigationInterface
{
    protected $layerResolver;
    protected $productCollectionFactory;
    protected $categoryFactory;
    protected $storeManager;
    protected $request;
    protected $eavAttributeCollectionFactory;
    protected $visibility;
    protected $categoryCollectionFactory;

    public function __construct(
        LayerResolver $layerResolver,
        ProductCollectionFactory $productCollectionFactory,
        CategoryFactory $categoryFactory,
        StoreManagerInterface $storeManager,
        RequestInterface $request,
        Visibility $visibility,
        CategoryCollectionFactory $categoryCollectionFactory,
        EavAttributeCollectionFactory $eavAttributeCollectionFactory
    ) {
        $this->layerResolver = $layerResolver;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->categoryFactory = $categoryFactory;
        $this->storeManager = $storeManager;
        $this->request = $request;
        $this->eavAttributeCollectionFactory = $eavAttributeCollectionFactory;
        $this->visibility = $visibility;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
    }

    public function getSearchResultLayeredNavigation($searchkey, $filters = [])
    {
        if (strlen($searchkey) < 3) {
            return [
                'error' => true,
                'message' => 'Search key must have at least 3 characters.',
            ];
        }

        $storeId = $this->storeManager->getStore()->getId();
        $attributesToFilter = ['price', 'manufacturer', 'color', 'series', 'delivery_time', 'type2', 'connection_size'];
        $filters = $this->processFilters($this->request->getParams());
		
		// Fetch the original product collection without filters
		$originalProductCollection = $this->getOriginalProductCollection($searchkey, $attributesToFilter, $storeId);
		
		// Fetch product collection based on selected filters
        $filteredProductCollection = $this->getAttributeOptionsByFilters($searchkey, $filters, $attributesToFilter, $storeId);
		
		// Replace filtered attribute options with original ones
		$filteredProductCollection = $this->replaceFilteredWithOriginalOptions($filteredProductCollection, $originalProductCollection, $filters);

        // Fetch matching categories
        $subcategories = $this->getCategoriesBySearchKey($searchkey);

        // Merge and return filtered data
        return array_filter(array_merge($subcategories, $filteredProductCollection));
    }

    private function processFilters($filters)
    {
        foreach ($filters as $attributeCode => $attributeValue) {
            if (strpos($attributeValue, ',') !== false) {
                $filters[$attributeCode] = explode(',', $attributeValue);
            }
        }
        return $filters;
    }
	
	private function replaceFilteredWithOriginalOptions($filteredCollection, $originalCollection, $appliedFilters)
{
    $replacedCollection = [];

    // Iterate over each attribute in the filtered collection
    foreach ($filteredCollection as $attribute) {
        // Check if the attribute is one of the applied filters
        if (isset($attribute['code']) && array_key_exists($attribute['code'], $appliedFilters)) {
            // Find the corresponding attribute in the original collection
            foreach ($originalCollection as $originalAttribute) {
                if (isset($originalAttribute['code']) && $originalAttribute['code'] === $attribute['code']) {
                    // Replace the options in the filtered collection with those from the original collection
                    $attribute['options'] = $originalAttribute['options'];
                    break; // Stop once we've found and replaced the matching attribute
                }
            }
        }
        $replacedCollection[] = $attribute; // Add the modified attribute to the new collection
    }

    return $replacedCollection;
}
	
	private function getOriginalProductCollection($searchkey, $attributesToFilter, $storeId)
	{
		$originalProductCollection = $this->productCollectionFactory->create();
        $originalProductCollection->addAttributeToSelect($attributesToFilter)
            ->addAttributeToFilter('visibility', ['in' => [Visibility::VISIBILITY_BOTH, Visibility::VISIBILITY_IN_CATALOG]])
            ->addAttributeToFilter('status', \Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED)
            ->addAttributeToFilter('type_id', ['neq' => 'configurable'])
            ->addAttributeToFilter(
                [
                    ['attribute' => 'name', 'like' => '%' . $searchkey . '%'],
                    ['attribute' => 'sku', 'like' => '%' . $searchkey . '%']
                ]
            );
		//echo 'originalProductCollection count: ' . $originalProductCollection->getSize();

	   // Retrieve and format the available attribute options
			$attributeValues = [];
			foreach ($attributesToFilter as $attributeCode) {
				if ($attributeCode !== 'price') {
					$formattedOptions = $this->getFormattedAttributeOptions($attributeCode, $originalProductCollection);
					if (!empty($formattedOptions)) {
						$attributeValues[] = $formattedOptions;
					}
				}
			}

			// Return the final response, omitting empty sections
			return array_filter($attributeValues);
	}

    private function getAttributeOptionsByFilters($searchkey, $filters, $attributesToFilter, $storeId)
    {
        $productCollection = $this->productCollectionFactory->create();
        $productCollection->addAttributeToSelect($attributesToFilter)
            ->addAttributeToFilter('visibility', ['in' => [Visibility::VISIBILITY_BOTH, Visibility::VISIBILITY_IN_CATALOG]])
            ->addAttributeToFilter('status', \Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED)
            ->addAttributeToFilter('type_id', ['neq' => 'configurable'])
            ->addAttributeToFilter(
                [
                    ['attribute' => 'name', 'like' => '%' . $searchkey . '%'],
                    ['attribute' => 'sku', 'like' => '%' . $searchkey . '%']
                ]
            );
	
        // Dynamically apply filters based on request parameters
		foreach ($filters as $attributeCode => $attributeValues) {
			if (in_array($attributeCode, $attributesToFilter) && !empty($attributeValues)) {
				$productCollection->addAttributeToFilter($attributeCode, ['in' => $attributeValues]);
			}
		}
		//echo 'Product collection after filter: ' . $productCollection->getSize();
        // Collect price range and other filtered data
        $pricesort = $this->getPriceSort($productCollection, $filters);
        $attributeValues = $this->getFormattedAttributes($attributesToFilter, $productCollection);

        return array_filter(array_merge([$pricesort], $attributeValues));
    }

    private function getPriceSort($productCollection, $filters)
    {
        $layerInfoPrice = [];
        foreach ($productCollection as $product) {
            if ($product->getPrice()) {
                $layerInfoPrice[] = round($product->getPrice(), 2);
            }
        }

        if (!empty($layerInfoPrice)) {
            $showPrice = array_unique($layerInfoPrice);
            $currency = $filters['currency'] ?? $this->storeManager->getStore()->getCurrentCurrencyCode();
            $currencyExchange = $this->storeManager->getStore()->getBaseCurrency()->getRate($currency);

            $minPrice = round(min($showPrice) * $currencyExchange, 2);
            $maxPrice = round(max($showPrice) * $currencyExchange, 2);

            return [
                'pricesort_navigation' => [
                    'price' => ['min' => $minPrice, 'max' => $maxPrice],
                    'showSort' => [
                        'position' => 'Topseller',
                        'name' => 'Product Name',
                        'price' => 'Price',
                        'delivery_time' => 'Delivery Time',
                    ],
                ]
            ];
        }
        return [];
    }

    private function getFormattedAttributes($attributesToFilter, $productCollection)
    {
        $attributeValues = [];
        foreach ($attributesToFilter as $attributeCode) {
            if ($attributeCode !== 'price') {
                $attributeValues[] = $this->getFormattedAttributeOptions($attributeCode, $productCollection);
            }
        }
        return array_filter($attributeValues);
    }

    private function getFormattedAttributeOptions($attributeCode, $productCollection)
	{
		$attributeValueCodes = [];

		// Collect unique attribute values from the filtered product collection
		foreach ($productCollection as $product) {
			$attributeValue = $product->getData($attributeCode);
			if ($attributeValue && !in_array($attributeValue, $attributeValueCodes)) {
				$attributeValueCodes[] = $attributeValue;
			}
		}

		if (empty($attributeValueCodes)) {
			return []; // Return empty array if no codes are found
		}

		// Retrieve the attribute using the EAV Attribute Collection Factory
		$attribute = $this->eavAttributeCollectionFactory->create()
			->addFieldToFilter('attribute_code', $attributeCode)
			->getFirstItem();

		if (!$attribute->getId()) {
			return []; // Return empty array if attribute is not found
		}

		// Set the store ID for store-specific labels
		$storeId = $this->storeManager->getStore()->getId();

		$formattedOptions = [];
		foreach ($attributeValueCodes as $valueCode) {
			// Load the attribute option label per store directly
			$label = $attribute->getSource()->getOptionText($valueCode);

			// Remove HTML tags from the label, if any
			$label = strip_tags($label);

			// Check if the label is an array (for multi-language support) and fallback to default label if needed
			if (is_array($label)) {
				$label = $label[$storeId] ?? reset($label); // Default to store label or fallback if none
			}
			$valueShow = (int)$valueCode;
			$formattedOptions[] = [
				'value' => (string)$valueShow,
				'label' => $label
			];
		}

		// Return attribute label and its available options
		return [
			'name' => $productCollection->getResource()->getAttribute($attribute->getAttributeCode())->getStoreLabel($storeId),
			'code' => $attribute->getAttributeCode(),
			'options' => $formattedOptions
		];
	}


    private function getCategoriesBySearchKey($searchkey)
    {
        $categoryCollection = $this->categoryCollectionFactory->create()
            ->addAttributeToSelect('name')
            ->addAttributeToFilter('is_active', 1)
            ->addAttributeToFilter('name', ['like' => '%' . $searchkey . '%']);

        return $this->getMainParentCategoriesNew($categoryCollection);
    }

    private function getMainParentCategoriesNew($categoryCollection)
    {
        $mainCategories = [];
        $baseUrl = $this->storeManager->getStore()->getBaseUrl();

        foreach ($categoryCollection as $category) {
            $mainCategory = $this->getMainParentCategory($category);
            $mainCategoryName = $mainCategory->getName();

            if (isset($mainCategories[$mainCategoryName])) {
                continue;
            }

            $formattedUrl = str_replace([$baseUrl, ".html"], ["/category/", ""], $mainCategory->getUrl());
            $mainCategories[$mainCategoryName] = ['name' => $mainCategoryName, 'url' => $formattedUrl];
        }
		
		// Format the final array with unique main categories
		$categoryArray[] = [
			'id' => 'category',
			'name' => 'category',
			'items' => array_values($mainCategories) // Keep only unique values
		];

		return $categoryArray;

    }

    private function getMainParentCategory($category)
    {
        foreach ($category->getParentCategories() as $parentCategory) {
            if ($parentCategory->getLevel() == 2) {
                return $parentCategory;
            }
        }
        return $category;
    }
}