<?php
namespace Xtwo\ProductListingApi\Model;

use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Catalog\Model\CategoryFactory;
use Magento\Catalog\Model\Product\Visibility;
use Xtwo\AttributeOption\Helper\Image as ImageHelper;
use Magento\Store\Model\StoreManagerInterface;
use Magento\CatalogInventory\Api\StockRegistryInterface;
use Magento\Wishlist\Model\WishlistFactory;
use Xtwo\CustomApis\Helper\Data as CustomApisHelper;
use Magento\Review\Model\ReviewFactory;
use Magento\Review\Model\ResourceModel\Review\CollectionFactory as ReviewCollectionFactory;
use Magento\Review\Model\ResourceModel\Rating\Option\Vote\Collection as RatingOptionCollection;
use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Catalog\Model\ResourceModel\Product\Collection as ProductCollection;

class ProductListing implements \Xtwo\ProductListingApi\Api\ProductListingInterface
{
    protected $categoryRepository;
    protected $productCollectionFactory;
    private $categoryFactory;
    protected $visibility;
    protected $imageHelper;
    protected $storeManager;
    protected $stockRegistry;
    protected $mediaBaseUrl;
    protected $wishlistFactory;
    protected $customApisHelper;
    protected $reviewFactory;
    protected $reviewCollectionFactory;
    protected $ratingOption;
    protected $helper;

    public function __construct(
        CategoryRepositoryInterface $categoryRepository,
        ProductCollectionFactory $productCollectionFactory,
        CategoryFactory $categoryFactory,
        Visibility $visibility,
        ImageHelper $imageHelper,
        StoreManagerInterface $storeManager,
        StockRegistryInterface $stockRegistry,
        WishlistFactory $wishlistFactory,
        CustomApisHelper $customApisHelper,
        ReviewFactory $reviewFactory,
        ReviewCollectionFactory $reviewCollectionFactory,
        RatingOptionCollection $ratingOption
    ) {
        $this->categoryRepository = $categoryRepository;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->categoryFactory = $categoryFactory;
        $this->visibility = $visibility;
        $this->imageHelper = $imageHelper;
        $this->storeManager = $storeManager;
        $this->stockRegistry = $stockRegistry;
        $this->mediaBaseUrl = $this->storeManager->getStore()->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA);
        $this->wishlistFactory = $wishlistFactory;
        $this->customApisHelper = $customApisHelper;
        $this->reviewFactory = $reviewFactory;
        $this->reviewCollectionFactory = $reviewCollectionFactory;
        $this->ratingOption = $ratingOption;
    }

    public function getProductListingByUrl($mainCategoryUrl, $subCategoryUrl1 = null, $subCategoryUrl2 = null, $currentPage = null, $pageSize = null, $customerId = null, $attrCode = null, $attrValue = null, $attrprice = null, $lteq = null, $gteq = null, $sortAttribute = null, $sortDirection = null)
    {
        $productCollection = $this->productCollectionFactory->create()
            ->addAttributeToFilter('visibility', ['in' => [Visibility::VISIBILITY_BOTH, Visibility::VISIBILITY_IN_CATALOG]])
            ->addAttributeToFilter('status', \Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED)
            ->addAttributeToSelect(['sku', 'name', 'price', 'created_at', 'delivery_time', 'msrp', 'manufacturer', 'image', 'url_key', 'color', 'series', 'connection_size', 'tiles_color_group', 'layer_images'])
            ->setPageSize($pageSize)
            ->setCurPage($currentPage);

        $categoryIds = [];

        if (!empty($mainCategoryUrl)) {
            $category = $this->categoryFactory->create()->loadByAttribute('url_key', $mainCategoryUrl);
            if ($category) {
                $mainCatId = $category->getId();
                if ($category->getIsAnchor()) {
                    $categoryIds = $category->getChildrenCategories()->getAllIds();
                } else {
                    $categoryIds = [$mainCatId];
                }
            }
        }

        if (!empty($subCategoryUrl1)) {
            $subCategory = $this->categoryFactory->create()->loadByAttribute('url_key', $subCategoryUrl1);
            if ($subCategory) {
                $categoryIds = [$subCategory->getId()];
            }
        }

        if (!empty($subCategoryUrl2)) {
            $subCategory = $this->categoryFactory->create()->loadByAttribute('url_key', $subCategoryUrl2);
            if ($subCategory) {
                $categoryIds = [$subCategory->getId()];
            }
        }

        if (!empty($categoryIds)) {
            $productCollection->addCategoriesFilter(['in' => $categoryIds]);
            $stringCategoryIds = implode(',', $categoryIds);
            if(!empty($subCategoryUrl1) || !empty($subCategoryUrl2)){
                $productCollection->getSelect()->join(
                    ['category_product' => $productCollection->getTable('catalog_category_product')],
                    'e.entity_id = category_product.product_id AND category_product.category_id = ' . $stringCategoryIds,
                    ['position']
                );
                $productCollection->getSelect()->where('category_product.position >= 0');
                $productCollection->getSelect()->order('category_product.position ' . $sortDirection);
            }
        }

        if (!empty($attrCode)) {
            $cusRequest = array_combine($attrCode, $attrValue);
            foreach ($cusRequest as $key => $val) {
                $productCollection->addAttributeToFilter($key, ['in' => $val]);
            }
        }

        if (!empty($attrprice)) {
            $productCollection->addAttributeToFilter('price', ['gteq' => $lteq])
                ->addAttributeToFilter('price', ['lteq' => $gteq]);
        }

        if (!empty($sortAttribute)) {
            if ($sortAttribute && $sortAttribute == 'position' && $mainCategoryUrl) {
                $mainCategory = $this->categoryFactory->create()->loadByAttribute('url_key', $mainCategoryUrl);
                if ($mainCategory && $mainCategory->getId()) {
                    $mainCategoryId = $mainCategory->getId();
                    if(empty($subCategoryUrl1) && empty($subCategoryUrl2)){
                        $productCollection->getSelect()->join(
                            ['category_product' => $productCollection->getTable('catalog_category_product')],
                            'e.entity_id = category_product.product_id AND category_product.category_id = ' . $mainCategoryId,
                            ['position']
                        );
                        $productCollection->getSelect()->where('category_product.position >= 0');
                        $productCollection->getSelect()->order('category_product.position ' . $sortDirection);
                    }

                }
            } elseif($sortAttribute && $sortAttribute == 'position' && $subCategoryUrl1){
                $subCategory1 = $this->categoryFactory->create()->loadByAttribute('url_key', $subCategoryUrl1);
                if ($subCategory1 && $subCategory1->getId()) {
                    $productCollection->getSelect()->where('category_product.position >= 0');
                    $productCollection->getSelect()->order('category_product.position ' . $sortDirection);
                }
            } elseif($sortAttribute && $sortAttribute == 'position' && $subCategoryUrl2){
                $subCategory2 = $this->categoryFactory->create()->loadByAttribute('url_key', $subCategoryUrl2);
                if ($subCategory2 && $subCategory2->getId()) {
                    $productCollection->getSelect()->where('category_product.position >= 0');
                    $productCollection->getSelect()->order('category_product.position ' . $sortDirection);
                }
            }else{
                $productCollection->addAttributeToSort($sortAttribute, $sortDirection);
            }
        }

        $finalArray = [];

        foreach ($productCollection as $data) {
            preg_match('/href="(.*?)"/', $data->getAttributeText('manufacturer'), $matches);

            $stockItem = $this->stockRegistry->getStockItem($data->getId(), $data->getStore()->getWebsiteId());
            $manufacturerImagePath = $this->getManufacturerImagePath($data);
            $layerImageOptionId = $data->getLayerImages();

            $isInWishlist = false;
            if (!empty($customerId)) {
                $wishlist = $this->wishlistFactory->create()->loadByCustomerId($customerId, true);
                $wishlistItems = $wishlist->getItemCollection();
                foreach ($wishlistItems as $item) {
                    if ($item->getProductId() == $data->getId()) {
                        $isInWishlist = true;
                        break;
                    }
                }
            }

            $showButtons = $this->customApisHelper->getButtonStatus($data);

            $basicInfo = [
                'id' => $data->getId(),
                'sku' => $data->getSku(),
                'name' => $data->getName(),
                'price' => $data->getPrice(),
                'created_at' => $data->getCreatedAt(),
                'extension_attributes' => [
                    'stock' => $stockItem->getQty(),
                    'is_in_stock' => $stockItem->getIsInStock()
                ],
                'custom_attributes' => [
                    [
                        'attribute_code' => 'delivery_time',
                        'value' => $data->getAttributeText('delivery_time')
                    ],
                    [
                        'attribute_code' => 'msrp',
                        'value' => $data->getData('msrp') ?: ''
                    ],
                    [
                        'attribute_code' => 'manufacturer',
                        'value' => [
                            strip_tags($data->getAttributeText('manufacturer')),
                            isset($matches[1]) ? str_replace('.html', '', $matches[1]) : '',
                            $manufacturerImagePath
                        ]
                    ],
                    [
                        'attribute_code' => 'layer_images',
                        'value' => $this->getLayerImagesPath($layerImageOptionId)
                    ],
                    [
                        'attribute_code' => 'image',
                        'value' => $data->getData('image')
                    ],
                    [
                        'attribute_code' => 'url_key',
                        'value' => $data->getData('url_key')
                    ],
                    [
                        'attribute_code' => 'color',
                        'value' => $data->getAttributeText('color')
                    ],
                    [
                        'attribute_code' => 'series',
                        'value' => $data->getAttributeText('series')
                    ],
                    [
                        'attribute_code' => 'connection_size',
                        'value' => $data->getAttributeText('connection_size')
                    ],
                    [
                        'attribute_code' => 'type2',
                        'value' => $data->getAttributeText('type2')
                    ]
                ],
                'is_in_wishlist' => $isInWishlist,
                'average_rating' => $this->customApisHelper->getAverageRating($data->getId()),
                'show_addtocart_button' => $showButtons,
            ];

            $finalArray[] = $basicInfo;
        }

        $result = [
            'items' => $finalArray,
            'search_criteria' => [
                'page_size' => $pageSize,
                'current_page' => $currentPage
            ],
            'total_count' => $productCollection->getSize()
        ];

        return [$result];
    }



    private function getManufacturerImagePath($data)
    {
        if ($this->imageHelper->getManufacturerImage($data)) {
            $imagePath = 'attributeoption/image/' . $this->imageHelper->getManufacturerImage($data);
            return $imagePath;
        } else {
            return '';
        }
    }

    private function getLayerImagesPath($optionId)
    {
        if ($this->imageHelper->getLayerImage($optionId)) {
            $imagePath = 'attributeoption/image/' . $this->imageHelper->getLayerImage($optionId);
            return $imagePath;
        } else {
            return '';
        }
    }
}

