<?php
namespace Xtwo\ProductListingApi\Model;

use Xtwo\ProductListingApi\Api\LayeredNavigationInterface;
use Magento\Catalog\Model\Layer\Resolver as LayerResolver;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Catalog\Model\CategoryFactory;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\App\RequestInterface;
use Magento\Eav\Model\ResourceModel\Entity\Attribute\CollectionFactory as EavAttributeCollectionFactory;
use Magento\Catalog\Model\Product\Visibility;

class LayeredNavigation implements LayeredNavigationInterface
{
    protected $layerResolver;
    protected $productCollectionFactory;
    protected $categoryFactory;
    protected $storeManager;
    protected $request;
    protected $eavAttributeCollectionFactory;
	protected $visibility;

    public function __construct(
        LayerResolver $layerResolver,
        ProductCollectionFactory $productCollectionFactory,
        CategoryFactory $categoryFactory,
        StoreManagerInterface $storeManager,
        RequestInterface $request,
		Visibility $visibility,
        EavAttributeCollectionFactory $eavAttributeCollectionFactory
    ) {
        $this->layerResolver = $layerResolver;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->categoryFactory = $categoryFactory;
        $this->storeManager = $storeManager;
        $this->request = $request;
        $this->eavAttributeCollectionFactory = $eavAttributeCollectionFactory;
		$this->visibility = $visibility;
    }

	public function getLayeredNavigation($categoryId, $filters = [])
	{
		$storeId = $this->storeManager->getStore()->getId();
		$layer = $this->layerResolver->get();
		$layer->setCurrentCategory($categoryId);

		// Define attributes to be filtered
		$attributesToFilter = ['price', 'manufacturer', 'color', 'series', 'delivery_time', 'type2', 'connection_size'];

		// Get selected filters from request
		$filters = $this->request->getParams();

		// Process filters to handle comma-separated values
		foreach ($filters as $attributeCode => $attributeValue) {
			// Check if the attribute value contains a comma (multiple values)
			if (strpos($attributeValue, ',') !== false) {
				// Split the values by comma into an array
				$filters[$attributeCode] = explode(',', $attributeValue);
			}
		}

		// Fetch the original product collection without filters
		$originalProductCollection = $this->getOriginalProductCollection($categoryId, $attributesToFilter, $storeId);

		// Fetch the filtered product collection based on selected filters
		$filteredProductCollection = $this->getAttributeOptionsByFilters($categoryId, $filters, $attributesToFilter, $storeId);

		// Replace filtered attribute options with original ones
		$filteredProductCollection = $this->replaceFilteredWithOriginalOptions($filteredProductCollection, $originalProductCollection, $filters);

		// Fetch subcategories (of current category or parent if none)
		$subcategories = $this->getSubcategoriesOrParent($categoryId);

		// Merge results, excluding empty data
		$mergedArray = array_filter(array_merge($subcategories, $filteredProductCollection));

		// Prepare response to include both collections
		/*return [
			//'original_product_collection' => $originalProductCollection,
			'filtered_product_collection' => $mergedArray
		];*/
		return $mergedArray;
	}

	private function replaceFilteredWithOriginalOptions($filteredCollection, $originalCollection, $appliedFilters)
	{
		$replacedCollection = [];

		// Iterate over each attribute in the filtered collection
		foreach ($filteredCollection as $attribute) {
			// Check if the attribute is one of the applied filters
			if (isset($attribute['code']) && array_key_exists($attribute['code'], $appliedFilters)) {
				// Find the corresponding attribute in the original collection
				foreach ($originalCollection as $originalAttribute) {
					if (isset($originalAttribute['code']) && $originalAttribute['code'] === $attribute['code']) {
						// Replace the options in the filtered collection with those from the original collection
						$attribute['options'] = $originalAttribute['options'];
						break; // Stop once we've found and replaced the matching attribute
					}
				}
			}
			$replacedCollection[] = $attribute; // Add the modified attribute to the new collection
		}

		return $replacedCollection;
	}


	private function getOriginalProductCollection($categoryId, $attributesToFilter, $storeId)
	{
		$originalProductCollection = $this->productCollectionFactory->create();
		$originalProductCollection->setStoreId($storeId);
		$originalProductCollection->addAttributeToFilter('visibility', ['in' => [Visibility::VISIBILITY_BOTH, Visibility::VISIBILITY_IN_CATALOG]]);
		$originalProductCollection->addAttributeToFilter('status', \Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED);

		// Explicitly filter for products directly assigned to the category
		$originalProductCollection->getSelect()->join(
			['ccp' => 'catalog_category_product'],
			'e.entity_id = ccp.product_id AND ccp.category_id = ' . (int)$categoryId,
			[]
		);

		// Add all attributes to the product collection
		$originalProductCollection->addAttributeToSelect($attributesToFilter);
		//echo 'originalProductCollection count: ' . $originalProductCollection->getSize();

	   // Retrieve and format the available attribute options
			$attributeValues = [];
			foreach ($attributesToFilter as $attributeCode) {
				if ($attributeCode !== 'price') {
					$formattedOptions = $this->getFormattedAttributeOptions($attributeCode, $originalProductCollection);
					if (!empty($formattedOptions)) {
						$attributeValues[] = $formattedOptions;
					}
				}
			}

			// Return the final response, omitting empty sections
			return array_filter($attributeValues);
	}


	private function getAttributeOptionsByFilters($categoryId, $filters, $attributesToFilter, $storeId)
	{
		$productCollection = $this->productCollectionFactory->create();
		$productCollection->setStoreId($storeId);
		$productCollection->addAttributeToFilter('visibility', ['in' => [Visibility::VISIBILITY_BOTH, Visibility::VISIBILITY_IN_CATALOG]]);
		$productCollection->addAttributeToFilter('status', \Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED);

		// Explicitly filter for products directly assigned to the category
		$productCollection->getSelect()->join(
			['ccp' => 'catalog_category_product'],
			'e.entity_id = ccp.product_id AND ccp.category_id = ' . (int)$categoryId,
			[]
		);

		

		// Add all attributes to the product collection
		$productCollection->addAttributeToSelect($attributesToFilter);
		
		// Dynamically apply filters based on request parameters
		foreach ($filters as $attributeCode => $attributeValues) {
			if (in_array($attributeCode, $attributesToFilter) && !empty($attributeValues)) {
				$productCollection->addAttributeToFilter($attributeCode, ['in' => $attributeValues]);
			}
		}
		
		//echo 'Product collection count: ' . $productCollection->getSize();
		//print_r($productCollection->getData());
		// Initialize price array
		$layerInfoPrice = [];
		$pricesort = [];

		// Loop through each product to collect prices
		foreach ($productCollection as $product) {
			if (!empty($product->getPrice())) {
				$price = round($product->getPrice(), 2);
				$layerInfoPrice[] = $price;
			}
		}

		// Calculate min and max prices
		if (!empty($layerInfoPrice)) {
			$showPrice = array_unique($layerInfoPrice);
			$currency = isset($filters['currency']) ? $filters['currency'] : $this->storeManager->getStore()->getCurrentCurrencyCode();
			$currencyExchange = $this->storeManager->getStore()->getBaseCurrency()->getRate($currency);

			if (!empty($currencyExchange)) {
				$pricesort['price']['min'] = round(min($showPrice) * $currencyExchange, 2);
				$pricesort['price']['max'] = round(max($showPrice) * $currencyExchange, 2);
			} else {
				$pricesort['price']['min'] = round(min($showPrice), 2);
				$pricesort['price']['max'] = round(max($showPrice), 2);
			}
		}

		// Only add showSort if pricesort is not empty
		if (!empty($pricesort)) {
			$pricesort['showSort'] = [
				'position' => 'Topseller',
				'name' => 'Product Name',
				'price' => 'Price',
				'delivery_time' => 'Delivery Time'
			];
		}

		// Prepare the final response with pricesort_navigation if it exists
		$response = !empty($pricesort) ? ['pricesort_navigation' => $pricesort] : [];

		// Retrieve and format the available attribute options
		$attributeValues = [];
		foreach ($attributesToFilter as $attributeCode) {
			if ($attributeCode !== 'price') {
				$formattedOptions = $this->getFormattedAttributeOptions($attributeCode, $productCollection);
				if (!empty($formattedOptions)) {
					$attributeValues[] = $formattedOptions;
				}
			}
		}

		// Return the final response, omitting empty sections
		return array_filter(array_merge([$response], $attributeValues));
	}

	private function getFormattedAttributeOptions($attributeCode, $productCollection)
	{
		$attributeValueCodes = [];

		// Collect unique attribute values from the filtered product collection
		foreach ($productCollection as $product) {
			$attributeValue = $product->getData($attributeCode);
			if ($attributeValue && !in_array($attributeValue, $attributeValueCodes)) {
				$attributeValueCodes[] = $attributeValue;
			}
		}

		if (empty($attributeValueCodes)) {
			return []; // Return empty array if no codes are found
		}

		// Retrieve the attribute using the EAV Attribute Collection Factory
		$attribute = $this->eavAttributeCollectionFactory->create()
			->addFieldToFilter('attribute_code', $attributeCode)
			->getFirstItem();

		if (!$attribute->getId()) {
			return []; // Return empty array if attribute is not found
		}

		// Set the store ID for store-specific labels
		$storeId = $this->storeManager->getStore()->getId();

		$formattedOptions = [];
		foreach ($attributeValueCodes as $valueCode) {
			// Load the attribute option label per store directly
			$label = $attribute->getSource()->getOptionText($valueCode);

			// Remove HTML tags from the label, if any
			$label = strip_tags($label);

			// Check if the label is an array (for multi-language support) and fallback to default label if needed
			if (is_array($label)) {
				$label = $label[$storeId] ?? reset($label); // Default to store label or fallback if none
			}
			$valueShow = (int)$valueCode;
			$formattedOptions[] = [
				'value' => (string)$valueShow,
				'label' => $label
			];
		}

		// Return attribute label and its available options
		return [
			'name' => $productCollection->getResource()->getAttribute($attribute->getAttributeCode())->getStoreLabel($storeId),
			'code' => $attribute->getAttributeCode(),
			'options' => $formattedOptions
		];
	}

    /**
	 * Get enabled subcategories for the given category or its parent if no subcategories are found.
	 */
	private function getSubcategoriesOrParent($categoryId)
	{
		$category = $this->categoryFactory->create()->load($categoryId);
		$subcategories = $category->getChildrenCategories();

		// If no subcategories found, get subcategories of the parent category
		if ($subcategories->count() == 0 && $category->getParentId() != 0) {
			$parentCategory = $this->categoryFactory->create()->load($category->getParentId());
			$subcategories = $parentCategory->getChildrenCategories();
		}

		// Format subcategories data, only for active (enabled) categories
		$subcategoryData = [];
		foreach ($subcategories as $subcategory) {
			if ($subcategory->getIsActive()) {  // Check if the subcategory is enabled
				$baseUrl = $this->storeManager->getStore()->getBaseUrl(); // Get the base URL
				$subcategoryUrl = $subcategory->getUrl();
				
				// Replace the base URL with "/category"
				$formattedUrl = str_replace($baseUrl, "/category/", $subcategoryUrl);
				
				$moreFormattedUrl = str_replace(".html", "", $formattedUrl);

				$subcategoryData[] = [
					'name' => $subcategory->getName(),
					'url' => $moreFormattedUrl
				];
			}
		}

		$categoryArray[] = [
			'id' => 'category',
			'name' => 'category',
			'items' => $subcategoryData
		];

		return $categoryArray;
	}

}