<?php
namespace Xtwo\PayonePostgateway\Model;


use Xtwo\PayonePostgateway\Api\CapturePaymentInterface;
use Payone\Core\Model\PayoneConfig;
use Payone\Core\Model\Methods\PayoneMethod;
use Magento\Sales\Model\Order;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Framework\App\RequestInterface;
use Xtwo\PayonePostgateway\Logger\Logger;


/**
 * PayonePayment API implementation.
 */
class CapturePayment implements CapturePaymentInterface 
{
	protected $logger;
    
	/**
     * Order id
     *
     * @var string
     */
    protected $sOrderId = null;

    /**
     * Array or request parameters
     *
     * @var array
     */
    protected $aParameters = [];

    /**
     * Response of the request
     *
     * @var array
     */
    protected $aResponse = false;

    /**
     * URL of PAYONE Server API
     *
     * @var string
     */
    protected $sApiUrl = 'https://api.pay1.de/post-gateway/';

    /**
     * Map for custom parameters to be added $sParamName => $sConfigName
     *
     * @var array
     */
    protected $aCustomParamMap = [
        'mid' => 'mid',
        'portalid' => 'portalid',
        'aid' => 'aid',
        'key' => 'key',
        'request' => 'request',
    ];

    /**
     * PAYONE shop helper
     *
     * @var \Payone\Core\Helper\Shop
     */
    protected $shopHelper;

    /**
     * PAYONE environment helper
     *
     * @var \Payone\Core\Helper\Environment
     */
    protected $environmentHelper;

    /**
     * PAYONE api helper
     *
     * @var \Payone\Core\Helper\Api
     */
    protected $apiHelper;

    /**
     * API-log resource model
     *
     * @var \Payone\Core\Model\ResourceModel\ApiLog
     */
    protected $apiLog;

    /**
     * Store id for the current context
     *
     * @var string
     */
    protected $storeCode = null;
	protected $cartRepository;
	protected $request;

    /**
     * Constructor
     *
     * @param \Payone\Core\Helper\Shop                $shopHelper
     * @param \Payone\Core\Helper\Environment         $environmentHelper
     * @param \Payone\Core\Helper\Api                 $apiHelper
     * @param \Payone\Core\Model\ResourceModel\ApiLog $apiLog
     */
    public function __construct(
        \Payone\Core\Helper\Shop $shopHelper,
        \Payone\Core\Helper\Environment $environmentHelper,
        \Payone\Core\Helper\Api $apiHelper,
        \Payone\Core\Model\ResourceModel\ApiLog $apiLog,
		CartRepositoryInterface $cartRepository,
		Logger $logger,
		RequestInterface $request
    ) {
        $this->shopHelper = $shopHelper;
        $this->environmentHelper = $environmentHelper;
        $this->apiHelper = $apiHelper;
        $this->apiLog = $apiLog;
		$this->cartRepository = $cartRepository;
		$this->request = $request;
		$this->logger = $logger;
        $this->initRequest();
    }

    /**
     * Initialize request
     * Set all default parameters
     *
     * @return void
     */
	 
	 protected function initRequest()
    {
		$cartId = $this->request->getParam('cartId');
		$txid = $this->request->getParam('txid');
		$this->logger->info('Payone Capture data: for CART ID=> '.$cartId.'txid=> '.$txid);
		$quote = $this->cartRepository->getActive($cartId);
		
		$GrandTotal = $quote->getGrandTotal()*100;
		$pseudocardpan = $this->request->getParam('pseudocardpan');	
        $this->aParameters = [];
        $this->addParameter('mid', $this->shopHelper->getConfigParam('mid', 'global', 'payone_general', $this->storeCode)); // PayOne Merchant ID
        $this->addParameter('portalid', $this->shopHelper->getConfigParam('portalid', 'global', 'payone_general', $this->storeCode)); // PayOne
        $this->addParameter('aid', '47566'); // PayOne Portal ID
		$this->addParameter('key', md5($this->shopHelper->getConfigParam('key', 'global', 'payone_general', $this->storeCode) ?? '')); // PayOne Portal Key     
        $this->addParameter('request', 'capture'); // Module version
        $this->addParameter('mode', 'test'); // Module version
        $this->addParameter('currency', $quote->getQuoteCurrencyCode()); // Module version  
        $this->addParameter('amount', $GrandTotal); // Module version
        $this->addParameter('txid', $txid); // Module version
    }
	
	 /**
     * Set current store code and reinit base parameters
     *
     * @param  string $sStoreCode
     * @return void
     */
    public function setStoreCode($sStoreCode)
    {
        if ($this->storeCode != $sStoreCode) {
            $this->storeCode = $sStoreCode;
            $this->initRequest(); //reinit base parameters
        }
    }
	
	
	/**
     * Add parameter to request
     *
     * @param  string $sKey               parameter key
     * @param  string $sValue             parameter value
     * @param  bool   $blAddAsNullIfEmpty add parameter with value NULL if empty. Default is false
     * @return void
     */
    public function addParameter($sKey, $sValue, $blAddAsNullIfEmpty = false)
    {
        if ($blAddAsNullIfEmpty === true && empty($sValue)) {
            $sValue = 'NULL'; // add value as string NULL - needed in certain situations
        }
        $this->aParameters[$sKey] = $sValue;
    }
	
    /**
     * Remove parameter from request
     *
     * @param  string $sKey parameter key
     * @return void
     */
    public function removeParameter($sKey)
    {
        if (array_key_exists($sKey, $this->aParameters)) {// is parameter set?
            unset($this->aParameters[$sKey]);
        }
    }

    /**
     * Get parameter from request or return false if parameter was not set
     *
     * @param  string $sKey parameter key
     * @return string|bool
     */
    public function getParameter($sKey)
    {
        if (array_key_exists($sKey, $this->aParameters)) {// is parameter set?
            return $this->aParameters[$sKey];
        }
        return false;
    }

    /**
     * Return all parameters
     *
     * @return array
     */
    public function getParameters()
    {
        return $this->aParameters;
    }

    /**
     * Set response array
     *
     * @param  $aResponse
     * @return void
     */
    public function setResponse($aResponse)
    {
        $this->aResponse = $aResponse;
    }

    /**
     * Return the response array
     *
     * @return array
     */
    public function getResponse()
    {
        return $this->aResponse;
    }

    /**
     * Add non-global parameters specifically configured in the payment type
     *
     * @param  PayoneMethod $oPayment
     * @return void
     */
    protected function addCustomParameters(PayoneMethod $oPayment)
    {
        foreach ($this->aCustomParamMap as $sParamName => $sConfigName) {// add all custom parameters
            $sCustomConfig = $oPayment->getCustomConfigParam($sConfigName); // get custom config param
            if (!empty($sCustomConfig)) { // only add if the param is configured
                if ($sConfigName == 'key') {
                    $this->addParameter($sParamName, md5($sCustomConfig)); // key isn't hashed in db
                } else {
                    $this->addParameter($sParamName, $sCustomConfig); // add custom param to request
                }
            }
        }
    }

    /**
     * Set the order id that is associated with this request
     *
     * @param  string $sOrderId
     * @return void
     */
    public function setOrderId($sOrderId)
    {
        $this->sOrderId = $sOrderId;
    }

    /**
     * Return order id if set
     *
     * @return string
     */
    public function getOrderId()
    {
        //if ($this->sOrderId !== null) {// was order id set?
            return '45';
        //}
        //return '';
    }

    /**
     * Add the redirect urls to the request
     *
     * @param  PayoneMethod $oPayment
     * @param  Order        $oOrder
     * @return void
     */
    protected function addRedirectUrls(PayoneMethod $oPayment, Order $oOrder = null)
    {
        $this->addParameter('successurl', $oPayment->getSuccessUrl($oOrder));
        $this->addParameter('errorurl', $oPayment->getErrorUrl());
        $this->addParameter('backurl', $oPayment->getCancelUrl());
    }

    /**
     * Validate if all general required parameters are set
     *
     * @return bool
     */
    protected function validateParameters()
    {
        if ($this->getParameter('mid') === false || $this->getParameter('portalid') === false ||
            $this->getParameter('key') === false || $this->getParameter('mode') === false) {
            return false;
        }
        return true;
    }

    /**
     * Send the previously prepared request, log request and response into the database and return the response

     * @param  PayoneMethod $oPayment
     * @return array
     */
    public function send(PayoneMethod $oPayment = null)
    {	
        if ($oPayment !== null && $oPayment->hasCustomConfig()) { // if payment type doesnt use the global settings
            $this->addCustomParameters($oPayment); // add custom connection settings
        }

        //$sRequestUrl = "https://api.pay1.de/post-gateway/?mid=47289&portalid=2034580&key=1898cd23b5fda169b9f59d009d8c0b5f&encoding=UTF-8&integrator_name=Magento2&integrator_version=2.4.4-p1&solution_name=fatchip&solution_version=3.7.0&request=authorization&mode=test&aid=47566&reference=100156993&amount=15000&currency=EUR&ip=117.214.187.109&firstname=sdfds&lastname=dsfd&street=Hoterheideweg&zip=40670&city=Meerbusch&country=DE&email=hemendra@ssds.com&telephonenumber=3453444&language=de&shipping_firstname=sdfds&shipping_lastname=dsfd&shipping_street=Hoterheideweg&shipping_zip=40670&shipping_city=Meerbusch&shipping_country=DE&clearingtype=cc&pseudocardpan=9410010000842625501&cardholder=hemendra+ajmera&successurl=http://xtwo.localhost/payone/onepage/returned/?incrementId=100156993&errorurl=http://xtwo.localhost/payone/onepage/cancel?error=1&backurl=http://xtwo.localhost/payone/onepage/cancel/";
		
      
       $sRequestUrl = $this->apiHelper->getRequestUrl($this->getParameters(), $this->sApiUrl);
       $aResponse = $this->apiHelper->sendApiRequest($sRequestUrl); // send request to PAYONE
	   
	   $this->logger->info('Payone Capture data: for Request=> '.print_r($sRequestUrl,true).'Payone Capture data: for Response=> '.print_r($aResponse,true).'Payone Capture data End');
	  
	  
        $this->setResponse($aResponse);

        //$this->apiLog->addApiLogEntry($this->getParameters(), $aResponse, $aResponse['status'], $this->getOrderId()); // log request to db

        return [$aResponse];
    }
	
}
