<?php
/**
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_Insurance
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 */
namespace Xtwo\Insurance\Helper;

class Data extends \Magento\Framework\App\Helper\AbstractHelper {

    const ATTRIBUTE_SET_NAME = 'Insurance';
    const TYPE_ID = 'buyerprotect';

    /**
     * @var \Magento\Checkout\Model\Session
     */
    protected $checkoutSession;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Checkout\Model\Cart
     */
    protected $checkoutCart;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Checkout\Model\Session $checkoutSession,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Checkout\Model\Cart $checkoutCart,
        \Magento\Eav\Model\Entity\Attribute\Set $attributesetICollection,
        \Magento\Catalog\Model\ResourceModel\Product\Attribute\CollectionFactory $productAttributeCollection,
        \Psr\Log\LoggerInterface $logger
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->storeManager = $storeManager;
        $this->scopeConfig = $scopeConfig;
        $this->checkoutCart = $checkoutCart;
        $this->attributesetICollection = $attributesetICollection;
        $this->productAttributeCollection = $productAttributeCollection;
        $this->logger = $logger;
        parent::__construct(
            $context
        );
    }

    /**
     * @param \Magento\Store\Model\Store $store
     * @return \Magento\Framework\DataObject|null
     */
    public function getTransportInsuranceProduct($store = null) {
        $attributeSetId = $this->getAttributeSetId();

        /** @var \Magento\Quote\Model\Quote $quote */
        $quote = $this->checkoutSession->getQuote();
        if (!$store) $store = $quote->getStore();

        if (!$this->isActive($store)) return null;

        $product = null;

        try {
            if ($this->isStaticActive($store)) {
                if (!$attributeSetId) {
                    $this->log('Insurance active, but required AttributeSet not configured.');
                    return null;
                }
                $product = $this->getStaticProductForQuote($quote, $store); // may be null

            } elseif ($this->isDynamicActive()) {
                if (!$attributeSetId) {
                    $this->log('Insurance active, but required AttributeSet not configured.');
                    return null;
                }
                $product = $this->getDynamicProductForQuote($quote, $store);
            }
        } catch (Mage_Core_Model_Store_Exception $e) {
            //ignore
        }
        return $product;
    }

    /**
     * @param \Magento\Quote\Model\Quote $quote
     * @param \Magento\Store\Model\Store $store
     * @return \Magento\Catalog\Model\Product|null
     * @throws \Mage\Core\Model\Store\Exception
     */
    private function getStaticProductForQuote($quote, $store=null) {
        $subTotal = (float)$quote->getBaseSubtotal();
        $sku = null;
        $config = $this->getStaticConfig($store);
        $mappings = array_reverse($config['mapping']);
        foreach ($mappings as $data) {
            if ($subTotal >= $data['cart_price']) $sku = $data['value'];
        }
        if ($sku) return $this->getInsuranceProduct($sku, $store);
        return null;
    }

    /**
     * @param \Magento\Quote\Model\Quote $quote
     * @param \Magento\Store\Model\Store $store
     * @throws \Mage\Core\Model\Store\Exception
     * @return \Magento\Catalog\Model\Product|null
     */
    private function getDynamicProductForQuote($quote, $store=null) {
        $config = $this->getDynamicConfig($store ? $store : $quote->getStore());
        if ($sku = $config['sku']) return $this->getInsuranceProduct($sku, $store);
        return null;
    }

    /**
     * @param \Magento\Catalog\Model\Product $product
     * @param \Magento\Checkout\Model\Cart $cart
     * @return bool true if price has been set
     */
    public function setDynamicPrice($product, $cart) {
        if ($this->isDynamicActive($product->getStore())) {
            $subTotal = (float)$cart->getQuote()->getBaseSubtotal();
            $config = $this->getDynamicConfig($product->getStore());
            $mappings = array_reverse($config['mapping']);
            foreach ($mappings as $data) {
                if ($subTotal >= $data['cart_price']) {
                    $price = $config['isPercentage'] ? (($subTotal/100) * floatval($data['value'])) :
                        floatval($data['value']);
                    $product->setPrice(round($price, 2));
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * @param $sku
     * @param $store
     * @return \Magento\Catalog\Model\Product|null
     * @throws \Mage\Core\Model\Store\Exception
     */
    private function getInsuranceProduct($sku, $store) {
        if (!$store) $store = $this->storeManager->getStore();
        $attributeSetId = $this->getAttributeSetId();
        /** @var \Magento\Catalog\Model\ResourceModel\Product\Collection $products */
        $products = Mage::getModel('catalog/product');
        $products->addFilter('sku', $sku);
        $products->addFieldToFilter('attribute_set_id', $attributeSetId);
        if ($store) $products->addStoreFilter($store);
        $products->addFieldToFilter('status', \Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED);
        $products->addFieldToFilter('type_id', self::TYPE_ID);
        $products->addAttributeToSelect('*');
        if ($products->getFirstItem() && $products->getFirstItem()->getId()) {
            /** @var \Magento\Catalog\Model\Product $product */
            $product = $products->getFirstItem();
            if ($store) {
                $websiteIds = $product->getWebsiteIds();
                if (!in_array($store->getWebsiteId(), $websiteIds)) {
                    return null;
                }
            }
            if (!$product->isInStock()) {
                return null;
            }
            return $product;
        }
        return null;
    }

    /**
     * @param \Magento\Store\Model\Store $store
     * @return bool
     */
    public function isActive($store=null) {
        $isActive = (bool) $this->scopeConfig->getValue('insurances/general/active', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $store);
        return $isActive;
    }

    /**
     * @param \Magento\Store\Model\Store $store
     * @return bool
     */
    protected function isStaticActive($store=null) {
        $isActive = (bool) $this->scopeConfig->getValue('insurances/static/active', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $store);
        return $isActive && !$this->isDynamicActive($store);
    }

    /**
     * @param \Magento\Store\Model\Store $store
     * @return bool
     */
    public function isDynamicActive($store=null) {
        $isActive = (bool) $this->scopeConfig->getValue('insurances/dynamic/active', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $store);
        return $isActive && !$this->isStaticActive($store);
    }

    protected function getStaticConfig($store=null) {
        $mapping = $this->getMappingConfig('insurances/static/mappings', $store);
        return [
            'mapping' => $mapping
        ];
    }

    protected function getDynamicConfig($store=null) {
        $mapping = $this->getMappingConfig('insurances/dynamic/mappings', $store);
        return [
            'mapping' => $mapping,
            'sku' => $this->scopeConfig->getValue('insurances/dynamic/product_sku', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $store),
            'isPercentage' => (bool)$this->scopeConfig->getValue('insurances/dynamic/percentage_active', \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $store)
        ];
    }

    /**
     * @param $path
     * @param null $store
     * @return array
     */
    protected function getMappingConfig($path, $store=null) {
        $result = [];
        $mappingLines = explode(PHP_EOL, $this->scopeConfig->getValue($path, \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $store));
        foreach ($mappingLines as $mappingLine) {
            $line = explode('#', trim($mappingLine));
            if (count($line) == 2) $result[] = [
                'cart_price' => floatval(str_replace(',', '.', $line[0])) ,
                'value' => $line[1]
            ];
        }
        return $result;
    }

    /**
     * @return int|null
     */
    private function getAttributeSetId() {
        /** @var \Magento\Eav\Model\ResourceModel\Entity\Attribute\Set\Collection $attributeSets */
        // $attributeSets = Mage::getModel("eav/entity_attribute_set");
        $attributeSets = $this->attributesetICollection->create();
        $attributeSets->addFieldToFilter("attribute_set_name", self::ATTRIBUTE_SET_NAME);
        $attributeSets->addFieldToFilter("entity_type_id", 4);
        /** @var \Magento\Eav\Model\Entity\Attribute\Set $attributeSet */
        $attributeSet = $attributeSets->getFirstItem();
        return $attributeSet && $attributeSet->getId() ? $attributeSet->getAttributeSetId() : null;
    }

    /**
     * @return \Magento\Quote\Model\Quote\Item[]
     * @throws Mage_Core_Model_Store_Exception
     */
    public function getInsuranceProductsInCart() {
        $result = [];
        /* @var $cart Mage_Checkout_Model_Cart */
        $cart = $this->checkoutCart
            ->setStore($this->storeManager->getStore());
        $quoteItems = $cart->getQuote()->getAllItems();
        if (!empty($quoteItems)) {
            $attributeSetId = $this->getAttributeSetId();
            /** @var \Magento\Quote\Model\Quote\Item $quoteItem */
            foreach ($quoteItems as $quoteItem) {
                $product = $quoteItem->getProduct();
                if ($product
                    && $product->getData('attribute_set_id') == $attributeSetId
                    && $product->getTypeId() == self::TYPE_ID) {
                    $result[] = $quoteItem;
                }
            }
        }
        return $result;
    }

    /**
     * @return int
     * @throws Mage_Core_Model_Store_Exception
     */
    public function hasInsuranceInCart() {
        $count = count($this->getInsuranceProductsInCart());
        return $count > 0;
    }

    /**
     * @param $msg
     * @param int $level
     */
    public function log($msg, $level=\Zend_Log::INFO) {
        $this->logger->log($level, $msg);
    }
}


