<?php

namespace Xtwo\CustomApis\Model;

use Magento\Customer\Api\AddressRepositoryInterface;
use Magento\Customer\Api\Data\AddressInterfaceFactory;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Framework\Api\ExtensibleDataInterface;
use Magento\Framework\Reflection\DataObjectProcessor;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Quote\Api\Data\AddressInterface;
use Magento\Quote\Api\Data\EstimateAddressInterface;
use Magento\Quote\Api\Data\ShippingMethodInterface;
use Magento\Quote\Model\Cart\ShippingMethodConverter;
use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\TotalsCollector;
use Magento\Quote\Model\ResourceModel\Quote\Address as QuoteAddressResource;
use Magento\Quote\Model\ShippingMethodManagement as MagentoShippingMethodManagement;
use Magento\Tax\Model\Calculation as TaxCalculation;
use Magento\Catalog\Model\Product;
use Magento\Catalog\Helper\Data as TaxHelper;

class ShippingMethodManagement extends MagentoShippingMethodManagement
{
    private $dataProcessor;
    private $customerSession;
    private $taxCalculation;
    protected $productModel;
    protected $taxHelper;


    public function __construct(
        CartRepositoryInterface $quoteRepository,
        ShippingMethodConverter $converter,
        AddressRepositoryInterface $addressRepository,
        TotalsCollector $totalsCollector,
        AddressInterfaceFactory $addressFactory = null,
        QuoteAddressResource $quoteAddressResource = null,
        CustomerSession $customerSession = null,
        DataObjectProcessor $dataProcessor = null,
        TaxCalculation $taxCalculation = null,
        Product $productModel,
        TaxHelper $taxHelper
    ) {
        parent::__construct(
            $quoteRepository,
            $converter,
            $addressRepository,
            $totalsCollector,
            $addressFactory,
            $quoteAddressResource,
            $customerSession
        );

        $this->customerSession = $customerSession ?: \Magento\Framework\App\ObjectManager::getInstance()
            ->get(CustomerSession::class);
        $this->dataProcessor = $dataProcessor ?: \Magento\Framework\App\ObjectManager::getInstance()
            ->get(DataObjectProcessor::class);
        $this->taxCalculation = $taxCalculation ?: \Magento\Framework\App\ObjectManager::getInstance()
            ->get(TaxCalculation::class);
        $this->productModel = $productModel;
        $this->taxHelper = $taxHelper;
    }

    public function estimateByExtendedAddress($cartId, AddressInterface $address)
    {
        /** @var Quote $quote */
        $quote = $this->quoteRepository->getActive($cartId);
        if ($quote->isVirtual() || 0 == $quote->getItemsCount()) {
            return [];
        }
        $shippingMethods = $this->getShippingMethods($quote, $address);

        $finalResponse = [];
        foreach ($shippingMethods as $shippingMethod) {
            $basePriceInclTax = $this->calculateBasePriceInclTax($quote, $shippingMethod->getBaseAmount());
            $methodData = [
                'carrier_code' => $shippingMethod->getCarrierCode(),
                'method_code' => $shippingMethod->getMethodCode(),
                'carrier_title' => $shippingMethod->getCarrierTitle(),
                'method_title' => $shippingMethod->getMethodTitle(),
                'amount' => $shippingMethod->getAmount(),
                'base_amount' => $shippingMethod->getBaseAmount(),
                'available' => $shippingMethod->getAvailable(),
                'error_message' => $shippingMethod->getErrorMessage(),
                'price_excl_tax' => $shippingMethod->getPriceExclTax(),
                'price_incl_tax' => $shippingMethod->getPriceInclTax(),
                'base_shipping_incl_tax' => $basePriceInclTax,
            ];

            $finalResponse[] = $methodData;
        }

        return $finalResponse;
    }

    private function getShippingMethods(Quote $quote, $address)
    {
        $output = [];
        $shippingAddress = $quote->getShippingAddress();
        $shippingAddress->addData($this->extractAddressData($address));
        $shippingAddress->setCollectShippingRates(true);

        $this->totalsCollector->collectAddressTotals($quote, $shippingAddress);
        $quoteCustomerGroupId = $quote->getCustomerGroupId();
        $customerGroupId = $this->customerSession->getCustomerGroupId();
        $isCustomerGroupChanged = $quoteCustomerGroupId !== $customerGroupId;
        if ($isCustomerGroupChanged) {
            $quote->setCustomerGroupId($customerGroupId);
        }
        $shippingRates = $shippingAddress->getGroupedAllShippingRates();
        foreach ($shippingRates as $carrierRates) {
            foreach ($carrierRates as $rate) {
                $output[] = $this->converter->modelToDataObject($rate, $quote->getQuoteCurrencyCode());
            }
        }
        if ($isCustomerGroupChanged) {
            $quote->setCustomerGroupId($quoteCustomerGroupId);
        }
        return $output;
    }

    private function extractAddressData($address)
    {
        $className = \Magento\Customer\Api\Data\AddressInterface::class;
        if ($address instanceof AddressInterface) {
            $className = AddressInterface::class;
        } elseif ($address instanceof EstimateAddressInterface) {
            $className = EstimateAddressInterface::class;
        }

        $addressData = $this->dataProcessor->buildOutputDataArray(
            $address,
            $className
        );
        unset($addressData[ExtensibleDataInterface::EXTENSION_ATTRIBUTES_KEY]);

        return $addressData;
    }

    private function calculateBasePriceInclTax(Quote $quote, $baseAmount)
    {
        $store = $quote->getStore();
        $taxRequest = $this->taxCalculation->getRateRequest(null, null, null, $store);
        $taxRate = $this->taxCalculation->getRate($taxRequest->setProductClassId(0));
        $basePriceInclTax = $baseAmount / (1.19);
        
        return $basePriceInclTax;
    }
}
