<?php
namespace Xtwo\CustomApis\Model;

use Xtwo\CustomApis\Api\CustomerReviewInterface;
use Xtwo\CustomApis\Helper\Data as CustomApisHelper;
use Magento\Review\Model\ResourceModel\Review\CollectionFactory as ReviewCollectionFactory;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Review\Model\RatingFactory;
use Magento\Review\Model\ReviewFactory;

class CustomerReview implements CustomerReviewInterface
{
    protected $reviewCollectionFactory;
    protected $productRepository;
    protected $storeManager;
    protected $ratingFactory;
    protected $customApisHelper;
    protected $reviewFactory;

    public function __construct(
        ReviewCollectionFactory $reviewCollectionFactory,
        ProductRepositoryInterface $productRepository,
        StoreManagerInterface $storeManager,
        RatingFactory $ratingFactory,
        CustomApisHelper $customApisHelper,
        ReviewFactory $reviewFactory
    ) {
        $this->reviewCollectionFactory = $reviewCollectionFactory;
        $this->productRepository = $productRepository;
        $this->storeManager = $storeManager;
        $this->ratingFactory = $ratingFactory;
        $this->customApisHelper = $customApisHelper;
        $this->reviewFactory = $reviewFactory;
    }

    public function getCustomerReviews($customerId, $currentPage, $pageSize)
    {
        $reviews = [];

        $reviewCollection = $this->reviewCollectionFactory->create()
            ->addCustomerFilter($customerId)
            ->addStatusFilter(\Magento\Review\Model\Review::STATUS_APPROVED)
            ->setCurPage($currentPage)
            ->setPageSize($pageSize)
            ->setDateOrder();

        $ratings = $this->customApisHelper->getRatingsForReviews($reviewCollection);
        foreach ($reviewCollection as $review) {
            $product = $this->productRepository->getById($review->getEntityPkValue());
            $rating = $ratings[$review->getId()] ?? 0;
            $reviewData = [
                'review_id' => $review->getId(),
                'created_date' => date("d-m-Y", strtotime($review->getCreatedAt())),
                'product_name' => $product->getName(),
                'product_link' => $product->getUrlKey(),
                'rating' => $rating,
                'evaluation' => $review->getDetail(),
            ];

            $reviews[] = $reviewData;
        }

        $actualTotalCount = $reviewCollection->getSize();

        $finalResponse = [
            'review_data' => $reviews,
            'searchCriteria' => [
                'page_size' => $pageSize,
                'current_page' => $currentPage,
                'total_count' => $actualTotalCount,
            ],
        ];
        return [$finalResponse];
    }

    public function getReviewDetails($reviewId,$customerId)
    {
        $reviewCollection = $this->reviewCollectionFactory->create()
                                ->addCustomerFilter($customerId)
                                ->addStatusFilter(\Magento\Review\Model\Review::STATUS_APPROVED)
                                ->addFieldToFilter('main_table.review_id', $reviewId);

        $ratings = $this->customApisHelper->getRatingsForReviews($reviewCollection);
        $productRatings = [];
        $reviewDetails = [];

        foreach ($reviewCollection as $review) {
            $product = $this->productRepository->getById($review->getEntityPkValue());
            $productImage = $this->getImageUrl($product);
            $rating = $ratings[$review->getId()] ?? 0;

            $reviewDetails[] = [
                'Product_name' => $product->getName(),
                'url_key' => $product->getUrlKey(),
                'Product_image' => $productImage,
                'Product_rating' => $rating,
                'product_average_rating' => $this->customApisHelper->getAverageRating($product->getId()),
                'Review_title' => $review->getTitle(),
                'Review_evaluation' => $review->getDetail(),
                'Review_date' => date("d F Y", strtotime($review->getCreatedAt())),
            ];

            $productRatings[$product->getId()][] = $rating;
        }
        if(!empty($reviewDetails)){
            return $reviewDetails;
        }else{
            throw new \Magento\Framework\Exception\NotFoundException(__('No Reviews'));
        }

    }

    private function getImageUrl($product)
    {
        $imageUrl = '';
        $imageAttribute = 'image';
        $image = $product->getData($imageAttribute);
        if(strpos($image,'/') === 0 ){
            $image = ltrim($image,'/');
        }
        if ($image) {
            $store = $this->storeManager->getStore();
            $imageUrl = $store->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA) . 'catalog/product/' . $image;
        }
        return $imageUrl;
    }

}
