<?php
namespace Xtwo\CustomApis\Helper;

use Magento\Review\Model\ResourceModel\Rating\Option\Vote\Collection as RatingOptionCollection;
use Magento\Review\Model\ResourceModel\Review\CollectionFactory as ReviewCollectionFactory;
use Magento\Catalog\Model\ProductRepository;
use Magento\Review\Model\RatingFactory;
use Magento\Directory\Model\CountryFactory;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\ObjectManagerInterface;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\Module\Dir\Reader;
use Magento\Framework\UrlInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Catalog\Model\CategoryFactory;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;

class Data extends AbstractHelper
{
    const XML_PATH_DEL_TIME = 'stockmanage/static/delivery_times';
    const XML_PATH_DEL_CUSTOM = 'stockmanage/static/custom';

    protected $ratingOption;
    protected $reviewCollectionFactory;
    protected $productRepository;
    protected $ratingFactory;
    protected $countryFactory;
    protected $storeManager;
    protected $currencyFactory;
    protected $objectManager;
    protected $moduleReader;
    protected $urlBuilder;
    protected $shippingInformationManagement;
    protected $productFactory;
    protected $quoteItemFactory;
    protected $orderItemFactory;
    protected $quoteRepository;
    protected $orderRepository;
    protected $cartRepository;
    protected $scopeConfig;
    protected $categoryFactory;
    protected $categoryCollectionFactory;

    public function __construct(
        RatingOptionCollection $ratingOption,
        ReviewCollectionFactory $reviewCollectionFactory,
        ProductRepository $productRepository,
        RatingFactory $ratingFactory,
        CountryFactory $countryFactory,
        StoreManagerInterface $storeManager,
        \Magento\Directory\Model\CurrencyFactory $currencyFactory,
        ObjectManagerInterface $objectManager,
        Reader $moduleReader,
        UrlInterface $urlBuilder,
        \Magento\Catalog\Model\ProductFactory $productFactory,
        \Magento\Quote\Model\Quote\ItemFactory $quoteItemFactory,
        \Magento\Sales\Model\Order\ItemFactory $orderItemFactory,
        \Magento\Quote\Api\CartRepositoryInterface $quoteRepository,
        \Magento\Sales\Model\OrderRepository $orderRepository,
        CartRepositoryInterface $cartRepository,
        ScopeConfigInterface $scopeConfig,
        CategoryFactory $categoryFactory,
        CategoryCollectionFactory $categoryCollectionFactory,
    )
    {
        $this->ratingOption = $ratingOption;
        $this->reviewCollectionFactory = $reviewCollectionFactory;
        $this->productRepository = $productRepository;
        $this->ratingFactory = $ratingFactory;
        $this->countryFactory = $countryFactory;
        $this->storeManager = $storeManager;
        $this->currencyFactory = $currencyFactory;
        $this->objectManager = $objectManager;
        $this->moduleReader = $moduleReader;
        $this->urlBuilder = $urlBuilder;
        $this->productFactory = $productFactory;
        $this->quoteItemFactory = $quoteItemFactory;
        $this->orderItemFactory = $orderItemFactory;
        $this->orderRepository = $orderRepository;
        $this->quoteRepository = $quoteRepository;
        $this->cartRepository = $cartRepository;
        $this->scopeConfig = $scopeConfig;
        $this->categoryFactory = $categoryFactory;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
    }

    public function getCurrencySymbol($storeId)
    {
        try {
            $store = $this->storeManager->getStore($storeId);
            $currencyCode = $store->getDefaultCurrencyCode();
            $currency = $this->currencyFactory->create()->load($currencyCode);
            $currencyCode = $currency->getCurrencyCode();
            if($currency->getCurrencySymbol() === null || $currency->getCurrencySymbol() === ''){
                return $currencyCode;
            }
            return $currency->getCurrencySymbol();
        } catch (\Exception $e) {
            return '$';
        }
    }

    public function calculateAverageRating($reviewCollection)
    {
        $ratings = $this->getRatingsForReviews($reviewCollection);
        if (empty($ratings)) {
            return 0;
        }
        $totalRating = array_sum($ratings);
        $averageRating = $totalRating / count($ratings);
        return $averageRating;
    }

    public function getRatingsForReviews($reviewCollection)
    {
        $reviewIds = [];
        foreach ($reviewCollection as $review) {
            $reviewIds[] = $review->getId();
        }

        $ratings = [];
        if (!empty($reviewIds)) {
            $ratingCollection = $this->ratingOption
                ->addRatingInfo()
                ->addOptionInfo()
                ->addRatingOptions()
                ->addFieldToFilter('review_id', ['in' => $reviewIds]);

            foreach ($ratingCollection as $data) {
                $ratings[$data->getReviewId()] = $data->getValue();
            }
        }
        return $ratings;
    }

    public function getCountryName($countryCode)
    {
        try {
            $locale = 'en_US';
            $country = $this->countryFactory->create()->loadByCode($countryCode);
            return $country->getName($locale);
        } catch (\Exception $e) {
            return 'Unknown';
        }
    }

    public function getStoreUrlByStoreId($storeId)
    {
        $store = $this->storeManager->getStore($storeId);
        $storeUrl = $store->getBaseUrl();

        return $storeUrl;
    }

    public function checkCountryAndStoreMatching($countryCode, $storeId)
    {
        $store = $this->storeManager->getStore($storeId);
        $storeCountryCode = $store->getConfig('general/country/default');

        return strtoupper($countryCode) === strtoupper($storeCountryCode);
    }

    public function getAverageRating($productId)
    {
        $product = $this->objectManager->create('Magento\Catalog\Model\Product')->load($productId);
        $reviewFactory = $this->objectManager->create('Magento\Review\Model\Review');
        $storeManager = $this->objectManager->create('\Magento\Store\Model\StoreManagerInterface');
        $storeId = $storeManager->getStore()->getStoreId();
        $reviewFactory->getEntitySummary($product, $storeId);
        $ratingSummary = $product->getRatingSummary()->getRatingSummary();
        $averageRating = round($ratingSummary / 20, 1);

        return $averageRating;
    }

    public function getCountryFlagIconUrl($countryCode)
    {
        $path = 'static/frontend/Xtwo/base/de_DE/Xtwo_Storeswitch/images/flags/';
        $flagIconPath = $path.strtolower($countryCode).'.svg';
        return $flagIconPath;
    }

    public function addProductToOrder($order, $tsProductId, $cartId)
    {
        $product = $this->productFactory->create()->load($tsProductId);
        $orderItem = $this->orderItemFactory->create();
        $orderItem->setProductId($product->getId());
        $orderItem->setName($product->getName());
        $orderItem->setSku($product->getSku());
        $orderItem->setQtyOrdered(1);
        $orderItem->setBasePrice($product->getPrice());
        $orderItem->setPrice($product->getFinalPrice());
        $orderItem->setRowTotal($product->getFinalPrice());
        $orderItem->setOriginalPrice($product->getPrice());
        $orderItem->setProductType($product->getTypeId());

        $order->addItem($orderItem);
        $this->orderRepository->save($order);

        return $order;
    }

    public function addProductToQuote($quote, $productId, $quantity = 1.0)
    {
        $product = $this->productRepository->getById($productId);
        $quoteItem = $this->quoteItemFactory->create();
        $quoteItem->setProduct($product);
        $quoteItem->setQty($quantity);
        $quote->addItem($quoteItem);
        if ($productId) {
            $quoteItem->setData('ts_product_id', $productId);
        }
    }

    public function getMainParentCategories($categoryCollection)
    {
        $mainCategories = [];

        foreach ($categoryCollection as $category) {
            $mainCategory = $this->getMainParentCategory($category);
            $mainCategoryName = $mainCategory->getName();
            $categoryName = $category->getName();

            $mainCategoryLink = $this->getCategoryLink($mainCategory);

            $categoryData = [
                'name' => $categoryName,
                'link' => 'category/'.$category->getUrlKey(),
            ];

            if (!isset($mainCategories[$mainCategoryName])) {
                $mainCategories[$mainCategoryName] = [
                    'main_category' => $mainCategoryName,
                    'link' => $mainCategoryLink,
                    'subcategories' => [],
                ];
            }

            $mainCategories[$mainCategoryName]['subcategories'][] = $categoryData;
        }

        return array_values($mainCategories);
    }

    public function getMainParentCategoriesNew($categoryCollection)
    {
        $mainCategories = [];

        foreach ($categoryCollection as $category) {
            $mainCategory = $this->getMainParentCategory($category);
            $mainCategoryName = $mainCategory->getName();
            $categoryName = $category->getName();

            $mainCategoryLink = $this->getCategoryLink($mainCategory);
			
            $categoryData = [
                'name' => $categoryName,
                'link' => 'category/'.$mainCategoryLink.'/'.$category->getUrlKey(),
            ];

            if (!isset($mainCategories[$mainCategoryName])) {
                $mainCategories[$mainCategoryName] = [
                    'main_category' => $mainCategoryName,
                    'link' => $mainCategoryLink,
                    'subcategories' => [],
                ];
            }

            $mainCategories[$mainCategoryName]['subcategories'][] = $categoryData;
        }

        return array_values($mainCategories);
    }

    private function getMainParentCategory($category)
    {
        $parentCategories = $category->getParentCategories();

        foreach ($parentCategories as $parentCategory) {
            if ($parentCategory->getLevel() == 2) {
                return $parentCategory;
            }
        }

        return $category;
    }

    private function getCategoryLink($category)
    {
        return $category->getUrlKey();
    }

    public function getButtonStatus($product){
        $storeScope = \Magento\Store\Model\ScopeInterface::SCOPE_STORE;
        $spec_time = $this->scopeConfig->getValue(self::XML_PATH_DEL_TIME, $storeScope);

        $spec_time = $this->scopeConfig->getValue(self::XML_PATH_DEL_TIME, $storeScope);
        $spec_custom = $this->scopeConfig->getValue(self::XML_PATH_DEL_CUSTOM, $storeScope);
        $product = $this->productRepository->getById($product->getData('entity_id'));

        $deltime = $product->getData('delivery_time');

        $exp = explode(',',$spec_custom);
        
        $cnt  = count($exp);
        $productPrice = $product->getPrice();
        if(($productPrice)<=0 ){
            return false;
        }else {

            if($cnt > 0 && !is_null($spec_custom)){
                if(in_array($deltime,$exp) && $spec_time == 'delivery_time'){
                    return false;
                }
            }
        }

        return true;
    }

    public function getSubcategoriesByParentId($parentCategoryId)
    {
        $categoryCollection = $this->categoryCollectionFactory->create();
        $categoryCollection->addAttributeToSelect('url_key') 
            ->addAttributeToSelect('name') 
            ->addAttributeToFilter('parent_id', $parentCategoryId)
            ->addIsActiveFilter()
            ->setOrder('position', 'ASC'); 

        $subcategories = [];
        foreach ($categoryCollection as $category) {
            $subcategories[] = [
                'id' => $category->getId(),
                'url_key' => $category->getUrlKey(),
            ];
        }

        return $subcategories;
    }

    public function findCategoryIdByUrlKey($categories, $urlKey)
    {
        foreach ($categories as $category) {
            if ($category['url_key'] === $urlKey) {
                return $category['id'];
            }
        }
        return null;
    }

    public function getCategoryLevels($mainCategoryUrl, $subCategoryUrl1, $subCategoryUrl2, $storeId)
    {
        $categoryIds = [];
        $urls = array_filter([$mainCategoryUrl, $subCategoryUrl1, $subCategoryUrl2]);
    
        if (!empty($urls)) {
            $categories = $this->loadCategoriesByUrls($urls, $storeId);
            $mainCategory = $this->findCategoryByUrlAndLevel($categories, $mainCategoryUrl, 2);
            if ($mainCategory) {
                $mainCatId = $mainCategory->getId();
                $categoryIds[] = $mainCatId;
                if (!empty($subCategoryUrl1)) {
                    $subCategory = $this->findCategoryByUrlParentAndLevel($categories, $subCategoryUrl1, $mainCatId, 3);
                    if ($subCategory) {
                        $categoryIds = [$subCategory->getId()];
                        if (!empty($subCategoryUrl2)) {
                            $subSubCategory = $this->findCategoryByUrlParentAndLevel($categories, $subCategoryUrl2, $subCategory->getId(), 4);
                            if ($subSubCategory) {
                                $categoryIds = [$subSubCategory->getId()];
                            } else {
                                throw new \Magento\Framework\Exception\NoSuchEntityException(
                                    __('Products not found for the provided sub-subcategory URL key: %1', $subCategoryUrl2)
                                );
                            }
                        }
                    } else {
                        throw new \Magento\Framework\Exception\NoSuchEntityException(
                            __('Products not found for the provided subcategory URL key: %1', $subCategoryUrl1)
                        );
                    }
                }
            }else{
                throw new \Magento\Framework\Exception\NoSuchEntityException(
                    __('Products not found for the provided subcategory URL key: %1', $urls)
                );
            }
        }
    
        return $categoryIds;
    }
    
    private function loadCategoriesByUrls($urls, $storeId)
    {
        $categoryCollection = $this->categoryCollectionFactory->create();
        $categoryCollection->addAttributeToSelect(['url_key', 'level', 'parent_id'])
            ->addAttributeToFilter('url_key', ['in' => $urls])
            ->addIsActiveFilter()
            ->setStoreId($storeId);
    
        return $categoryCollection->getItems();
    }
    
    private function findCategoryByUrlAndLevel($categories, $urlKey, $level)
    {
        foreach ($categories as $category) {
            if ($category->getUrlKey() == $urlKey && $category->getLevel() == $level) {
                return $category;
            }
        }
        return null;
    }
    
    private function findCategoryByUrlParentAndLevel($categories, $urlKey, $parentId, $level)
    {
        foreach ($categories as $category) {
            if ($category->getUrlKey() == $urlKey && $category->getParentId() == $parentId && $category->getLevel() == $level) {
                return $category;
            }
        }
        return null;
    }    

}
