<?php
/**
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_CoreHelpers
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 * @author      Hemendra Ajmera
 */

 namespace Xtwo\CoreHelpers\Model;
 
use Magento\Framework\Model\AbstractModel;

class StatusNotificationMessage extends AbstractModel
{
    const STATUS_NOTIFICATION_MSG_FILE_PREFIX = 'status_notification_message_';
    const DIR_FILECACHE = '/var/js-helpers';

    protected $_lineErrors = array();
    protected $_lineWarnings = array();
    protected $_errors = array();
    protected $_successes = array();
    protected $_warnings = array();
    protected $_mailTopic = '';
    protected $_emailRecipients = array();
    protected $_usage = 'default';

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    public function __construct(
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Store\Model\StoreManagerInterface $storeManager
       ) {
        $this->scopeConfig = $scopeConfig;
        $this->storeManager = $storeManager;
       }

    /**
     * @param string $msg
     * @param int $line
     * @return Xtwo_CoreHelpers_Model_StatusNotificationMessage
     */
    public function addErrorForLine($msg, $line) {
        if (!empty($msg)) {
            $newContent = $msg;
            if (array_key_exists($line, $this->_lineErrors)) {
                $newContent = $this->_lineErrors[$line].', '.$msg;
            }
            $this->_lineErrors[$line] = $newContent;
        }
        return $this;
    }

    /**
     * @param string $msg
     * @param int $line
     * @return Xtwo_CoreHelpers_Model_StatusNotificationMessage
     */
    public function addWarningForLine($msg, $line) {
        if (!empty($msg)) {
            $newContent = $msg;
            if (array_key_exists($line, $this->_lineWarnings)) {
                $newContent = $this->_lineWarnings[$line].', '.$msg;
            }
            $this->_lineWarnings[$line] = $newContent;
        }
        return $this;
    }

    /**
     * @param string $msg
     * @return Xtwo_CoreHelpers_Model_StatusNotificationMessage
     */
    public function addError($msg) {
        if (!empty($msg)) $this->_errors[] = $msg;
        return $this;
    }

    /**
     * @param string $msg
     * @return Xtwo_CoreHelpers_Model_StatusNotificationMessage
     */
    public function addWarning($msg) {
        if (!empty($msg)) $this->_warnings[] = $msg;
        return $this;
    }

    /**
     * @param string $msg
     * @return Xtwo_CoreHelpers_Model_StatusNotificationMessage
     */
    public function addSuccess($msg) {
        if (!empty($msg)) $this->_successes[] = $msg;
        return $this;
    }

    /**
     * @param string $value
     * @return string
     */
    public function setMailSubjectTopic($value) {
        $this->_mailTopic = $value;
        return $this;
    }

    /**
     * Adds given recipients as recipient
     * @param string|array $recipient
     * @return Xtwo_CoreHelpers_Model_StatusNotificationMessage
     */
    public function addRecipient($recipient) {
        $data = is_array($recipient) ? $recipient : array($recipient);
        $this->_emailRecipients = array_merge($this->_emailRecipients, $data);
        return $this;
    }

    /**
     * Builds standard message. It is by default only sent once per day if given $force is not set.
     */
    public function send() {
        $subject = sprintf('Statusnotification %s (%s): %s',
            Js_SystemMode::getProjectName(), Js_SystemMode::getSystemScope(), $this->_mailTopic);
        $msg = $this->getFullMessage();
        if (empty($msg)) return;

        $sender = Js_SystemMode::getProjectName().'_'.Js_SystemMode::getSystemScope().'@justselling.de';
        $header = 'From: '.$sender. "\r\n" .
            'Reply-To: noreply@justselling.de' . "\r\n" .
            'X-Mailer: PHP/' . phpversion();
        $recipients = $this->getEmailRecipients();
        foreach ($recipients as $recipient) {
            if (!mail($recipient, $subject, $msg, $header)) {
                Mage::log($subject.' (mail) could not be sent, rec='.$recipient, \Zend_Log::ERR);
            }
        }
        $logFile = Mage::getBaseDir().Justselling_CoreHelpers_Helper_Data::DIR_WORK.
            '/status_notification_message_'.$this->_usage.'.log';
        if (!file_put_contents($logFile, "Subject: \n".$subject."\n\nContent: \n".$msg)) {
            Mage::log($subject.' (mail) could not be logged into '.
                Xtwo_CoreHelpers_Helper_Data::DIR_WORK, \Zend_Log::ERR);
        }
        $this->removeFile();
    }

    protected function getFullMessage() {
        $msg = '';
        {
            if (count($this->_successes)) $msg .= "\nErfolgreich: " . count($this->_successes) . "\n";
            foreach ($this->_successes as $item) {
                $msg .= $item . "\n";
            }
            if (count($this->_errors)) $msg .= "\nFehler: " . count($this->_errors) . "\n";
            foreach ($this->_errors as $item) {
                $msg .= $item . "\n";
            }
            if (count($this->_warnings)) $msg .= "\nWarnungen: " . count($this->_warnings) . "\n";
            foreach ($this->_warnings as $item) {
                $msg .= $item . "\n";
            }
        }
        {
            if (count($this->_lineErrors)) $msg .= "\nFehler in Zeilen: " . count($this->_lineErrors) . "\n";
            foreach ($this->_lineErrors as $line => $item) {
                $msg .= $line . ': ' . $item . "\n";
            }
            if (count($this->_lineWarnings)) $msg .= "\nWarnungen in Zeilen: " . count($this->_lineWarnings) . "\n";
            foreach ($this->_lineWarnings as $line => $item) {
                $msg .= $line . ': ' . $item . "\n";
            }
        }
        return $msg;
    }

    /** @return array */
    protected function getEmailRecipients() {
        $configValue = trim($this->scopeConfig->getValue('dev/statusnoti/emails',
        \Magento\Store\Model\ScopeInterface::SCOPE_STORE));
        $defaultRecipients = empty($configValue) ? array() : explode(',', $configValue);
        $recipients = array_merge($this->_emailRecipients, $defaultRecipients);
        $recipients = array_unique($recipients);
        return $recipients;
    }

    /**
     * @param $value
     * @return Xtwo_CoreHelpers_Model_StatusNotificationMessage
     */
    protected function setUsage($value) {
        $this->_usage = $value;
        return $this;
    }

    /**
     * Reads persisted or
     * @param $usage
     * @return Xtwo_CoreHelpers_Model_StatusNotificationMessage|mixed
     */
    public static function init($usage) {
        $fileName = sprintf('/%s%s.txt', self::STATUS_NOTIFICATION_MSG_FILE_PREFIX, $usage);
        $filePath = Mage::getBaseDir().self::DIR_FILECACHE.$fileName;
        if (file_exists($filePath)) {
            $data = file_get_contents($filePath);
            if ($data) {
                $instance = unserialize($data);
                if ($instance) {
                    return $instance;
                } else {
                    Js_Log::log(sprintf('jSCoreHelper: Unable to unserialize status_notification message "%s". Deleted it, served new.',
                        $filePath), '', \Zend_Log::ALERT);
                }
            } else {
                unlink($filePath);
                Js_Log::log(sprintf('jSCoreHelper: Unable to load status_notification message "%s". Deleted it, served new.',
                    $filePath), '', \Zend_Log::ALERT);
            }
        }
        $model = Mage::getModel('jscorehelpers/statusNotificationMessage');
        return $model->setUsage($usage);
    }

    /** Removes the usage-related status notification message file */
    protected function removeFile() {
        $fileName = sprintf('/%s%s.txt', self::STATUS_NOTIFICATION_MSG_FILE_PREFIX, $this->_usage);
        $filePath = Mage::getBaseDir().self::DIR_FILECACHE.$fileName;
        if (file_exists($filePath)) {
            if (!unlink($filePath)) {
                //Js_Log::log(sprintf('jSCoreHelper: Unable to remove status_notification message "%s". ', $filePath),
                //    '', \Zend_Log::ALERT);
            }
        }
    }

    /**
     * Persists message for later re-usage by serializing it.
     */
    public function remember() {
        $fileName = sprintf('/%s%s.txt', self::STATUS_NOTIFICATION_MSG_FILE_PREFIX, $this->_usage);
        $filePath = Mage::getBaseDir().self::DIR_FILECACHE.$fileName;
        $data = serialize($this);
        if (!file_put_contents($filePath, $data)) {
            //Js_Log::log(sprintf('jSCoreHelper: Unable to write "%s". ', $filePath), '', \Zend_Log::ALERT);
        }
    }
}