<?php
/**
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_CoreHelpers
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 * @author      Hemendra Ajmera
 */

 namespace Xtwo\CoreHelpers\Model\Ftp;
 
use Magento\Framework\Model\AbstractModel;

class Client extends AbstractModel
{
    const DEFAULT_RETRIES = 3;
    const DEFAULT_DELAY_SEC = 2;
    const LOG_FILENAME = 'js-ftp';
    const DIR_FILECACHE = '/var/js-helpers/ftpfilecache';

    protected $_ftpConnection = null;
    protected $_isLoggedIn = false;
    public function __construct(
        \Magento\Quote\Model\Quote\ItemFactory $quoteItemFactory,
        \Magento\Quote\Model\ResourceModel\Quote\Item $itemResourceModel,
        \Xtwo\CoreHelpers\Model\Ftp\DirectoryFilter $directoryFilter,
        \Xtwo\CoreHelpers\Model\Ftp\FileFilter $fileFilter,
        \Xtwo\CoreHelpers\Model\Ftp\FileProxy $fileProxy
      ) {
         $this->quoteItemFactory = $quoteItemFactory;
         $this->itemResourceModel = $itemResourceModel;
         $this->directoryFilter = $directoryFilter;
         $this->fileFilter = $fileFilter;
         $this->fileProxy = $fileProxy;
      }
    /**
     * Initialisation.
     * @param Xtwo_CoreHelpers_Model_Ftp_Credentials $credentials
     * @throws Xtwo_CoreHelpers_Model_Ftp_Exception on any error
     * @return Xtwo_CoreHelpers_Model_Ftp_Client
     */
    public function init($credentials) {
        $this->setCredentials($credentials);
        $this->getFileCacheLocal();
        return $this;
    }

    /**
     * Uploads given file. At least FileProxy localFilePath must be set. If you want to put it remote to a
     * specific path, set remoteFilePath as well.
     * The returned FileProxy::getIsUploaded() returns the result.
     * @param Xtwo_CoreHelpers_Model_Ftp_FileProxy $fileProxy
     * @return Xtwo_CoreHelpers_Model_Ftp_FileProxy
     */
    public function uploadFile($fileProxy) {
        if ($this->connect() && $this->login()) {
            if (file_exists($fileProxy->getLocalFilePath())) {

                $remote = $fileProxy->getRemoteFilePath() ? $fileProxy->getRemoteFilePath() :
                            '/'.$fileProxy->getLocalFileName();
                $isSuccess = ftp_put($this->_ftpConnection, $remote, $fileProxy->getLocalFilePath(), FTP_BINARY);
                if ($isSuccess) {
                    $fileProxy->setIsUploaded(true);
                    //Js_Log::log(sprintf('Uploading file succeed, local=%s, remote=%s',
                    //                    $fileProxy->getLocalFilePath(), $remote),
                    //                    self::LOG_FILENAME, \Zend_Log::INFO);
                } else {
                    //Js_Log::log(sprintf('Uploading file failed as ftp_put returns false for local=%s, remote=%s',
                    //    $fileProxy->getLocalFilePath(), $remote),
                    //    self::LOG_FILENAME, \Zend_Log::ERR);
                }
            } else {
                $fileProxy->setProblemReason(sprintf('file does not exist: %s', $fileProxy->getLocalFilePath()));
                //Js_Log::log(sprintf('Uploading file failed as local file path is not specified'),
                //    self::LOG_FILENAME, \Zend_Log::ERR);
            }
        }
        return $fileProxy;
    }

    /**
     * After processing you need to check <br/>
     * Xtwo_CoreHelpers_Model_Ftp_FileProxy::getIsDownloaded()<br/>
     * whether the download succeed or not.
     * @param Xtwo_CoreHelpers_Model_Ftp_FileProxy $fileProxy
     * @return Xtwo_CoreHelpers_Model_Ftp_FileProxy
     */
    public function download($fileProxy) {
        if ($this->connect() && $this->login()) {
            $localPath = $fileProxy->getLocalFilePath() ? $fileProxy->getLocalFilePath() :
                            Mage::getBaseDir().self::DIR_FILECACHE.dirname($fileProxy->getRemotePath());
            $pathReady = true;
            if (!file_exists($localPath)) {
                 $pathReady = mkdir($localPath, 0777, true);
            }
            if ($pathReady) {
                $fileProxy->setLocalFilePath($localPath.'/'.$fileProxy->getRemoteFileName());

                $isSuccess = ftp_get($this->_ftpConnection, $fileProxy->getLocalFilePath(), $fileProxy->getRemoteFilePath(), FTP_BINARY);
                if ($isSuccess) {
                    $fileProxy->setIsDownloaded(true);
                    //Js_Log::log(sprintf('Download "%s" to "%s" success!',
                    //    $fileProxy->getRemoteFilePath(), $fileProxy->getLocalFilePath()),
                    //    self::LOG_FILENAME, \Zend_Log::DEBUG);
                } else {
                    $errorData = error_get_last();
                    //Js_Log::log(sprintf('Download "%s" to "%s" failed: %s',
                    //    $fileProxy->getRemoteFilePath(), $fileProxy->getLocalFilePath(), $errorData['message']),
                    //    self::LOG_FILENAME, \Zend_Log::ERR);
                }
            } else {
                //Js_Log::log(sprintf('Download "%s" failed as cannot create local path "%s"',
                //    $fileProxy->getRemoteFilePath(), $fileProxy->getLocalFilePath()),
                //    self::LOG_FILENAME, \Zend_Log::ERR);
            }
        }
        return $fileProxy;
    }

    /**
     * Can be used to get a specific file (with known name) from a specific folder.
     * @param Xtwo_CoreHelpers_Model_Ftp_DirectoryFilter $directoryFilter
     * @param Xtwo_CoreHelpers_Model_Ftp_FileFilter $fileFilter
     * @throws Xtwo_CoreHelpers_Model_Ftp_Exception
     * @return bool|Xtwo_CoreHelpers_Model_Ftp_FileProxy false if directory of file not found
    public function getFile($directoryFilter, $fileFilter) {
        if ($this->connect() && $this->login()) {
            if ($this->changeDir($directoryFilter)) {

            } else {
                return false;
            }
        }
    }
     */

    /**
     * @param Xtwo_CoreHelpers_Model_Ftp_DirectoryFilter $directoryFilter
     * @param Xtwo_CoreHelpers_Model_Ftp_FileFilter $fileFilter
     * @return bool|array false if folder doesn't exists. Empty array if no file found, of file is a directory
     * @throws Xtwo_CoreHelpers_Model_Ftp_Exception
    public function getFileProxies($directoryFilter, $fileFilter) {
        $fileProxies = array();
        if ($this->connect() && $this->login()) {
            if ($this->changeDir($directoryFilter)) {
                if ($currentDir = ftp_pwd($this->_ftpConnection)) {
                    if ($dirContents = ftp_nlist($this->_ftpConnection, '.')) {
                        foreach ($dirContents as $fileName) {
                            if ($fileFilter->matches($fileName)) {
                                $localFile = $this->getFileCacheLocal().$fileName;
                                $isSuccess = ftp_get($this->_ftpConnection, $localFile, $fileName, FTP_ASCII);
                                if ($isSuccess) {
                                    $fileProxy = Mage::getModel('jscorehelpers/ftp_fileProxy')
                                                    ->setLocalFilePath($localFile)
                                                    ->setRemoteFileName($fileName)
                                                    ->setRemoteFilePath($currentDir.$fileName)
                                                    ->setIsDownloaded(true);
                                    $fileProxies[] = $fileProxy;
                                } else {
                                    Js_Log::log(sprintf('Download failed for %s to (local) %s', $fileName, $localFile),
                                        self::LOG_FILENAME, Zend_Log::ERR);
                                }
                                if (count($fileProxies) >= $fileFilter->getCountLimit()) {
                                    break;
                                }
                            }
                        }
                    }                    
                } else return false; // Error ignored
            } else return false; // Error ignored
        }
    }
     */


    /**
     * Returns an array of Xtwo_CoreHelpers_Model_Ftp_FileProxy elements. The FileProxy will have a remote file
     * name and a remote file path.
     * @param Xtwo_CoreHelpers_Model_Ftp_DirectoryFilter $directoryFilter
     * @param Xtwo_CoreHelpers_Model_Ftp_FileFilter $fileFilter (optional)
     * @return bool|array
     */
    public function listFiles($directoryFilter, $fileFilter=null) {
        if (is_null($fileFilter)) $fileFilter = $this->fileFilter;
        $fileProxies = array();
        if ($this->connect() && $this->login()) {
            if ($this->changeDir($directoryFilter)) {
                if ($currentDir = ftp_pwd($this->_ftpConnection)) {
                    if ($dirContents = ftp_nlist($this->_ftpConnection, '.')) {
                        foreach ($dirContents as $fileName) {
                            $lastModifiedTimestamp = ftp_mdtm($this->_ftpConnection, $fileName);

                            if ($fileFilter->isNameReleased($fileName) &&
                                $fileFilter->isAgeReleased($lastModifiedTimestamp)) {

                                //$fileProxy = Mage::getModel('jscorehelpers/ftp_fileProxy')
                                $fileProxy=$this->fileProxy
                                    ->setRemoteFileName($fileName)
                                    ->setRemoteFilePath($currentDir .'/'. $fileName)
                                    ->setIsDownloaded(false)
                                    ->setLastModifiedTimestamp($lastModifiedTimestamp)
                                    ->setRemotePath($currentDir);

                                $fileProxies[] = $fileProxy;
                            }
                            if (count($fileProxies) >= $directoryFilter->getCountLimit()) {
                                break;
                            }
                        }
                    }
                } else return false; // Error ignored
            } else return false; // Error ignored
        }
        return $fileProxies;
    }

    /**
     * Performs tests and throws exception on failure.
     * @throws Xtwo_CoreHelpers_Model_Ftp_Exception
     */
    public function test() {
        if ($this->connect() && $this->login()) {
            $directoryFilter = $this->directoryFilter->setPath(
                $this->getCredentials()->getRemoteBasePath()
            );
            if ($this->changeDir($directoryFilter)) {
                $testFile = __DIR__.'/accesstest.txt';
                if (file_exists($testFile)) {
                    if (!ftp_put($this->_ftpConnection, 'accesstest.txt', $testFile, FTP_ASCII)) {
                        throw new \Xtwo\CoreHelpers\Model\Ftp\Exception(sprintf('Cannot upload test file (into root)!'));
                    }
                    ftp_delete($this->_ftpConnection, 'accesstest.txt'); // ignore result
                } else {
                    throw new \Xtwo\CoreHelpers\Model\Ftp\Exception(sprintf('Access test not possible as of missing test file!'));
                }
            }
        }
    }

    /**
     * Checks whether file exists or not. May return file size, depending on option $returnFileSize parameter.
     * @param string|Xtwo_CoreHelpers_Model_Ftp_FileProxy $target
     * @param bool $returnFileSize if you want file size it returns it, but may be zero!
     * @return bool|int
     */
    public function fileExists($target, $returnFileSize=false) {
        $path = is_object($target) ? $target->getRemoteFilePath() : $target;
        if (strlen($path) && $path != '/') {
            if ($this->connect() && $this->login()) {
                $size = ftp_size($this->_ftpConnection, $path);
                if ($size < 0) {
                    return false;
                }
                return $returnFileSize ? $size : true;
            }
        }
        return false;
    }

    /**
     * @param $target
     * @return bool
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function fileDelete($target) {
        $path = is_object($target) ? $target->getRemoteFilePath() : $target;
        if (strlen($path) && $path != '/') {
            if ($this->connect() && $this->login()) {
                $isSuccess = ftp_delete($this->_ftpConnection, $path);
                return $isSuccess;
            }
        }
        return false;
    }

    /**
     * Checks whether folder exists or not.
     * @param string|Xtwo_CoreHelpers_Model_Ftp_FileProxy $target
     * @return bool|int
     */
    public function folderExists($target) {
        $path = is_object($target) ? $target->getRemoteFilePath() : $target;
        if ($this->connect() && $this->login()) {
            if (strlen($path) && $path != '/') {
                return ftp_chdir($this->_ftpConnection, $path);
            }
            return true;
        }
    }
    /**
     * @param string|Xtwo_CoreHelpers_Model_Ftp_FileProxy $src
     * @param string|Xtwo_CoreHelpers_Model_Ftp_FileProxy $target
     * @return bool
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function rename($src, $target) {
        $srcPath = is_object($src) ? $src->getRemoteFilePath() : $src;
        $targetPath = is_object($target) ? $target->getRemoteFilePath() : $target;
        if (strlen($srcPath) && $srcPath != '/' && strlen($targetPath)) {
            if ($this->connect() && $this->login()) {
                if (!ftp_rename($this->_ftpConnection, $srcPath, $targetPath)) {
                    throw new \Xtwo\CoreHelpers\Model\Ftp\Exception(
                        sprintf('Unable moving "%s" to "%s"', $srcPath, $targetPath));
                }
                return true;
            }
        }
        return true;
    }

    /**
     * @param string|Xtwo_CoreHelpers_Model_Ftp_FileProxy $target
     * @return bool
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function makeDir($target) {
        $path = is_object($target) ? $target->getRemoteFilePath() : $target;
        if (strlen($path) && $path != '/') {
            if ($this->connect() && $this->login()) {
                if (!ftp_mkdir($this->_ftpConnection, $path)) {
                    throw new \Xtwo\CoreHelpers\Model\Ftp\Exception(
                        sprintf('Unable to create dir "%s"', $path));
                }
                return true;
            }
        }
        return false;
    }

    /**
     * @param Xtwo_CoreHelpers_Model_Ftp_DirectoryFilter $directoryFilter
     * @throws \Magento\Framework\Exception\LocalizedException
     * @return bool true on success, false otherwise
     */
    protected function changeDir($directoryFilter) {
        $currentDir = ftp_pwd($this->_ftpConnection);
        if ($currentDir === false) {
            throw new \Xtwo\CoreHelpers\Model\Ftp\Exception(sprintf('Unable to get current directory!'));
        }
        if ($currentDir != $directoryFilter->getPath()) {
            $status = true;
            if ($currentDir != '/') {
                $status = ftp_chdir($this->_ftpConnection, '/');
            }
            if ($status) {
                $value = ftp_chdir($this->_ftpConnection, $directoryFilter->getPath());
                if (!$value) {
                    $msg = error_get_last();
                    throw new \Xtwo\CoreHelpers\Model\Ftp\Exception(
                            sprintf('Unable to change to directory "%s" with msg "%s"',
                                $directoryFilter->getPath(), $msg['message']));
                }
                return $value;
            } else {
                // TODO exception
            }
        }
        return true;
    }

    /**
     * @return bool true on success
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function connect() {
        if ($this->_ftpConnection) {
            return true;
        }
        if (!$this->getCredentials()) {
            throw new \Xtwo\CoreHelpers\Model\Ftp\Exception('No ftp credentials set!');
        }
        $credentials = $this->getCredentials();
        for($i=1; $i<=$this->getRetries(); $i++) {
            $this->_ftpConnection = ftp_connect($credentials->getHost(), $credentials->getPort(), $credentials->getConnectTimeout());
            if (!$this->_ftpConnection) {
                $sec = self::DEFAULT_DELAY_SEC * $i;
                sleep($sec);
                //Js_Log::log(sprintf('Retry %s after %s sec to ftp connect to %s'),
                //    $i+1, $sec, $credentials->getHost(), self::LOG_FILENAME, \Zend_Log::WARN);
            } else break;
        }
        if (!$this->_ftpConnection) {
            throw new \Xtwo\CoreHelpers\Model\Ftp\Exception(
                sprintf('Unable to connect "%s:%s" even after %s retries',
                    $credentials->getHost(), $credentials->getPort(), $this->getRetries()));
        }
        return true;
    }

    /**
     * Disconnects
     */
    public function disconnect() {
        if ($this->_ftpConnection) {
            if (!ftp_close($this->_ftpConnection)) {
                //Js_Log::log(sprintf('Close connection to "%s" failed.', $this->getCredentials()->getHost()),
                //    self::LOG_FILENAME, \Zend_Log::WARN);
            } else {
                //Js_Log::log(sprintf('Successfully disconnected from "%s"', $this->getCredentials()->getHost()),
                //    self::LOG_FILENAME, \Zend_Log::INFO);
            }
        }
        $this->_isLoggedIn = false;
    }

    /**
     * @return bool true on success
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function login() {
        if ($this->connect() && !$this->_isLoggedIn) {
            $credentials = $this->getCredentials();
            if (!ftp_login($this->_ftpConnection, $credentials->getUsername(), $credentials->getPassword())) {
                throw new \Xtwo\CoreHelpers\Model\Ftp\Exception(
                    sprintf('Unable to login "%s:%s" with user %s',
                        $credentials->getHost(), $credentials->getPort(), $credentials->getUsername()));
            } else {
                $this->_isLoggedIn = true;
                //Js_Log::log(sprintf('FTP: Successfully connected to "%s" with user "%s".',
                //        $credentials->getHost(), $credentials->getUsername()),
                //    self::LOG_FILENAME, \Zend_Log::INFO);
            }
            $this->afterLogin();
        }
        return true;
    }

    /**
     * Executes further things after login
     * @throws Xtwo_CoreHelpers_Model_Ftp_Exception
     */
    protected function afterLogin() {
        $credentials = $this->getCredentials();
        if ($credentials->getPassiveMode()) {
            if (!ftp_pasv($this->_ftpConnection, true)) {
                throw new \Xtwo\CoreHelpers\Model\Ftp\Exception(
                    sprintf('Cannot set passive mode on "%s:%s"',
                        $credentials->getHost(), $credentials->getPort()));
            }
        }
    }

    /** @return int */
    protected function getRetries() {
        $value = $this->getCredentials()->getConnectRetries();
        return $value ? $value : self::DEFAULT_RETRIES;
    }

    /**
     * @return string folder for file cache
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function getFileCacheLocal() {
        $base = Mage::getBaseDir().self::DIR_FILECACHE;
        if (!file_exists($base)) {
            if (!mkdir($base, 0777, true)) {
                throw new \Xtwo\CoreHelpers\Model\Ftp\Exception('Unable to create folder '.self::DIR_FILECACHE);
            }
        }
        return $base;
    }
}