<?php
/**
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_Automationshell
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 */

namespace Xtwo\Catalog\Plugin;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product as ModelProduct;
use Magento\Catalog\Model\Session;
use Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\Configurable;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\App\Response\Http;
use Magento\Framework\DataObject;
use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Catalog\Helper\Product as ProductHelper;
use Magento\Catalog\Model\Product\Type as ModelProductType;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Framework\Registry;
use Magento\Store\Model\StoreManagerInterface;

class CatalogHelperProductPlugin
{
    /**
     * @var \Magento\Catalog\Api\ProductRepositoryInterface
     */
    private ProductRepositoryInterface $productRepository;
    /**
     * @var \Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\Configurable
     */
    private Configurable $catalogProductTypeConfigurable;
    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    private StoreManagerInterface $storeManager;
    /**
     * @var \Magento\Framework\App\Response\Http
     */
    private Http $redirect;
    /**
     * @var \Magento\Framework\Registry
     */
    private Registry $registry;
    /**
     * @var \Magento\Framework\App\RequestInterface
     */
    private RequestInterface $request;
    /**
     * @var \Magento\Framework\Event\ManagerInterface
     */
    private ManagerInterface $eventManager;
    /**
     * @var \Magento\Catalog\Model\Session
     */
    private Session $catalogSession;
    /**
     * @var \Magento\Catalog\Api\CategoryRepositoryInterface
     */
    private CategoryRepositoryInterface $categoryRepository;

    /**
     * CatalogHelperProductPlugin constructor.
     *
     * @param \Magento\Catalog\Api\ProductRepositoryInterface $productRepository
     * @param \Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\Configurable $catalogProductTypeConfigurable
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param \Magento\Framework\App\Response\Http $redirect
     * @param \Magento\Framework\Registry $registry
     * @param \Magento\Framework\App\RequestInterface $request
     * @param \Magento\Framework\Event\ManagerInterface $eventManager
     * @param \Magento\Catalog\Model\Session $catalogSession
     * @param \Magento\Catalog\Api\CategoryRepositoryInterface $categoryRepository
     */
    public function __construct(
        ProductRepositoryInterface $productRepository,
        Configurable $catalogProductTypeConfigurable,
        StoreManagerInterface $storeManager,
        Http $redirect,
        Registry $registry,
        RequestInterface $request,
        ManagerInterface $eventManager,
        Session $catalogSession,
        CategoryRepositoryInterface $categoryRepository
    ) {
        $this->productRepository = $productRepository;
        $this->catalogProductTypeConfigurable = $catalogProductTypeConfigurable;
        $this->storeManager = $storeManager;
        $this->redirect = $redirect;
        $this->registry = $registry;
        $this->request = $request;
        $this->eventManager = $eventManager;
        $this->catalogSession = $catalogSession;
        $this->categoryRepository = $categoryRepository;
    }

    /**
     * Inits product to be used for product controller actions and layouts
     * $params can have following data:
     *   'category_id' - id of category to check and append to product as current.
     *     If empty (except FALSE) - will be guessed (e.g. from last visited) to load as current.
     *
     * @param int $productId
     * @param \Magento\Framework\App\Action\Action $controller
     * @param \Magento\Framework\DataObject $params
     *
     * @return bool|ModelProduct
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     */
    public function aroundInitProduct(
        ProductHelper $subject,
        callable $proceed,
        $productId,
        $controller,
        $params = null
    ) {
        // Prepare data for routine
        if (!$params) {
            $params = new DataObject();
        }

        // Init and load product
        $this->eventManager->dispatch(
            'catalog_controller_product_init_before',
            ['controller_action' => $controller, 'params' => $params]
        );

        if (!$productId) {
            return false;
        }

        $product = $this->getProduct($productId);

        if (!$subject->canShow($product)) {
            return false;
        }
        if (!in_array($this->storeManager->getStore()->getWebsiteId(), $product->getWebsiteIds())) {
            return false;
        }

        // Load product current category
        $categoryId = $params->getCategoryId();
        if (!$categoryId && $categoryId !== false) {
            $lastId = $this->catalogSession->getLastVisitedCategoryId();
            if ($product->canBeShowInCategory($lastId)) {
                $categoryId = $lastId;
            }
        } elseif (!$product->canBeShowInCategory($categoryId)) {
            $categoryId = null;
        }

        if ($categoryId) {
            try {
                $category = $this->categoryRepository->get($categoryId);
            } catch (NoSuchEntityException $e) {
                $category = null;
            }
            if ($category) {
                $product->setCategory($category);
                $this->registry->register('current_category', $category);
            }
        }

        // Register current data and dispatch final events
        $this->registry->register('current_product', $product);
        $this->registry->register('product', $product);

        try {
            $this->eventManager->dispatch(
                'catalog_controller_product_init_after',
                ['product' => $product, 'controller_action' => $controller]
            );
        } catch (LocalizedException $e) {
            $this->_logger->critical($e);

            return false;
        }

        return $product;
    }

    /**
     * @param int $productId
     *
     * @return false|\Magento\Catalog\Api\Data\ProductInterface
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function getProduct(int $productId)
    {
        try {
            $product = $this->productRepository->getById($productId, false, $this->storeManager->getStore()->getId());
        } catch (NoSuchEntityException $e) {
            return false;
        }

        if ($product->getTypeId() == ModelProductType::TYPE_SIMPLE) {
            $parentByChild = $this->catalogProductTypeConfigurable->getParentIdsByChild($product->getId());
            if (isset($parentByChild[0])) {

                //@TODO
                $configurableId = $parentByChild[0];
                $configProduct = $this->productRepository->getById(
                    $configurableId,
                    false,
                    $this->storeManager->getStore()->getId()
                );
                $this->registry->register('simple_product', $product);

                return $configProduct;
            }
        }

        return $product;
    }
}
