<?php

namespace Xtwo\Catalog\Console\Command;

use Magento\Eav\Model\ResourceModel\Entity\Attribute\CollectionFactory;
use Magento\Framework\Api\FilterBuilder;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Exception;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Eav\Api\Data\AttributeGroupSearchResultsInterface;
use Magento\Catalog\Api\ProductAttributeGroupRepositoryInterface;
use Magento\Catalog\Api\AttributeSetRepositoryInterface;
use Magento\Eav\Api\AttributeManagementInterface;
use Magento\Eav\Api\AttributeRepositoryInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Magento\Catalog\Model\Product as ProductModel;


class UpdateAttributeGroupsCommand extends Command
{
    static private array $attributesListGeneralUnderscore = [
        'manufacturer' => 2, //?
        'base_number' => 9,
        'series' => 3,
        'delivery_time' => 10,
        'barcode' => 7,
        'designer' => 8,
        'style' => 11,
        'grade' => 12,
        'advertisement_cms_block' => 1, //
        'order_confirmation' => 11,
        'googleshoppingcat' => 14,
        'type' => 4,
    ];

    static private $attributesListGeneral = [
        'description' => 2,
        'short_description' => 3,
        'url_key' => 8,
        'shipment_type' => 25,
        'default_configuration_id' => 11,
        'nav_id' => 12,
        'shopsync_active' => 35,
        'edited_by_user' => 14,
        'asin' => 4,
        'sup_stock' => 19,
        'sup_nav_id' => 13,
        'product_seo_name' => 15,
        'amazon_price' => 16,
        'layer_images' => 17,
        'upsell_cms_block' => 18,
        'promotion_text' => 18,
        'type2' => 5,
    ];

    static private $attributesMultiple = [
        'note',
        'technologies',
        'equipment',
        'scope_of_delivery',
        'spray_mode_overhead_shower',
        'spray_mode_hand_shower',
        'technologies_head_shower',
        'technologies_hand_shower',
        'technologies_thermostat',
        'technologies_side_showers',
        'technologies_singlelever_mixer',
        'product_characteristic',
        'boiler_note',
        'filter_note',
        'kitchen_faucet_note',
        'cooler_note',
        'showroom',
    ];

    static private $attributesLimitedAttributeSets = [
        'promotext' => 2,
    ];

    static private $attributesSetsLimited = [
        'Dusche-Wannenarmaturen',
        'Duschsysteme',
    ];

    static private $attributesVisibleOnFrontHTMLAllowed = [
        'manufacturer',
        'color',
    ];
    /**
     * @var \Magento\Framework\Api\SearchCriteriaBuilder
     */
    private SearchCriteriaBuilder $searchCriteriaBuilder;
    /**
     * @var \Magento\Catalog\Api\ProductAttributeGroupRepositoryInterface
     */
    private ProductAttributeGroupRepositoryInterface $productAttributeGroup;
    /**
     * @var \Magento\Catalog\Api\AttributeSetRepositoryInterface
     */
    private AttributeSetRepositoryInterface $attributeSetRepository;
    /**
     * @var \Magento\Eav\Model\ResourceModel\Entity\Attribute\CollectionFactory
     */
    private CollectionFactory $attributeCollectionFactory;
    /**
     * @var \Magento\Eav\Api\AttributeRepositoryInterface
     */
    private AttributeRepositoryInterface $attributeRepository;
    /**
     * @var \Magento\Framework\Api\FilterBuilder
     */
    private FilterBuilder $filterBuilder;
    /**
     * @var \Magento\Eav\Api\AttributeManagementInterface
     */
    private AttributeManagementInterface $attributeManagement;

    /**
     * UpdateAttributeGroupsCommand constructor.
     *
     * @param \Magento\Catalog\Api\ProductAttributeGroupRepositoryInterface $productAttributeGroup
     * @param \Magento\Framework\Api\SearchCriteriaBuilder $searchCriteriaBuilder
     * @param \Magento\Catalog\Api\AttributeSetRepositoryInterface $attributeSetRepository
     * @param \Magento\Eav\Model\ResourceModel\Entity\Attribute\CollectionFactory $attributeCollectionFactory
     * @param \Magento\Eav\Api\AttributeRepositoryInterface $attributeRepository
     * @param \Magento\Framework\Api\FilterBuilder $filterBuilder
     * @param \Magento\Eav\Api\AttributeManagementInterface $attributeManagement
     */
    public function __construct(
        ProductAttributeGroupRepositoryInterface $productAttributeGroup,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        AttributeSetRepositoryInterface $attributeSetRepository,
        CollectionFactory $attributeCollectionFactory,
        AttributeRepositoryInterface $attributeRepository,
        FilterBuilder $filterBuilder,
        AttributeManagementInterface $attributeManagement
    ) {
        parent::__construct();

        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->productAttributeGroup = $productAttributeGroup;
        $this->attributeSetRepository = $attributeSetRepository;
        $this->attributeCollectionFactory = $attributeCollectionFactory;
        $this->attributeRepository = $attributeRepository;
        $this->filterBuilder = $filterBuilder;
        $this->attributeManagement = $attributeManagement;
    }

    /**
     * @param \Symfony\Component\Console\Input\InputInterface $input
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     *
     * @return int|null
     * @throws \Exception
     */
    public function execute(InputInterface $input, OutputInterface $output)
    {
        $io = new SymfonyStyle($input, $output);

        try {
            $this->updateMultiselectAttributes();
            $this->updateIsVisibleOnFront();

            $this->searchCriteriaBuilder->setPageSize(100);
            $this->searchCriteriaBuilder->addFilter('entity_type_id', 4);
            $this->searchCriteriaBuilder->addFilter('attribute_set_name', 'Default', 'neq');

            $attributeSets = $this->attributeSetRepository->getList($this->searchCriteriaBuilder->create());
            $this->searchCriteriaBuilder->setPageSize(1000);
            $this->searchCriteriaBuilder->addFilters(
                [
                    $this->filterBuilder
                        ->setField('main_table.attribute_group_name')
                        ->setConditionType('in')
                        ->setValue(
                            ['Product Details', '_General', '_Color-Material-Surface', '_Colors-Coating-Material']
                        )
                        ->create(),
                ]
            );

            $productAttributeGroups = $this->productAttributeGroup->getList($this->searchCriteriaBuilder->create());

            $groupsToSetsGeneral = $groupsToSetsGeneralUnderscore = $groupsToSetsMaterialSurface = $groupsToSetsCoatingMaterial = [];
            foreach ($productAttributeGroups->getItems() as $attributeGroup) {
                if ($attributeGroup->getAttributeGroupName() == 'Product Details') {
                    $groupsToSetsGeneral[$attributeGroup->getAttributeSetId()] = $attributeGroup->getId();
                } elseif ($attributeGroup->getAttributeGroupName() == '_General') {
                    $groupsToSetsGeneralUnderscore[$attributeGroup->getAttributeSetId()] = $attributeGroup->getId();
                } elseif ($attributeGroup->getAttributeGroupName() == '_Color-Material-Surface') {
                    $groupsToSetsMaterialSurface[] = [
                        'attribute_group_id' => $attributeGroup->getId(),
                        'attribute_set_id' => $attributeGroup->getAttributeSetId(),
                    ];
                } elseif ($attributeGroup->getAttributeGroupName() == '_Colors-Coating-Material') {
                    $groupsToSetsCoatingMaterial[] = [
                        'attribute_group_id' => $attributeGroup->getId(),
                        'attribute_set_id' => $attributeGroup->getAttributeSetId(),
                    ];
                }
            }

            $attributeSetsGroupsGeneral = $attributeSetsGroupsGeneralUnderscore = $attributeSetsGroupsLimited = [];

            foreach ($attributeSets->getItems() as $attributeSet) {

                if (isset($groupsToSetsGeneral[$attributeSet->getId()])) {
                    $attributeSetsGroupsGeneral[] = [
                        'attribute_set_id' => $attributeSet->getId(),
                        'attribute_group_id' => $groupsToSetsGeneral[$attributeSet->getId()],
                    ];
                }

                if (isset($groupsToSetsGeneralUnderscore[$attributeSet->getId()])) {
                    $attributeSetsGroupsGeneralUnderscore[] = [
                        'attribute_set_id' => $attributeSet->getId(),
                        'attribute_group_id' => $groupsToSetsGeneralUnderscore[$attributeSet->getId()],
                    ];
                    if (in_array($attributeSet->getAttributeSetName(), self::$attributesSetsLimited)) {
                        $attributeSetsGroupsLimited[] = [
                            'attribute_set_id' => $attributeSet->getId(),
                            'attribute_group_id' => $groupsToSetsGeneralUnderscore[$attributeSet->getId()],
                        ];
                    }

                }
            }
            $this->assignAttributes($attributeSetsGroupsGeneral, self::$attributesListGeneral);
            $this->assignAttributes($attributeSetsGroupsGeneralUnderscore, self::$attributesListGeneralUnderscore);
            $this->assignAttributes($attributeSetsGroupsLimited, self::$attributesLimitedAttributeSets);
            $this->assignColorAttribute($groupsToSetsMaterialSurface);
            $this->assignColorAttribute($groupsToSetsCoatingMaterial);

        } catch (Exception $exception) {
            $io->error("There were problems during adding attributes to attribute groups or by attribute updates");
            throw new Exception($exception->getMessage());

            return 0;
        }

        $io->success('The attributes were successfully assigned and updated.');

        return 0;
    }

    /**
     *
     */
    protected function updateMultiselectAttributes()
    {
        $this->searchCriteriaBuilder->setPageSize(100);
        $this->searchCriteriaBuilder->addFilter('attribute_code', self::$attributesMultiple, 'in');

        $attributes = $this->attributeRepository->getList(
            ProductModel::ENTITY,
            $this->searchCriteriaBuilder->create()
        );

        foreach ($attributes->getItems() as $attribute) {
            $attribute->setBackendType('text');
            $attribute->save();
        }
    }

    /**
     *
     */
    protected function updateIsVisibleOnFront()
    {
        $this->searchCriteriaBuilder->setPageSize(100);
        $this->searchCriteriaBuilder->addFilter('attribute_code', self::$attributesVisibleOnFrontHTMLAllowed, 'in');

        $attributes = $this->attributeRepository->getList(
            ProductModel::ENTITY,
            $this->searchCriteriaBuilder->create()
        );

        foreach ($attributes->getItems() as $attribute) {
            $attribute->setIsVisibleOnFront(1);
            $attribute->setIsHtmlAllowedOnFront(1);
            $attribute->save();
        }
    }

    /**
     * @param array $attributeSetsGroups
     * @param array $attributesList
     *
     * @throws \Magento\Framework\Exception\InputException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function assignAttributes($attributeSetsGroups, $attributesList)
    {
        foreach ($attributeSetsGroups as $attributeSetsGroup) {
            foreach (array_keys($attributesList) as $attributeCode) {

                $this->attributeManagement->assign(
                    ProductModel::ENTITY,
                    $attributeSetsGroup['attribute_set_id'],
                    $attributeSetsGroup['attribute_group_id'],
                    $attributeCode,
                    $attributesList[$attributeCode] //sort order
                );
            }
        }
    }

    /**
     * @param array $attributeSetsGroups
     *
     * @throws \Magento\Framework\Exception\InputException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    protected function assignColorAttribute(array $attributeSetsGroups)
    {
        foreach ($attributeSetsGroups as $attributeSetsGroup) {

            $this->attributeManagement->assign(
                ProductModel::ENTITY,
                $attributeSetsGroup['attribute_set_id'],
                $attributeSetsGroup['attribute_group_id'],
                'color',
                2 //sort order
            );

        }
    }

    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this->setName('xtwo:catalog:update-attribute-groups-service');
        $this->setDescription('Update Attribute Groups service.');

        $helpText = <<<TEXT
 Usage: xtwo:catalog:update-attribute-groups-service

TEXT;
        $this->addUsage($helpText);

        parent::configure();
    }

}
