<?php
/**
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_Automationshell
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 */

namespace Xtwo\Automationshell\Console;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Filesystem\Exception\FileNotFoundException;



/**
 * Command for executing cron jobs
 */

class LinkedProductCommand extends Command
{
    /** SKU parameter code */
    const PARAM_SKU = 'sku';
    const PARAM_SKU_LINKED = 'link';
    const PARAM_SKIP_BAN = 'skipvarnish';

    /** Action: delete linked  */
    const PARAM_ACTION_DELETE = 'delete';
    const PARAM_ACTION_ADD = 'add';
    const PARAM_ACTION_INFO = 'info';

    /** @var string @see $supportedLinkTypes */
    const PARAM_LINK_TYPE = 'type';

    const DEFAULT_TYPE = 'relation';

    /** @var array Currently supported product-link types */
    private static $supportedLinkTypes = ['relation' => 1, 'upsell' => 4, 'cross_sell' => 5,
                                          'series_sell' => 6, 'alternative_sell' => 7, 'spare_sell' => 8, 'sample_sell' => 9];

    /** @var array Id's fo the position linkage of each link-type in table 'catalog_product_link_attribute_int' */
    private static $productLinkAttributeId = ['relation' => 1, 'upsell' => 4, 'cross_sell' => 5,
                                              'series_sell' => 6, 'alternative_sell' => 7, 'spare_sell' => 8, 'sample_sell' => 9];

    private static $storeCodesForVarnishBan = ['xtwostore_de', 'xtwostore_com', 'xtwostore_fr', 'xtwostore_cn',
        'xtwostore_at', 'xtwostore_couk', 'xtwostore_ch'];

    /** File to log messages to */
    const LOGFILE = 'js_linkedproducts.log';

    /** Max. limit products for varnish ban */
    const VARNISH_BAN_MAX_LIMIT = 200;

    /**
     * Product model
     *
     * @var \Magento\Catalog\Model\Product
     */
    protected $productModel;

    /**
     * Varnish ban model
     *
     * @var Nexcessnet_Turpentine_Model_Observer_Ban
     */
    protected $varnish;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var \Magento\Framework\DataObjectFactory
     */
    protected $dataObjectFactory;

    /**
     * @var 
     */
  
     public function __construct(        
        \Magento\Eav\Model\Entity\TypeFactory $eavEntityTypeFactory,
        \Magento\Eav\Model\Entity\Attribute\SetFactory $eavEntityAttributeSetFactory,
        \Magento\Eav\Model\Entity\AttributeFactory $eavEntityAttributeFactory,
        \Psr\Log\LoggerInterface $logger,       
        \Magento\Eav\Model\Entity $entity,
        \Magento\Framework\ObjectManagerInterface $objectManager,
        \Magento\Framework\App\State $state,       
        \Magento\Store\Model\Store $store,
        \Magento\Catalog\Model\ResourceModel\Product\CollectionFactory $productCollection,  
        \Magento\Catalog\Model\ProductFactory $product,
        \Magento\Framework\App\ResourceConnection $resourceConnection,
        \Magento\Catalog\Model\ResourceModel\Product\Action $productAction,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Eav\Model\ResourceModel\Entity\Attribute\Group\CollectionFactory $attributeGroupCollection,
        \Magento\Catalog\Model\ResourceModel\Product\Attribute\CollectionFactory $productAttributeCollection
    ) {

      
        $this->logger = $logger;      
        $this->state = $state;       
        $this->store = $store;
        $this->productCollection = $productCollection;       
        $this->resourceConnection = $resourceConnection;
        $this->connection = $this->resourceConnection->getConnection();      
        $this->productModel = $product;
        $this->storeManager = $storeManager;
          
       // $this->checkRequiredFields();
        
        parent::__construct();
    }




    protected function configure()
    {
        $this->setName('Xtwo:LinkedProductCommand')
            ->setDescription('Commands:LinkedProductCommand')
            ->addArgument(
                'actions',
                InputArgument::REQUIRED,
                'add|delete|info'
            )       
            ->addOption('sku', "sku", InputOption::VALUE_OPTIONAL, "--sku List of SKUs to link products to or to delete all the linked products from.")
            ->addOption('link', "link", InputOption::VALUE_OPTIONAL, "--link Store code. Example: ytwo_xtwostore_de")
            ->addOption('type', "type", InputOption::VALUE_OPTIONAL, "--types relation|updsell|cross_sell|series_sell|alternative_sell|spare_sell|sample_sell] Specify type of product linkage. Default: relation")
            ->addOption('skipvarnish', "skipvarnish", InputOption::VALUE_OPTIONAL, "--skipvarnish If you don't want to ban the products on varnish. Default: false");
            parent::configure();
    }

    



    /**
     * @param string $msg
     */
    protected function log($msg) {
        $this->logger->log('100',$msg);
    }

    /**
     * Echo error and log to file
     *
     * @param string $msg
     * @param bool   $isIntro
     * @param bool   $skipLineBreak
     */
    protected function echoAndLog($msg, $isIntro = false, $skipLineBreak = false)
    {
        $echo = ($isIntro) ? $msg : '   ' . $msg;
        $lb = $skipLineBreak ? '' : "\n";
        echo $echo . $lb;
        $this->log($echo);
    }

    /**
     * Init function
     */
    protected function init()
    {
       // $this->productModel = Mage::getSingleton('catalog/product');
       $this->productModel =  $this->productModel->create();
       //  $this->varnish = Mage::getSingleton('turpentine/observer_ban');
    }

    /**
     * Check required fields existence
     * @throws \Exception
     */
    protected function checkRequiredFields()
    {
        $this->getAction();
        $skuList = $this->getSkuList();
       // if (empty($skuList)) throw new InvalidInputException('No sku list given!');
    }

    /**
     *
     */
    protected function checkProducts() {
        {
            $skuList = $this->getSkuList();
            $productsCollection = $this->getProducts($skuList);
            $finalSkuList = $productsCollection->getColumnValues('sku');
            $diff = array_diff($skuList, $finalSkuList);
            if (!empty($diff)) {
                $this->echoAndLog(sprintf('Waring! Products not found: %s', implode(',', $diff)));
            }
        }
        {
            $skuList = $this->getLinkedSkuList();
            if (!empty($skuList)) {
                $productsCollection = $this->getProducts($skuList);
                $finalSkuList = $productsCollection->getColumnValues('sku');
                $diff = array_diff($skuList, $finalSkuList);
                if (!empty($diff)) {
                    $this->echoAndLog(sprintf('Waring! Linked Products not found: %s', implode(',', $diff)));
                }
            }
        }

    }

    /**
     * Get sku list from provided shell parameter
     *
     * @return array
     */
    protected function getSkuList()
    {
        $skuParamValue = $this->input->getOption(self::PARAM_SKU);
        $skuList = [];
        if ($skuParamValue) {
            $skuList = explode(',', trim($skuParamValue));
        }
        return $skuList;
    }

    /**
     * Get sku list from provided shell parameter
     *
     * @return array
     */
    protected function getLinkedSkuList()
    {
        $skuParamValue = $this->input->getOption(self::PARAM_SKU_LINKED);
        $skuList = [];
        if ($skuParamValue) {
            $skuList = explode(',', trim($skuParamValue));
        }
        return $skuList;
    }

    /**
     * Get sku list from provided shell parameter
     *
     * @return array
     * @throws \Exception
     */
    protected function getLinkType()
    {
        $linkType = $this->input->getOption(self::PARAM_LINK_TYPE);
        if (empty($linkType)) $linkType = self::DEFAULT_TYPE;
        $this->getLinkTypeId($linkType); // validate
        return $linkType;
    }

    /**
     * @return string
     * @throws \Exception
     */
    protected function getAction() {
        $actions = $this->input->getArgument('actions');
        if($actions == 'add' || $actions == 'delete' || $actions == 'info'){
           return $actions;

        }

        
       /* if (isset($this->input->getOption[self::PARAM_ACTION_ADD])) {
            return 'add';
        } elseif (isset($this->input->getOption[self::PARAM_ACTION_DELETE])) {
            return 'delete';
        } elseif (isset($this->input->getOption[self::PARAM_ACTION_INFO])) {
            return 'info';
        }*/

        throw new \Exception('Not supported or missing action.');
    }

    /**
     * @param $linkType
     * @return int
     * @throws \Exception
     */
    protected function getLinkTypeId($linkType) {
        if (!array_key_exists($linkType, self::$supportedLinkTypes)) throw new \Exception('Unsupported type of link: '.$linkType);
        return self::$supportedLinkTypes[$linkType];
    }

    /**
     * @param $skuList
     */
    protected function getInfo($skuList) {
        $resultData = [];
        /** @var \Magento\Framework\Db\Adapter\AdapterInterface $adapter */
        // $adapter = $this->productModel->getResource()->getReadConnection();
        $adapter = $this->connection->create();
        $products = $this->getProducts($skuList);
        foreach ($products as $product) {
            if ($product->getTypeId() == 'simple') {
                $select = $adapter->select()
                    ->from('catalog_product_super_link', 'parent_id')
                    ->where('product_id = :product_id');
                $binds = array(
                    'product_id' => $product->getId()
                );
                $result = $adapter->fetchCol($select, $binds);
                if (!empty($result)) {
                    $parentProducts = $this->getProductsById($result);
                    foreach ($parentProducts as $parentProduct) {
                        $resultData[$parentProduct->getSku()] = [
                            'childProducts' => [$product],
                            'parentProduct' => $parentProduct
                        ];
                    }
                } else {
                    $resultData[$product->getSku()] = 'Not assigned to any configurable.';
                }

            } elseif ($product->getTypeId() == 'configurable') {
                $select = $adapter->select()
                    ->from('catalog_product_super_link', 'product_id')
                    ->where('parent_id = :parent_id');
                $binds = array(
                    'parent_id' => $product->getId()
                );
                $result = $adapter->fetchCol($select, $binds);
                if (!empty($result)) {
                    $childProducts = $this->getProductsById($result)->getItems();
                    $resultData[$product->getSku()] = [
                        'childProducts' => $childProducts,
                        'parentProduct' => $product
                    ];
                } else {
                    $resultData[$product->getSku()] = 'Does not have any child.';
                }
            } else {
                $resultData[$product->getSku()] = 'Type not supported: '.$product->getTypeId();
            }
        }
        return $resultData;
    }

    /**
     * Deletes Links for the given type of the given list of SKU's.
     *
     * @param string[] $skuList
     * @param string $linkType
     *
     * @return array list of product ID's which have been added links to
     * @throws \Exception
     */
    protected function deleteLinksOfType($skuList, $linkType=self::DEFAULT_TYPE)
    {
        $realUpdates = null;
        /** @var \Magento\Framework\Db\Adapter\AdapterInterface $adapter */
        // $adapter = $this->productModel->getResource()->getWriteConnection();
        $adapter = $this->connection;

        $linkTypeId = $this->getLinkTypeId($linkType);
        $productIdList = $this->getProducts($skuList)->getAllIds();
        $productIds = implode(',', $productIdList);
        $realUpdates = $adapter->delete('catalog_product_link',
            sprintf("link_type_id = '%s' AND product_id IN(%s)", $linkTypeId, $productIds));

        $this->echoAndLog(sprintf('Deleted %s links of type "%s" for %s products successfully.',
            $realUpdates, $linkType, count($skuList)));

        return $productIdList;
    }

    /**
     * Deletes Links for the given type of the given list of SKU's.
     *
     * @param string[] $skuList
     * @param string $linkType
     * @param string[] $linkSkuList
     *
     * @throws \Exception
     * @return array list of product ID's which have been added links to
     */
    protected function addLinksOfType($skuList, $linkSkuList, $linkType=self::DEFAULT_TYPE)
    {
        $realUpdates = null;
        /** @var \Magento\Framework\Db\Adapter\AdapterInterface $adapter */
        // $adapter = $this->productModel->getResource()->getWriteConnection();
        // $adapter = $this->adapter->getResource()->getConnection();
        $adapter = $this->connection;

        $linkTypeId = $this->getLinkTypeId($linkType);
        $productIdList = $this->getProducts($skuList)->getAllIds();
        foreach ($productIdList as $productId) {
            $linkedProductsList = $this->getProductsInSkuOrder($linkSkuList);
            $total = 0;
            $sortOrder = 0;
            foreach ($linkedProductsList as $linkedProduct) {
                try {
                    // Linking
                    $inserted = 'not inserted';
                    try {
                        $insertCount = $adapter->insert('catalog_product_link',
                            ['product_id'        => $productId,
                             'linked_product_id' => $linkedProduct->getId(),
                             'link_type_id'      => $linkTypeId]);
                        if ($insertCount) $inserted = 'inserted';
                    } catch (Zend_Db_Exception $e) {
                        if (strpos($e->getMessage(), 'Duplicate entry') !== false) {
                            $inserted = 'already exists';
                        }
                    }
                    $this->echoAndLog(sprintf('Link %s: %s (to product#%s, type=%s/%s)',
                        $inserted, $linkedProduct->getSku(), $productId, $linkType, $linkTypeId));

                    // Positioning
                    $select = $adapter->select()
                        ->from('catalog_product_link', 'link_id')
                        ->where('product_id = :product_id')
                        ->where('linked_product_id = :linked_product_id')
                        ->where('link_type_id = :link_type_id');
                    $binds = array(
                        'product_id'        => (int)$productId,
                        'linked_product_id' => (int)$linkedProduct->getId(),
                        'link_type_id'      => (int)$linkTypeId
                    );
                    $link_id = $adapter->fetchOne($select, $binds);
                    $linkAttributeId = self::$productLinkAttributeId[$linkType];
                    try {
                        $result = $adapter->insert('catalog_product_link_attribute_int', [
                            'product_link_attribute_id' => $linkAttributeId,
                            'link_id'                   => $link_id,
                            'value'                     => $sortOrder
                        ]);
                    } catch (Zend_Db_Exception $e) {
                        if (strpos($e->getMessage(), 'Duplicate entry') !== false) {
                            // ok
                        } else {
                            $this->echoAndLog(sprintf('Unexpected problem: %s', $e->getMessage()));
                        }
                    }
                    $sortOrder += 10;
                    $total = $inserted;


                } catch (Zend_Db_Exception $e) {
                    $this->echoAndLog(sprintf('Unexpected problem: %s', $e->getMessage()));
                }
            }
            $this->echoAndLog(sprintf('Linked products found: %s of %s', count($linkedProductsList), count($linkSkuList)));
            if (empty($total)) {
                $this->echoAndLog(sprintf('Problem: could not insert any link of type "%s" for product %s (maybe already existing link).',
                    $linkType, $productId));
            } elseif ($total != count($linkedProductsList)) {
                $this->echoAndLog(sprintf('Problem: could insert only %s links from %s of type "%s" for product %s successfully (maybe already existing link).',
                    $total, count($linkedProductsList), $linkType, $productId));
            } else {
                $this->echoAndLog(sprintf('Inserted %s links of type "%s" for product %s successfully.',
                    $total, $linkType, $productId));
            }
        }
        return $productIdList;
    }


    /**
     * Ban varnish URLs for updated products
     *
     * @param $storeId
     * @param $productIds
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function banVarnishUrls($productIds, $storeId=[])
    {
        if (count($productIds) > self::VARNISH_BAN_MAX_LIMIT) {
            $this->echoAndLog(sprintf(' ... skipped as limit reached to ban varnish.'));
            return;
        }

        $this->echoAndLog(sprintf('Clearing varnish cache for %s products on %s stores',
            count($productIds), count(self::$storeCodesForVarnishBan)));

        foreach (self::$storeCodesForVarnishBan as $storeCode) {
            $store = $this->storeManager->getStore($storeCode);
            foreach ($productIds as $productId) {
                $product = $this->productModel->reset()->setStoreId($store->getId())->load($productId);
                $event = $this->dataObjectFactory->create();
                $this->echoAndLog(sprintf('Banning %s', $storeCode));
                $this->varnish->banProductPageCache($event->setProduct($product));
            }
        }
        $this->echoAndLog(sprintf(' ... done'));
    }

    /**
     * Run script
     */
       // public function run()
    protected function execute(InputInterface $input, OutputInterface $output)    
    {
       $this->input = $input;

       

        try {
            //init models, check required fields
            $this->init();

            $this->checkRequiredFields();

            $this->checkProducts();

            $finallyModifiedProductIds = [];
            $action = $this->getAction();
            switch ($action) {
                case 'delete':
                    {
                        $skuList = $this->getSkuList();
                        if (empty($skuList)) throw new InvalidInputException('No or empty SKU list specified!');
                        $linkType = $this->getLinkType();
                       

                        $this->echoAndLog(sprintf('Start deleting in 5 seconds...'));
                        sleep(5);
                        $finallyModifiedProductIds = $this->deleteLinksOfType($skuList, $linkType);
                        break;
                    }
                case 'add':
                    {
                        $skuList = $this->getSkuList();
                        if (empty($skuList))  throw new InvalidInputException('No or empty SKU list specified!');
                        $linkedSkuList = $this->getLinkedSkuList();
                        if (empty($linkedSkuList))  throw new InvalidInputException('No or empty SKU list for linked products specified!');
                        $linkType = $this->getLinkType();

                        $this->echoAndLog(sprintf('Start adding in 3 seconds...'));
                        sleep(3);
                        $finallyModifiedProductIds = $this->addLinksOfType($skuList, $linkedSkuList, $linkType);
                        break;
                    }
                case 'info':
                    {
                        $messages = [];
                        $skuList = $this->getSkuList();
                        if (empty($skuList)) // throw new InvalidInputException('No or empty SKU list specified!');
                        $result = $this->getInfo($skuList);
                        if (!empty($result)) {
                            $this->echoAndLog('SKU-Configurable,SKU-Child,Status-Child');
                            foreach ($result as $parentSku => $data) {
                                if (!is_array($data)) {
                                    $messages[$parentSku] = $data;
                                    continue;
                                }
                                foreach ($data['childProducts'] as $childProduct) {
                                    $this->echoAndLog(sprintf('%s,%s,%s',
                                        $parentSku, $childProduct->getSku(),
                                        $childProduct->getStatus() ? 'enabled':'disabled'));
                                }
                            }
                            if (!empty($messages)) {
                                $this->echoAndLog("\nNotices:\n");
                                foreach ($messages as $sku => $message) {
                                    $this->echoAndLog(sprintf('%s: %s', $sku, $message));
                                }
                            }
                            $this->echoAndLog("\nSuccessfully finished.");
                        } else {
                            $this->echoAndLog("No info available.");
                        }
                    }
            }

            //clear varnish
            if (!$this->input->getOption(self::PARAM_SKIP_BAN) && !empty($finallyModifiedProductIds)) {
              //  $this->banVarnishUrls($finallyModifiedProductIds);
            }

        } catch (InvalidInputException $e) {
            $this->echoAndLog(sprintf('Stopped processing, Reason: %s', $e->getMessage()));
            $this->echoAndLog("\n\n" . $this->usageHelp());

        } catch (Exception $e) {
            $this->echoAndLog(sprintf('Stopped processing, Reason Exception: %s', $e->getMessage()));
            $this->echoAndLog("\n\n" . $this->usageHelp());
        }
    }


    /**
     * Get product collection
     *
     * @param array $skuList
     * @return \Magento\Catalog\Model\ResourceModel\Product\Collection
     */
    protected function getProducts($skuList) {
        /** @var \Magento\Catalog\Model\ResourceModel\Product\Collection $collection */
      //  $collection = Mage::getResourceModel('catalog/product_collection');
        $collection = $this->productCollection->create();
        //if sku list provided, as as filter
        if (!empty($skuList)) {
            $collection->addAttributeToFilter('sku', array('in' => $skuList));
        }
        $collection->setOrder('entity_id', 'asc');
        return $collection;
    }

    /**
     * Returns an array of products ordered by the order of the given sku list.
     *
     * @param string[] $skuList
     * @return \Magento\Catalog\Model\Product[]
     */
    protected function getProductsInSkuOrder($skuList) {
        $productsOrdered = []; $products = [];
        $productsCollection = $this->getProducts($skuList);
        foreach ($productsCollection as $product) {
            $products[$product->getSku()] = $product;
        }
        foreach ($skuList as $sku) {
            if (array_key_exists($sku, $products) && !array_search($sku, $productsOrdered)) {
                $productsOrdered[] = $products[$sku];
            }
        }
        return $productsOrdered;
    }

    /**
     * Get product collection
     *
     * @param array $idList
     * @return \Magento\Catalog\Model\ResourceModel\Product\Collection
     */
    protected function getProductsById($idList) {
        /** @var \Magento\Catalog\Model\ResourceModel\Product\Collection $collection */
        $collection = Mage::getResourceModel('catalog/product_collection');
        //if sku list provided, as as filter
        if (!empty($idList)) {
            $collection->addIdFilter($idList);
        }
        $collection->setOrder('entity_id', 'asc');
        return $collection;
    }


    /**
     * Retrieve Usage Help Message
     *
     */
    public function usageHelp()
    {
        return <<<USAGE
Usage:  php bin/magento Xtwo:LinkedProductCommand add|delete [--options]

  add|delete|info           The action type 
  
  --sku <sku_list>          List of SKUs to link products to or to delete all the linked products from.
                            In context of 'info' it lists the configurable products this are linked with and enabled/disabled.
  
  [--link <sku_list>]       Required in case of 'add' action: the products to be linked to the onces specified via '--sku'
  
  [--type relation|updsell|cross_sell|series_sell|alternative_sell|spare_sell|sample_sell] 
                            Specify type of product linkage. Default: "relation"
  
  [--skipvarnish]           If you don't want to ban the products on varnish. Default: false

  help                      This help
  
  
  <sku_list>    List of product SKUs. Multiple ones separated by comma.

USAGE;
    }
}

/**
 * Class InvalidInputException
 * justselling GmbH EULA
 * http://www.justselling.de/
 * Read the license at http://www.justselling.de/lizenz
 *
 * Do not edit or add to this file, please refer to http://www.justselling.de for more information.
 *
 * @copyright   Copyright (c) 2018 justselling Germany Ltd. (http://www.justselling.de)
 * @license     http://www.justselling.de/lizenz
 * @author      Bodo Schulte
 * Date: 06.02.19
 * Time: 09:40
 */
// class InvalidInputException extends Exception { }

