<?php
/**
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_Automationshell
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 */

namespace Xtwo\Automationshell\Console;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Filesystem\Exception\FileNotFoundException;



/**
 * Command for executing cron jobs
 */

class  AttributesetCommand extends Command
{
    /** Target store parameter code */
    const PARAM_TARGET_STORE = 'store';

    /** SKU parameter code */
    const PARAM_SKU = 'sku';

    /** Attributes parameter */
    const PARAM_ATTRIBUTE = 'attribute';

    /**
     * ALL products parameter code
     */
    const PARAM_ALL = 'all';

    /**
     * File to log messages to
     */
    const LOGFILE = 'js_attributereset.log';

    /**
     * Chunk size to update
     */
    const CHUNK_SIZE = 2000;

    /**
     * Max. limit products for varnish ban
     */
    const VARNISH_BAN_MAX_LIMIT = 200;

    /**
     * List of required parameters
     *
     * @var array
     */
    protected $requiredParameters = [
        self::PARAM_TARGET_STORE
    ];

    /**
     * List of attributes to copy values for
     *
     * @var array
     */
    protected $attributeList = [
        // We do not list attributes here which are listed in
        // script attributevaluecopy!
        'short_description',
        'canonical_cross_domain',
        'canonical_url',
        'meta_robots',
        'exclude_from_crosslinking',
        'exclude_from_sitemap',
        'exclude_from_html_sitemap',
        'special_price',
        'special_from_date',
        'special_to_date',
        'news_from_date',
        'news_to_date',
        'country_of_manufacture',
        'product_seo_name',
        'amazon_price',
        'msrp_enabled',
        'msrp_display_actual_price_type',
        'msrp',
        'tax_class_id',
        'custom_design',
        'custom_design_from',
        'custom_design_to',
        'custom_layout_update',
        'page_layout',
        'options_container',
        'configurable_alert',
        'barcode',
        'visibility',
        'status'
    ];

    /**
     * Map of attributes and their tables
     *
     * @var array
     */
    protected $attributeMap = [];

    /**
     * Product action model
     *
     * @var \Magento\Catalog\Model\Product\Action
     */
    protected $productAction;

    /**
     * Product model
     *
     * @var \Magento\Catalog\Model\Product
     */
    protected $productModel;

    /**
     * Varnish ban model
     *
     * @var Nexcessnet_Turpentine_Model_Observer_Ban
     */
    protected $varnish;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * @var \Magento\Framework\DataObjectFactory
     */
    protected $dataObjectFactory;

    /**
     * @var 
     */
  

    public function __construct(
      
    
      
        \Psr\Log\LoggerInterface $logger,
        \Magento\Eav\Model\Attribute $eavAttribute,
        \Magento\Framework\App\State $state, 
        \Magento\Store\Model\Store $store,
        \Magento\Catalog\Model\ResourceModel\Product\CollectionFactory $productCollection,       
        \Magento\Catalog\Model\ProductFactory $product,
        \Magento\Framework\App\ResourceConnection $resourceConnection,
        \Magento\Catalog\Model\ResourceModel\Product\Action $productAction,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Eav\Model\ResourceModel\Entity\Attribute\Group\CollectionFactory $attributeGroupCollection

    ) {

       
        $this->logger = $logger;     
        $this->state = $state;      
        $this->store = $store;
        $this->productCollection = $productCollection;       
        $this->product = $product;
        $this->resourceConnection = $resourceConnection;
        $this->connection = $this->resourceConnection->getConnection();     
        $this->productAction = $productAction;
        $this->productModel = $product;
        $this->storeManager = $storeManager;
        $this->attributeGroupCollection = $attributeGroupCollection;        
        parent::__construct();
    }


    protected function configure()
    {
       
        $this->setName('Xtwo:AttributesetCommand')
            ->setDescription('Add Attribute Set')         
            ->addOption('store', "store", InputOption::VALUE_OPTIONAL, "--store  List of store-codes to process for")
            ->addOption('sku', "sku", InputOption::VALUE_OPTIONAL, "--sku  List of SKU's to process")
            ->addOption('attribute', "attribute", InputOption::VALUE_OPTIONAL, "--attribute  If specified, it RESETs to the DEFAULT values for the given SKU's")
            ->addOption('all', "all", InputOption::VALUE_OPTIONAL, "--all  Flag to reset all products in provided store (ignored with --attribute)");
            
            parent::configure();
    }


  
    /**
     * @param string $msg
     */
    protected function log($msg)
    {
        $this->logger->log('100',$msg);
    }

    /**
     * Echo error and log to file
     *
     * @param string $msg
     * @param bool   $isIntro
     * @param bool   $skipLineBreak
     */
    protected function echoAndLog($msg, $isIntro = false, $skipLineBreak = false)
    {
        $echo = ($isIntro) ? $msg : '   ' . $msg;
        $lb = $skipLineBreak ? '' : "\n";
        echo $echo . $lb;
        $this->log($echo);
    }

    /**
     * Throw exception
     *
     * @param $message
     *
     * @throws \Exception
     */
    protected function throwException($message)
    {
        throw new \Exception($message);
    }

    /**
     * Init function
     * @throws \Exception
     */
    protected function init()
    {
      // $this->productAction = $this->productAction->create();
       $this->productAction = $this->productAction;
       $this->productModel = $this->product->create();
      //  $this->productModel = $this->product;
        

      //  $this->productAction = Mage::getSingleton('catalog/product_action');
     // $this->productAction = $this->productAction->create(); 
      //  $this->productModel = Mage::getSingleton('catalog/product');
       // $this->varnish = Mage::getSingleton('turpentine/observer_ban');
       //  $this->checkRequiredFields();
       // $this->prepareAttributes();

    }

    /**
     * Check required fields existence
     * @throws \Exception
     */
    protected function checkRequiredFields()
    {
        //check for required parameters
        foreach ($this->requiredParameters as $parameter) {
            if (!$this->input->getOption($parameter)) {
                $this->throwException(sprintf('Required parameter %s missing', $parameter));
            }
        }

        //if --sku parameters or --all not provided
        if (!$this->input->getOption(self::PARAM_SKU) && !$this->input->getOption(self::PARAM_ALL)) {
            $this->throwException(sprintf('Required parameter "%s" or "%s" missing', self::PARAM_SKU, self::PARAM_ALL));
        }
    }

    /**
     * Get sku list from provided shell parameter
     *
     * @param string $skuParamValue
     *
     * @return array
     */
    protected function getSkuList($skuParamValue)
    {
        $skuList = [];

        if ($skuParamValue) {
            $skuList = explode(',', trim($skuParamValue));
        }

        return $skuList;
    }

    /**
     * Get store ID by store code
     *
     * @param $codes
     *
     * @return array
     * @throws \Exception
     */
    protected function getStoresByCode($codes)
    {
        $stores = [];
        $codes = explode(',', $codes);
        foreach ($codes as $storeCode) {
            /** @var \Magento\Store\Model\Store $store */
            // $store = Mage::getModel('core/store')->load($storeCode, 'code');
            
            // $store = $this->store->create()->load($storeCode, 'code');
            $store = $this->store->load($storeCode, 'code');    
            $storeId = null;
            if ($store && $store->getId()) {
                $storeId = $store->getId();
            }

            if ($storeId === "0") {
                // hint a big mistake
                $this->throwException(sprintf("Your're changing default values!"));
            }
            if (!$storeId) {
                $this->throwException(sprintf('Store with code %s does not exist', $storeCode));
            }

            $stores[] = $store;
        }
        return $stores;
    }

    /**
     * Get product collection
     *
     * @param array $skuList
     *
     * @return \Magento\Catalog\Model\ResourceModel\Product\Collection
     */
    protected function getProducts($skuList)
    {
        /** @var \Magento\Catalog\Model\ResourceModel\Product\Collection $collection */
        // $collection = Mage::getResourceModel('catalog/product_collection');
        
        $collection = $this->productCollection->create();

        

        //if sku list provided, as as filter
        if (!empty($skuList)) {
            $collection->addAttributeToFilter('sku', array('in' => $skuList));
        }

        $collection->setPageSize(self::CHUNK_SIZE);
        $collection->setCurPage(1);
        $collection->setOrder('entity_id', 'asc');

        return $collection;
    }

    /**
     * Prepare product table and id and store them
     *
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function prepareAttributes()
    {
        if (!$this->attributeMap) {
            $attributeMap = [];
            foreach ($this->attributeList as $attributeCode) {
                $attribute = $this->productModel->getResource()->getAttribute($attributeCode);
                if ($attribute) {
                    $attributeMap[$attribute->getBackend()->getTable()][] = $attribute->getId();
                }
            }
            $this->attributeMap = $attributeMap;
        }

        return $this->attributeMap;
    }


    /**
     * @return array
     * @throws \Exception
     */
    protected function getAttributes() {
        $attributeCodes = $this->input->getOption(self::PARAM_ATTRIBUTE);
        if (!empty($attributeCodes)) {
            $attributeMap = [];
            // We take the attributes given as parameter
            $attributeCodes = explode(',', $attributeCodes);
            foreach ($attributeCodes as $attributeCode) {
                $attribute = $this->productModel->getResource()->getAttribute($attributeCode);
                if (!$attribute) throw new \Exception('Attribute not found: '.$attributeCode);
                $attributeMap[$attribute->getBackend()->getTable()][] = $attribute->getId();
            }
            return $attributeMap;
        } else {
            // We return the hard-coded attribute "default" list
            return $this->attributeMap;
        }
    }

    /**
     * Returns true if the script call contains an attribute list which resets to the default values. Reset means that
     * the store-level value will be removed even if it is not equal to the default value.
     * @return bool
     */
    protected function isResetAttributesActive() {
        $attributeCodes = $this->input->getOption(self::PARAM_ATTRIBUTE);
        return !empty($attributeCodes);
    }

    /**
     * Cleanup attributes to admin values for provided product ids and store id. It removes all entries which
     * equal to the default value (=> where it is not required to have a store-level value.
     *
     * @param $sourceProductIds
     * @param $storeId int target store ID
     *
     * @return int the number of real updated products
     * @throws \Exception
     */
    protected function cleanupAttributesToAdminValues($sourceProductIds, $storeId)
    {
        $realUpdates = null;
        $attributeMap = $this->getAttributes();
        /** @var \Magento\Framework\Db\Adapter\AdapterInterface $adapter */
       // $adapter = $this->productModel->getResource()->getWriteConnection();
       $adapter = $this->connection;
        foreach ($attributeMap as $table => $attributeIds) {
            //check if attributes have the same value on default store
            //if so, delete these attributes (by value from attribute table)
            $select = $adapter->select()->reset();
            $query = $select->from(
                $table,
                ['value_id']
            )->joinLeft(
                ['default_values' => $table],
                $table . '.attribute_id = default_values.attribute_id AND '
                . $table . '.entity_id = default_values.entity_id AND ('
                . $table . '.value = default_values.value OR ('
                . $table . '.value IS NULL AND default_values.value IS NULL))' ,
                []
            )->where(
                $table.'.attribute_id IN (?)',
                $attributeIds
            )->where(
                $table.'.entity_id IN (?)',
                $sourceProductIds
            )->where(
                $table.'.store_id = ?',
                $storeId
            )->where(
                'default_values.store_id = ?',
                0
            );
            //$this->echoAndLog('query: '.$query->__toString());
            $realUpdates += $adapter->exec($adapter->deleteFromSelect($query, $table));
        }

        return $realUpdates;
    }

    /**
     * Resets attributes to admin values for provided product ids and store id. It removes all entries independently from
     * the value of the store-level attribute.
     *
     * @param $sourceProductIds
     * @param $storeId int target store ID
     *
     * @return int the number of real updated products
     * @throws \Exception
     */
    protected function resetAttributesToAdminValues($sourceProductIds, $storeId)
    {
        $realUpdates = null;
        $attributeMap = $this->getAttributes();
        /** @var \Magento\Framework\Db\Adapter\AdapterInterface $adapter */
       // $adapter = $this->productModel->getResource()->getWriteConnection();
       $adapter = $this->connection;
        foreach ($attributeMap as $table => $attributeIds) {
            //check if attributes have the same value on default store
            //if so, delete these attributes (by value from attribute table)
            $select = $adapter->select()->reset();
            $query = $select->from(
                $table,
                ['value_id']
            )->joinLeft(
                ['default_values' => $table],
                $table . '.attribute_id = default_values.attribute_id AND '
                . $table . '.entity_id = default_values.entity_id',
                []
            )->where(
                $table.'.attribute_id IN (?)',
                $attributeIds
            )->where(
                $table.'.entity_id IN (?)',
                $sourceProductIds
            )->where(
                $table.'.store_id = ?',
                $storeId
            )->where(
                'default_values.store_id = ?',
                0
            );
           // $this->echoAndLog('query: '.$query->__toString());
            
            $realUpdates += $adapter->exec($adapter->deleteFromSelect($query, $table));
        }

        return $realUpdates;
    }

    /**
     * Ban varnish URLs for updated products
     *
     * @param $storeId
     * @param $productIds
     */
    protected function banVarnishUrls($productIds, $storeId)
    {
        if (count($productIds) > self::VARNISH_BAN_MAX_LIMIT) {
            $this->echoAndLog(sprintf(' ... skipped as limit reached to ban varnish.'), false, true);

            return;
        }
        $this->echoAndLog(sprintf('Clearing varnish cache for %s products', count($productIds)), false, true);
        foreach ($productIds as $productId) {
            $product = $this->productModel->reset()->setStoreId($storeId)->load($productId);
            $event = $this->dataObjectFactory->create();
          //  $this->varnish->banProductPageCache($event->setProduct($product));
        }
        $this->echoAndLog(sprintf(' ... done'), false, true);
    }

      /**
     * Run script
     * @param InputInterface $input, OutputInterface $output
     * @return string
     */

    protected function execute(InputInterface $input, OutputInterface $output)    
    {
        //collect parameters
        $this->input =  $input;
        $targetCode = $this->input->getOption(self::PARAM_TARGET_STORE);
        $skuParam = $this->input->getOption(self::PARAM_SKU);
        $skuList = $this->getSkuList($skuParam);

        try {
            //init models, check required fields
            $this->init();

            if ($this->isResetAttributesActive() && empty($skuList)) {
                throw new \Exception('To reset to default it is required to provide a list of SKUs!');
            }

            $targetStores = $this->getStoresByCode($targetCode);

            /** @var \Magento\Store\Model\Store $targetStore */
            foreach ($targetStores as $targetStore) {

                $targetStoreId = $targetStore->getId();

                $productIds = [];

                //get store ids and product collections
                
                $process = $this->isResetAttributesActive() ? 'Reset store-level attribute values to DEFAULT' :
                    'Clean up attribute value where possible';
                $this->echoAndLog(sprintf('%s for store %s', $process, $targetStore->getCode()));

                //set source store, to get valid attribute values
               // Mage::app()->setCurrentStore(Mage_Core_Model_App::ADMIN_STORE_ID);
                $this->storeManager->setCurrentStore(0);
                //get source products and values
                $sourceCollection = $this->getProducts($skuList);
                $pages = $sourceCollection->getLastPageNumber();
                $currentPage = 1;

                if (!$sourceCollection->getSize()) {
                    $this->throwException(sprintf('No products found for store %s', $targetCode));
                }

                do {
                    //get all ids with offset of 0 first, and chunksize + pagenum on other passes
                    $sourceProductIds = $sourceCollection->getAllIds(
                        self::CHUNK_SIZE,
                        self::CHUNK_SIZE * ($currentPage - 1)
                    );

                    // echo "ok".$sourceCollection->getSize()."okend";

                    if ($sourceCollection->getSize()) {
                        //chunkSize
                        if ($sourceCollection->getSize() < $sourceCollection->getPageSize()) {
                            $chunkSize = $sourceCollection->getSize();
                        } else {
                            $chunkSize = $sourceCollection->getPageSize();
                        }

                        $firstProductId = reset($sourceProductIds);
                        $this->echoAndLog(
                            sprintf('Processing chunk of %s products, start@%s', $chunkSize, $firstProductId),
                            false,
                            true
                        );

                        //set target values and get affected products
                        
                        if ($this->isResetAttributesActive()) {
                          
                            $realUpdates = $this->resetAttributesToAdminValues($sourceProductIds, $targetStoreId);
                        } else {
                           
                            $realUpdates = $this->cleanupAttributesToAdminValues($sourceProductIds, $targetStoreId);
                        }
                       // print_r($productIds);
                       // print_r($sourceProductIds);
                        // $productIds = array_merge($productIds, array_keys($sourceProductIds));
                        $productIds = array_merge($productIds, array_values($sourceProductIds));
                 
                        $this->echoAndLog(sprintf(' ... done (number of attributes: %s)', $realUpdates), false);
                      
                    }
                    $currentPage++;
                    $sourceCollection->clear();
                } while ($currentPage <= $pages);

                //clear varnish
               // $this->banVarnishUrls($productIds, $targetStoreId);
            }


        } catch (Exception $e) {
            $this->echoAndLog(sprintf('Stopped processing, Reason Exception: %s', $e->getMessage()));
            $this->echoAndLog("\n\n" . $this->usageHelp());
        }
    }

    /**
     * Retrieve Usage Help Message
     *
     */
    public function usageHelp()
    {
        return <<<USAGE
Usage:  php bin/magento Commands:AttributesetCommand --[options]
The script can do 2 things:
1) Cleanup: without a provided list of attributes it removes all not-required store-level entries from database for a
 specific list of attributes and where the store-level value is equal to the default value.
2) Reset: with a provided list of attribute codes it removes all store-level values from database independently from the 
attribute's default value.  

  --store <store_code>              List of store-codes to process for
  
  --sku "<sku_list>"                List of SKU's to process
  
  --attribute "<attribute_list>"    If specified, it RESETs to the DEFAULT values for the given SKU's

  --all                             Flag to reset all products in provided store (ignored with --attribute)

  help                              This help
  
  <store_code>  Store code, e.g. xtwostore_de, xtwostore_com etc.
  <sku_list>    List of product SKUs. If not provided, all products will be used.

USAGE;
    }
}

