<?php
/**
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_Automationshell
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 */

namespace Xtwo\Automationshell\Console;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Filesystem\Exception\FileNotFoundException;

/**
 * Command for executing cron jobs
 */

class AttributesValueCopyCommand extends Command
{
    /**
     * Source store parameter code
     */
    const PARAM_SOURCE_STORE = 'src';

    /**
     * Target store parameter code
     */
    const PARAM_TARGET_STORE = 'target';

    /** copy even if they equal to the defaults */
    const PARAM_IGNORE_DEFAULTS = 'ignoredefaults';

    /**
     * SKU parameter code
     */
    const PARAM_SKU = 'sku';

    /**
     * File to log messages to
     */
    const LOGFILE = 'js_attributevaluecopy.log';

    /**
     * Chunk size to update
     */
    const CHUNK_SIZE = 5000;

    /**
     * Max. limit products for varnish ban
     */
    const VARNISH_BAN_MAX_LIMIT = 200;

    /**
     * List of required parameters
     *
     * @var array
     */
    protected $requiredParameters = [
        self::PARAM_SOURCE_STORE,
        self::PARAM_TARGET_STORE
    ];

    /**
     * List of attributes to copy values for
     *
     * @var array
     */
    protected $attributeList = [
        'description',
        /*'short_description',*/
        'name',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'url_key'
    ];

    /**
     * Product action model
     *
     * @var \Magento\Catalog\Model\Product\Action
     */
    protected $productAction;

    /**
     * Varnish ban model
     *
     * @var Nexcessnet_Turpentine_Model_Observer_Ban
     */
    protected $varnish;

    protected $_mediaSingleton = null;
    protected $_mediaAttribute = null;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var \Magento\Framework\DataObjectFactory
     */
    protected $dataObjectFactory;


 //  \Magento\Catalog\Model\Product\Attribute\Backend\Media $media,

    public function __construct(
        \Magento\Catalog\Model\Product\Gallery\ReadHandler $galleryReadHandler,
        \Magento\Framework\DataObjectFactory $dataObjectFactory,
        \Magento\Catalog\Model\Product $productsingleton,     
        \Psr\Log\LoggerInterface $logger,
        \Magento\Eav\Model\Attribute $eavAttribute,
     
       
        \Magento\Framework\App\State $state,

        \Magento\Store\Model\Store $store,
        \Magento\Catalog\Model\ResourceModel\Product\Collection $productCollection,
       
        \Magento\Catalog\Model\Product $product,
        \Magento\Framework\App\ResourceConnection $resourceConnection,
        \Magento\Catalog\Model\ResourceModel\Product\Action $productAction,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Catalog\Helper\Image $imgHelper

    ) {
        $this->logger = $logger;
        $this->dataObjectFactory = $dataObjectFactory;
        $this->galleryReadHandler = $galleryReadHandler;     
        $this->state = $state;       
        $this->store = $store;
        $this->productCollection = $productCollection;      
        $this->product = $product;
        $this->resourceConnection = $resourceConnection;
        $this->connection = $this->resourceConnection->getConnection();      
       $this->productAction = $productAction;      
       $this->storeManager = $storeManager;    
       $this->imageHelper = $imgHelper;   
       // $this->checkRequiredFields();
        $this->productsingleton = $productsingleton;        
        parent::__construct();
    }


    protected function configure()
    {
                
        $this->setName('Xtwo:AttributesValueCopyCommand')
            ->setDescription('Attributes Value Autofills')         
            ->addOption('src', "s", InputOption::VALUE_OPTIONAL, "--src xtwostore_de")
            ->addOption('target', "t", InputOption::VALUE_OPTIONAL, "--target xtwostore_at")
            ->addOption('sku',"sku",InputOption::VALUE_OPTIONAL, "--skuParam 1234,A03-100")
            ->addOption('ignoredefaults',"isIgnoreDefaults",InputOption::VALUE_OPTIONAL, "--ignoredefaults 1")
            ->addOption('skuParam',"skuParam",InputOption::VALUE_OPTIONAL, "--skuParam check");
            
            parent::configure();
    }

    /**
     * @param string $msg
     *
     * @return Mage_Log_Model_Log
     */
    protected function log($msg)
    {
        // $this->logger->log(\Monolog\Logger::INFO, $msg);
        $this->logger->info($msg);
    }

    /**
     * Echo error and log to file
     *
     * @param string $msg
     * @param bool   $isIntro
     * @param bool   $skipLineBreak
     */
    protected function echoAndLog($msg, $isIntro = false, $skipLineBreak = false)
    {
        $echo = ($isIntro) ? $msg : '   '.$msg;
        $lb = $skipLineBreak ? '' : "\n";
        echo $echo.$lb;
        $this->log($echo);
    }

    /**
     * Throw exception
     *
     * @param $message
     *
     * @throws \Exception
     */
    protected function throwException($message)
    {
        throw new \Exception($message);
    }

    /**
     * Init function
     */
    protected function init()
    {
        // will see latter
       // $this->productAction = Mage::getSingleton('catalog/product_action');
       // $this->varnish = Mage::getSingleton('turpentine/observer_ban');
       // $this->checkRequiredFields();
    }

    /**
     * Check required fields existence
     */
    protected function checkRequiredFields()
    {
        //check for required parameters
        foreach ($this->requiredParameters as $parameter) {
           // if (!$this->getArg($parameter)) {
            //    $this->throwException(sprintf('Required parameter %s missing', $parameter));
          //  }
        }
    }

    /**
     * Get sku list from provided shell parameter
     *
     * @param string $skuParamValue
     *
     * @return array
     */
    protected function getSkuList($skuParamValue)
    {
        $skuList = [];

        if ($skuParamValue) {
            $skuList = explode(',', trim($skuParamValue));
        }

        return $skuList;
    }

    protected function getStoreIdByCode($code)
    {
        $storeId = null;

        /** @var \Magento\Store\Model\Store $store */
        
        // $store = Mage::getModel('core/store')->load($code, 'code');
         $store = $this->store->load($code, 'code');

        if ($store && $store->getId()) {
            $storeId = $store->getId();
        }

        if ($storeId === "0") {
            // hint a big mistake
            $this->throwException(sprintf("Your're changing default values!", $code));
        }
        if (!$storeId) {
            $this->throwException(sprintf('Store with code %s does not exist', $code));
        }

        return $storeId;
    }

    /**
     * Get product collection
     *
     * @param array $skuList
     * @param \Magento\Store\Model\Store store
     * @return \Magento\Catalog\Model\ResourceModel\Product\Collection
     */
    protected function getProducts($skuList, $store)
    {
        /** @var \Magento\Catalog\Model\ResourceModel\Product\Collection $collection */
       // $collection = Mage::getResourceModel('catalog/product_collection');
        $collection = $this->productCollection;
        $collection->setStore($store);
        $collection->addAttributeToSelect($this->attributeList);

        //if sku list provided, as as filter
        if (!empty($skuList)) {
            $collection->addAttributeToFilter('sku', array('in' => $skuList));
        }

        $collection->setPageSize(self::CHUNK_SIZE);
        $collection->setCurPage(1);
        $collection->setOrder('entity_id', 'asc');

        return $collection;
    }


    

    /**
     * Get attribute values from source product collection
     * e.g. product_id => array('name' => 'Product name')
     *
     * @param array $sourceProductsArray
     * @param int $storeId
     * @return array
     */
    protected function getSourceAttributeValues($sourceProductsArray, $storeId)
    {
       // echo "-source-".$storeId."-source-";
        $attributes = [];
        //go through all products
        /** @var \Magento\Catalog\Model\Product $sourceProduct */
        foreach ($sourceProductsArray as $sourceProduct) {
            $sourceProduct->setStoreId($storeId);
            $productAttributes = [];
            // go through all attributes from list
            foreach ($this->attributeList as $attribute) {
             
                if ($sourceProduct->hasData($attribute)) {
                    $productAttributes[$attribute] = $sourceProduct->getData($attribute);
                
                }
            }
            $galleryImages = $this->loadGallery($sourceProduct);

          //  foreach($galleryImages as $images){
               // print_r($images);
           // }

            

            //if we got any data, save it to an array to return
            if (!empty($productAttributes)) {
                if (!empty($galleryImages)) {
                    $productAttributes['gallery_images_internal'] = $galleryImages;
                }
                $attributes[$sourceProduct->getId()] = $productAttributes;
            }
        }

        if (empty($attributes)) {
            $this->throwException(sprintf('No data collected from source products'));
        }

        return $attributes;
    }


    public function addGallery($product) {
        $this->galleryReadHandler->execute($product);
    }

    public function getMediaGalleryImages(\Magento\Catalog\Api\Data\ProductInterface $product)
    {

       // $product = $objectManager->create('\Magento\Catalog\Model\Product');
/*$product = $product->getCollection()
->addFieldToSelect("*")
// ->addAttributeToFilter('sku', 'sku-01')
->addMediaGalleryData()
->getFirstItem();
*/

        $images = $product->getMediaGalleryImages();
        if ($images instanceof \Magento\Framework\Data\Collection) {
            foreach ($images as $image) {
                /** @var $image \Magento\Catalog\Model\Product\Image */
                $image->setData(
                    'small_image_url',
                    $this->imageHelper->init($product, 'product_page_image_small')
                        ->setImageFile($image->getFile())
                        ->getUrl()
                );
                $image->setData(
                    'medium_image_url',
                    $this->imageHelper->init($product, 'product_page_image_medium')
                        ->constrainOnly(true)->keepAspectRatio(true)->keepFrame(false)
                        ->setImageFile($image->getFile())
                        ->getUrl()
                );
                $image->setData(
                    'large_image_url',
                    $this->imageHelper->init($product, 'product_page_image_large')
                        ->constrainOnly(true)->keepAspectRatio(true)->keepFrame(false)
                        ->setImageFile($image->getFile())
                        ->getUrl()
                );
            }
        }
        return $images;
    }


    private function loadGallery($product) {

        $this->addGallery($product);
        $galleryImage = $this->getMediaGalleryImages($product);
        return $galleryImage;

      
    }

    

    /**
     * Returns the default attributes for the given list of source products.
     * @param $sourceProductsArray
     * @param int $storeId
     * @return array
     */
    protected function getDefaultAttributeValues($sourceProductsArray, $storeId) {
        // echo "-default-".$storeId."-default-";
       
        $attributes = array();
        $productIds = array_keys($sourceProductsArray);
       // $products = Mage::getModel('catalog/product');
       $products = $this->product->getCollection();
      
        $products->setStore($this->storeManager->getStore(0));
       
        
        $products->addAttributeToSelect($this->attributeList);
        $products->addIdFilter($productIds);
        foreach ($products as $product) {
            $product->setStoreId($storeId);
            $productAttributes = [];
            foreach ($this->attributeList as $attribute) {
                // if data is set for attribute, add it to array
                if ($product->hasData($attribute)) {
                    $productAttributes[$attribute] = $product->getData($attribute);
                }
            }
            if (!empty($productAttributes)) {
                $attributes[$product->getId()] = $productAttributes;
            }
        }
        return $attributes;
    }

    /**
     * Set product attribute values for target store products
     *
     * @param $sourceValues
     * @param $defaultValues
     * @param $storeId int target store ID
     * @param $isIgnoreDefaults true to copy even if the values are equal to the defaults
     * @return int the number of real updated products
     */
    protected function setTargetAttributeValues($sourceValues, $storeId, $defaultValues, $isIgnoreDefaults)
    {
        echo "-target--".$storeId."--target--";
       
        $realUpdates = 0;
        // go through all target products
        foreach ($sourceValues as $productId => $attributes) {
            $resultAttributes = array();
            // update attribute values for product
            if (array_key_exists($productId, $defaultValues)) {
             
               $defaultAttributes = $defaultValues[$productId];
            
            
                foreach ($attributes as $attributeKey => $value) {
                

                    if ($attributeKey != 'gallery_images_internal' &&
                       ($isIgnoreDefaults || $defaultAttributes[$attributeKey] != $value)) {
                    
                        $resultAttributes[$attributeKey] = $value; // we have a change
                    }
                }
            }

           // print_r($resultAttributes);
            
            if (!empty($resultAttributes)) {
                $this->productAction->updateAttributes(array($productId), $resultAttributes, $storeId);
                $realUpdates++;
            }

            // Image labels: the 'label' is the source image label
            if (isset($attributes['gallery_images_internal'])) {
             

                // $coreResource = Mage::getSingleton('core/resource');
                $coreResource = $this->resourceConnection;
                $tblGalVal = $coreResource->getTableName('catalog_product_entity_media_gallery_value');
                foreach ($attributes['gallery_images_internal'] as $imgData) {

                   // print_r($imgData);
                   
                    if ($imgData['label'] && $imgData['label_default'] &&
                        ($imgData['label'] != $imgData['label_default'])) {

                          
                       
                         $v = $this->connection->fetchOne("SELECT value_id FROM $tblGalVal WHERE value_id=:valueId AND store_id=:storeId",
                            array('valueId' => $imgData['value_id'], 'storeId' => $storeId));

                        // $v = $coreResource->getConnection('core_read')->fetchOne("SELECT value_id FROM $tblGalVal WHERE value_id=:valueId AND store_id=:storeId",
                          //  array('valueId' => $imgData['value_id'], 'storeId' => $storeId));

                        if ($v) {
                            $sql = "UPDATE $tblGalVal SET label=:label WHERE value_id=:valueId AND store_id=:storeId";
                           // $coreResource->getConnection('core_write')
                            //    ->query($sql, array('valueId'=>$imgData['value_id'], 'storeId'=>$storeId,
                                         //                  'label'=>$imgData['label']));
                        echo "UPDATE $tblGalVal SET label=".$imgData['label']." WHERE value_id=".$imgData['value_id']." AND store_id=".$storeId;                       
                        
                        $this->connection->query($sql, array('valueId'=>$imgData['value_id'], 'storeId'=>$storeId,
                                                          'label'=>$imgData['label']));
                        } else {
                            $sql = "INSERT IGNORE INTO $tblGalVal (value_id, store_id, label, position, disabled) VALUES " .
                                '(:valueId, :storeId, :label, :pos, :disabled)';
                          //  $coreResource->getConnection('core_write')
                              //  ->query($sql, array('valueId'  => $imgData['value_id'], 'storeId' => $storeId,
                                                  //  'label'    => $imgData['label'], 'pos' => $imgData['position'],
                                                   // 'disabled' => $imgData['disabled']));

                           
                            $this->connection->query($sql, array('valueId'  => $imgData['value_id'], 'storeId' => $storeId,
                                                'label'    => $imgData['label'], 'pos' => $imgData['position'],
                                                  'disabled' => $imgData['disabled']));
                        }
                    }
                }
            }
        }
        return $realUpdates;
    }

    /**
     * Ban varnish URLs for updated products
     *
     * @param $storeId
     * @param $productIds
     */
    protected function banVarnishUrls($productIds, $storeId)
    {
        if (count($productIds) > self::VARNISH_BAN_MAX_LIMIT) {
            $this->echoAndLog(sprintf(' ... skipped as limit reached to ban varnish.'), false, true);
            return;
        }
        $this->echoAndLog(sprintf('Clearing varnish cache for %s products', count($productIds)), false, true);
        foreach ($productIds as $productId) {
            $product =  $this->product->setStoreId($storeId)->load($productId);
            $event = $this->dataObjectFactory->create();
            // $this->varnish->banProductPageCache($event->setProduct($product));
        }
        $this->echoAndLog(sprintf(' ... done'), false, true);
    }

   
    /**
     * Run script
     * @param InputInterface $input, OutputInterface $output
     * @return string
     */
    
    protected function execute(InputInterface $input, OutputInterface $output)  
    {
     
 
        $this->state->setAreaCode(\Magento\Framework\App\Area::AREA_FRONTEND); 
        $this->input = $input;       

        //collect parameters
        $sourceCode = $this->input->getOption(self::PARAM_SOURCE_STORE); 
        $targetCode = $this->input->getOption(self::PARAM_TARGET_STORE);
        $isIgnoreDefaults = (bool) $this->input->getOption(self::PARAM_IGNORE_DEFAULTS);
        $skuParam = $this->input->getOption(self::PARAM_SKU);
        // $skuList = $this->input->getOption($skuParam);
        $skuList = $this->getSkuList($skuParam);


        try {

             
            //init models, check required fields
            // $this->init();
            $productIds = [];

            //get store ids and product collections
            $sourceStoreId = $this->getStoreIdByCode($sourceCode);
            $targetStoreId = $this->getStoreIdByCode($targetCode);

            $this->echoAndLog(sprintf('Setting attribute values for store %s', $targetCode));

            //set source store, to get valid attribute values
            //Mage::app()->setCurrentStore($sourceStoreId);

            //get source products and values
            $sourceCollection = $this->getProducts($skuList, $this->storeManager->getStore($sourceStoreId));
            $pages = $sourceCollection->getLastPageNumber();
            $currentPage = 1;

            if (!$sourceCollection->getSize()) {
                $this->throwException(sprintf('No products found for store %s', $targetCode));
            }

            do {
                $sourceCollection->setCurPage($currentPage);
                $sourceProducts = $sourceCollection->getItems();

                if ($sourceCollection->getSize()) {
                    //chunkSize
                    if ($sourceCollection->getSize() < $sourceCollection->getPageSize()) {
                        $chunkSize = $sourceCollection->getSize();
                    } else {
                        $chunkSize = $sourceCollection->getPageSize();
                    }


                    $firstProduct = reset($sourceProducts);
                    $this->echoAndLog(sprintf('Processing chunk of %s products, start@%s', $chunkSize, $firstProduct->getId()), false, true);
                    $sourceValues = $this->getSourceAttributeValues($sourceProducts, $sourceStoreId);
                    //print_r($sourceValues);
                    $defaultValues = $this->getDefaultAttributeValues($sourceProducts, $targetStoreId);

                    //set source values and get affected products
                    $realUpdates = $this->setTargetAttributeValues($sourceValues, $targetStoreId, $defaultValues, $isIgnoreDefaults);
                    $productIds = array_merge($productIds, array_keys($sourceValues));

                    $this->echoAndLog(sprintf(' ... done (real:%s)', $realUpdates), false);
                }
                $currentPage++;
                $sourceCollection->clear();
            } while ($currentPage <= $pages);

            // revert back to admin store
            $this->storeManager->setCurrentStore(0);

            //clear varnish
           // $this->banVarnishUrls($productIds, $targetStoreId);

          // $this->state->setAreaCode($originalArea);

        } catch (Exception $e) {
            $this->echoAndLog(sprintf('Stopped processing, Reason Exception: %s', $e->getMessage()));
            $this->echoAndLog("\n\n".$this->usageHelp());
        }
    }

    /**
     * Retrieve Usage Help Message
     *
     */
    public function usageHelp()
    {
        return <<<USAGE
Usage:  php bin/magento Xtwo:AttributesValueCopyCommand --[options]

Copies only a specific list of attributes: 'description', 'name', 'meta_title', 'meta_description', 'meta_keywords', 'url_key'.

  --src <store_code>    Source store to copy values from
  
  --target <store_code> Target store to copy values to
  
  --sku <sku_list>      Optional comma separated list of SKUs to copy values for

  --ignoredefaults      Optionally. If set, the values will be copied even if they equal to the defaults (admin-store)

  help                  This help
  
  <store_code>  Store code, e.g. xtwostore_de, xtwostore_com etc.
  <sku_list>    List of product SKUs. If not provided, all products will be used.

USAGE;
    }
}

// $shell = $this->->create();
// $shell->run();