<?php
/**
 * Xtwo
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the metawolf.com license that is
 * available through the world-wide-web at this URL:
 * https://www.metawolf.com/license.txt
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade this extension to newer
 * version in the future.
 *
 * @category    Xtwo
 * @package     Xtwo_Automationshell
 * @copyright   Copyright (c) MetaWolf (https://www.metawolf.com/)
 * @license     https://www.metawolf.com/license.txt
 */

namespace Xtwo\Automationshell\Console;


use \Statickidz\GoogleTranslate;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Filesystem\Exception\FileNotFoundException;

/**
 * Command for executing cron jobs
 */

class  AttributesAutofillsCommand extends Command{

    protected $attributeGroups = array();

    /** @var array should be NULL to support ALL */
    protected $supportedAttributSets = null; //array('Badmöbel');

    const NOTATION_EXCLUDE_ATTRSET = '!!';
    const NOTATION_INCLUDE_ALL_ATTRSET = '*';

    /** target store parameter code */
    const PARAM_TARGET_STORE = 'store';

    /** SKU parameter code */
    const PARAM_SKU = 'sku';

    /** attributeset parameter */
    const PARAM_ATTRIBUTESET = 'attributeset';
    const PARAM_ATTRIBUTSET = 'attributeset';

    /** attributeset parameter */
    const PARAM_TARGET_ATTRIBUTE = 'attribute';

    /**
     * File to log messages to
     */
    const LOGFILE = 'js_attributeautofill.log';

    /**
     * Chunk size to update
     */
    const CHUNK_SIZE = 2000;

    /**
     * Max. limit products for varnish ban
     */
    const VARNISH_BAN_MAX_LIMIT = 200;

    protected $productMediaGalleryBackendModel = null;

    /**
     * List of required parameters
     *
     * @var array
     */
    protected $requiredParameters = [
        'store' => 'Currently at least one store is required!'
    ];

    /**
     * List of attributes to copy values for
     *
     * @var array
     */
    protected $attributeList = [
        // We do not list attributes here which are listed in
        // script attributevaluecopy!
        'short_description',
        'description',
        'name'
    ];

    /**
     * Map of attributes and their tables
     *
     * @var array
     */
    protected $attributeMap = [];

    /**
     * Product action model
     *
     * @var \Magento\Catalog\Model\Product\Action
     */
    protected $productAction;

    /**
     * Product model
     *
     * @var \Magento\Catalog\Model\Product
     */
    protected $productModel;

    /** @var array  */
    protected $processingErrors = array();

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Framework\TranslateInterface
     */
    protected $translator;

    /**
     * @var 
     */
    
    protected $phraseRenderer;
    
    public function __construct(
        \Psr\Log\LoggerInterface $logger,
        \Magento\Eav\Model\Attribute $eavAttribute,
        \Magento\Eav\Model\Entity $entity,
        \Magento\Framework\ObjectManagerInterface $objectManager,
        \Magento\Framework\App\State $state,
        \Magento\Eav\Model\ResourceModel\Entity\Attribute\Set\CollectionFactory $attributesetCollection,
        \Magento\Store\Model\Store $store,
        \Magento\Catalog\Model\ResourceModel\Product\CollectionFactory $productCollection,
        \Magento\Store\Model\App\Emulation $storeimulation,
        \Magento\Catalog\Model\ProductFactory $product,
        \Magento\Framework\App\ResourceConnection $resourceConnection,
        \Magento\Catalog\Model\ResourceModel\Product\Action $productAction,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Framework\TranslateInterface $translator,
        \Magento\Store\Model\App\Emulation $appEmulation,
        \Magento\Framework\App\AreaList $areaList,
        \Magento\Eav\Api\AttributeSetRepositoryInterface $attributesetICollection,
        \Magento\Eav\Model\ResourceModel\Entity\Attribute\Group\CollectionFactory $attributeGroupCollection,
        \Magento\Catalog\Model\ResourceModel\Product\Attribute\CollectionFactory $productAttributeCollection
    ) {

        $this->appEmulation = $appEmulation;
        $this->areaList = $areaList;
        $this->translate = $translator;
        $this->logger = $logger;
        $this->objectManager = $objectManager;
        $this->state = $state;
        $this->attributesetCollection = $attributesetCollection;
        $this->store = $store;
        $this->productCollection = $productCollection;
        $this->storeimulation = $storeimulation;
        $this->product = $product;
        $this->resourceConnection = $resourceConnection;
        $this->connection = $this->resourceConnection->getConnection();
        $this->attributesetICollection = $attributesetICollection;
        $this->eavAttribute = $eavAttribute;
        $this->scopeConfig = $scopeConfig;
        $this->translator = $translator;
        $this->productAction = $productAction;
        $this->productModel = $product;
        $this->storeManager = $storeManager;
        $this->attributeGroupCollection = $attributeGroupCollection;
        parent::__construct();
    }
    /**
     * Return the translator instance by correct language.
     *
     * @param string $langCode
     * @return mixed
     */
 

    /**
     * {@inheritdoc}
     */
    protected function configure(){
        $this->setName('Xtwo:AttributesAutofills')
            ->setDescription('Autofill Attributes')         
            ->addOption('store', "s", InputOption::VALUE_REQUIRED, "--stores xtwostore_de,xtwostore_ch")
            ->addOption('sku', "u", InputOption::VALUE_OPTIONAL, "--sku 34840000,34841000")
            ->addOption('attributeset',"a",InputOption::VALUE_OPTIONAL, "--attributeset Waschtischarmaturen");
            parent::configure();
    }
    
    /**
     * @param string $msg
     *
     * @return Mage_Log_Model_Log
     */
   protected function log($msg){
    $this->logger->log('100',$msg);
   }
    /**
     * Echo error and log to file
     *
     * @param string $msg
     * @param bool   $isIntro
     * @param bool   $skipLineBreak
     */
    protected function echoAndLog($msg, $isIntro = false, $skipLineBreak = false){
        $echo = ($isIntro) ? $msg : '   ' . $msg;
        $lb = $skipLineBreak ? '' : "\n";
        echo $echo . $lb;
        $this->log($echo);
    }
    
    /**
     * Throw exception
     *
     * @param $message
     *
     * @throws \Exception
     */
    protected function throwException($message){
        throw new \Exception($message);
    }
    
    /**
     * Check required fields existence
     */
    protected function checkRequiredFields(){
        //check for required parameters
       foreach ($this->requiredParameters as $parameter => $msg) {
            if (!$this->input->getOption(self::PARAM_SKU) 
            && !$this->input->getOption('store') 
            && !$this->input->getOption('attributeset')) {
                $this->throwException(sprintf('Required parameter "--%s" missing: %s', $parameter, $msg));
            }
        }
    }

    /**
     * Get sku list from provided shell parameter
     *
     * @param string $skuParamValue
     *
     * @return array
     */
    protected function getSkuList($skuParamValue){
        $skuList = [];
        if ($skuParamValue) {
            $skuList = explode(',', trim($skuParamValue));
        }
        return $skuList;
    }

    /**
     * Get sku list from provided shell parameter
     *
     * @return array
     */
    protected function getAttributeSetIdList(){
        $ids = array();
        $items = [];
        $value = $this->input->getOption(self::PARAM_ATTRIBUTESET);
        if (!$value) {
            $value = $this->input->getOption(self::PARAM_ATTRIBUTSET);
        }
        if ($value) {
            $items = explode(',', trim($value));
        }
        if (!empty($items)) {
            /** @var \Magento\Eav\Model\ResourceModel\Entity\Attribute\Set\Collection $attributeSets */
            
            $getResourceModel = $this->attributesetCollection->create();
            $attributeSets = $getResourceModel->setEntityTypeFilter(4); /*product*/
            return ;
            if (!empty($items)) {
                $attributeSets->addFilter('attribute_set_name', array('in' => $items));
            }
            /** @var \Magento\Eav\Model\Entity\Attribute\Set $attributeSet */
            foreach ($attributeSets as $attributeSet) {
                $ids[] = $attributeSet->getId();
            }
        }
        return $ids;
    }

    /**
     * @return array
     * @throws Mage_Core_Model_Store_Exception
     */
     
    protected function getStores() {
        $storeCodes = $this->input->getOption(self::PARAM_TARGET_STORE);
        if ($storeCodes) {
            $storeCodes = explode(',', $storeCodes);
            $stores = array();
            foreach ($storeCodes as $storeCode) {
                $stores[] = $this->storeManager->getStore($storeCode);
            }
        } else {
            $stores = $this->storeManager->getStores();
        }
        return $stores;
    }

    /**
     * Get store ID by store code
     *
     * @param $code
     *
     * @return int|null
     */
    protected function getStoreIdByCode($code){
        $storeId = null;

        /** @var \Magento\Store\Model\Store $store */
         $store = $this->store->create()->load($code, 'code');
        if ($store && $store->getId()) {
            $storeId = $store->getId();
        }
        if ($storeId === "0") {
            // hint a big mistake
            $this->throwException(sprintf("Your're changing default values!", $code));
        }
        if (!$storeId) {
            $this->throwException(sprintf('Store with code %s does not exist', $code));
        }
        return $storeId;
    }

    /**
     * Get product collection
     *
     * @return \Magento\Catalog\Model\ResourceModel\Product\Collection
     * @throws \Exception
     */
    protected function getProducts(){
        $skuList = $this->getSkuList($this->input->getOption(self::PARAM_SKU));
        $attributeSets = $this->getAttributeSetIdList();
        if (!empty($skuList) && !empty($attributeSets)) {
           // throw new \Exception('You cannot process for SKUs and AttributeSets together.');
        }

        $limitLog = '';
        /** @var \Magento\Catalog\Model\ResourceModel\Product\Collection $collection */
        $collection = $this->productCollection->create();
        $collection->addAttributeToFilter('status', '1');
        //if sku list provided, as as filter
        if (!empty($skuList)) {
            $collection->addAttributeToFilter('sku', array('in' => $skuList));
            $limitLog .= ' sku: '.count($skuList);
        }
        if (!empty($attributeSets)) {
            $collection->addAttributeToFilter('attribute_set_id', array('in' => $attributeSets));
            $limitLog .= ' attributesets: '.count($attributeSets);
        }

        $collection->setPageSize(self::CHUNK_SIZE);
        $collection->setCurPage(1);
        $collection->setOrder('entity_id', 'asc');
        return $collection;
    }

    /**
     * Result is array with key <attributeset-name> => values
     * @param \Magento\Store\Model\Store $store
     * @return array|bool
     * @throws \Exception
     */
    protected function getConfigItems($store) {
        if (!$this->scopeConfig->getValue('attribute_autofill/general/active',
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $store)) {
            return false;
        }
        $lastOne = $firstOne = null;
        $configItems = array();
        $lines = explode("\n", $this->scopeConfig->getValue('attribute_autofill/general/configs',
        \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $store));
        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line) || strpos($line, '--') === 0) continue;
            $parts = explode('#', $line);
            if ((count($parts) > 0 && count($parts) != 3) && (strpos($parts[0], self::NOTATION_EXCLUDE_ATTRSET) !== 0)) {
                throw new \Exception(
                    sprintf('Invalid configuration line (store=%s): %s', $store->getCode(), $line));
            }
            $itemData = array(
                'attributeset' => mb_strtolower($parts[0]),
                'attribute' => mb_strtolower($parts[1]),
                'pattern' => $parts[2]);

            if ($lastOne == null && $parts[0] == self::NOTATION_INCLUDE_ALL_ATTRSET) {
                $lastOne = $itemData;
            } elseif ($firstOne == null && strpos($parts[0], self::NOTATION_EXCLUDE_ATTRSET) === 0) {
                $firstOne = $itemData;
            } else {
                $configItems[] = $itemData;
            }
        }
        if (!is_null($lastOne)) {
            // * to end
            $configItems[] = $lastOne;
        }
        if (!is_null($firstOne)) {
            // exclusion to begin
            $firstOnes = array();
            $attributeSets = str_replace('!!', '', $firstOne['attributeset']);
            $attributeSets = explode(',', $attributeSets);
            foreach ($attributeSets as $attributeSetName) {
                $data = $firstOne;
                $data['attributeset'] = self::NOTATION_EXCLUDE_ATTRSET.$attributeSetName;
                $firstOnes[] = $data;
            }
            $configItems = array_merge($firstOnes, $configItems);
        }
        return $configItems;
    }

   /**
     * Run script
     * @param InputInterface $input, OutputInterface $output
     * @return string
     */

    protected function execute(InputInterface $input, OutputInterface $output){
        $this->state->setAreaCode(\Magento\Framework\App\Area::AREA_FRONTEND); 
        try {
            $this->input = $input;
            $this->checkRequiredFields();
            // Get the stores to iterate through
            $stores = $this->getStores();
            // Get source products and valuesinput 
            $sourceCollection = $this->getProducts();
            $totalCount = $sourceCollection->getSize();
            if (!$totalCount) {
               $this->throwException(sprintf('No products selected!'));
            }
            $this->echoAndLog(sprintf('Processing for %s product candidates', $totalCount));

            /** @var \Magento\Store\Model\Store $store */
            foreach ($stores as $store) {
                $locale =   $this->scopeConfig->getValue('general/locale/code',
                \Magento\Store\Model\ScopeInterface::SCOPE_STORE, $store->getStoreId());
              
                /** @var \Magento\Store\Model\App\Emulation $appEmulation */
                // $appEmulation = Mage::getSingleton('core/app_emulation');
                $appEmulation = $this->storeimulation;
                $initialEnvironmentInfo = $appEmulation->startEnvironmentEmulation($store->getId());
                $area = $this->areaList->getArea(\Magento\Framework\App\Area::AREA_FRONTEND);
                $area->load(\Magento\Framework\App\Area::PART_TRANSLATE);
                $this->storeManager->setCurrentStore($store);
                $this->appEmulation->startEnvironmentEmulation($store->getId());
    
                $configItems = $this->getConfigItems($store);
                if (!$configItems) {
                    $this->echoAndLog(sprintf('Skipped processing for store=%s (as deactivated)', $store->getCode()));
                    continue;
                } 

                $this->echoAndLog(sprintf('Found %s attribute configurations for store %s', count($configItems), $store->getCode()));

                $pages = $sourceCollection->getLastPageNumber();
                $processingCount = 0;
                $currentPage = 1;
                do {
                    $sourceCollection->setCurPage($currentPage);

                    /** @var $product Mage_Catalog_Model_Product */
                    $i=0;
                    foreach ($sourceCollection as $product) {
                        $updateData = array();
                        foreach ($configItems as $configItem) {
                            $matchType = $this->configItemMatchesForProduct($configItem, $product);
                            if ($matchType === false) {
                                // Skip to next config item
                            } elseif ($matchType) {
                               $product = $this->product->create()->setStoreId($store->getId())->load($product->getId());
                               $attributeCode = $configItem['attribute'];
                                $pattern = $configItem['pattern'];
                                $result = $this->calculateAttributeValue($attributeCode, $pattern, $product, $store);
                                
                                // print_r($result);
                                if ($result['isOk']) {
                                    //$trans = new GoogleTranslate();
                                   
                                    // comment/uncomment if limit exceeds
                                  //  $result['value'] = $trans->translate('en', substr($locale,0,2), $result['value']);
                                    $updateData[$result['type']][$attributeCode] = $result['value'];
                                
                                } else {
                                   //  $this->addError($result['value']);
                               //   echo "error".$i++.$result['value']."\r\n";
                                }

                            } else {
                                // Must be null => we don't take this product as explicitely excluded
                                break;
                            }
                        }

                        if (isset($updateData['attributes']) && !empty($updateData['attributes'])) {
                            $this->productAction->updateAttributes(array(
                            $product->getId()), 
                            $updateData['attributes'], 
                            $store->getId());
                        }
                        

                        if(isset($updateData['attributes']['url_key']) && !empty($updateData['attributes']['url_key'])){
                            $url = strtolower(str_replace(' ', '-', $updateData['attributes']['url_key']));
                            $product = $this->productModel->create()->load($product->getId());
                            $product->setUrlKey($url);
                            $product->save();                                                        
                        }

                        if (isset($updateData['gallery_images']) && !empty($updateData['gallery_images'])) {
                            $label = $updateData['gallery_images']['gallery_image_label'];
                            $table = 'catalog_product_entity_media_gallery_value';
                            $images = $product->getMediaGalleryImages();
                            foreach ($images as $imageId => $dataObject) {
                               // echo $dataObject->getRecordId();
                                $imgNo = count($images) > 1 ? (', '.__('Image').' '.$dataObject->getData('position')) : '';

                                $row = $this->connection->fetchRow(
                                    sprintf("SELECT * FROM %s WHERE value_id = '%s' AND store_id = '%s' ",
                                    $table, $imageId, $store->getId()));
 
                                if (!empty($row)) {
                                    $this->connection->update($table,[
                                        'label' => $label . $imgNo], 
                                        'value_id = ' . $imageId . ' and store_id = ' . $store->getId());
                                } else {
                                    $row = $this->connection->fetchRow(sprintf(
                                        "SELECT * FROM %s WHERE value_id = '%s' AND store_id = '%s' ", 
                                        $table, $imageId, 0));
                                    $row['label'] = $label.$imgNo;
                                    $row['store_id'] = $store->getId();
                                    $this->connection->insert($table, $row);
                                }

                                // Update image label of the active image (pos 1)
                                $imageLabels = ['image_label' => $label, 'small_image_label' => $label, 'thumbnail_label' => $label];
                                $this->productAction->updateAttributes(array($product->getId()), $imageLabels, $store->getId());
                                if ($store->getId() == 1) {
                                    $this->productAction->updateAttributes(array($product->getId()), $imageLabels, 0);
                                }
                            }
                        }
                        $processingCount++;
                        if ($processingCount % 500 == 0) {
                           $this->echoAndLog(sprintf('Processed %s of %s...', $processingCount, $totalCount));
                        }
                    }

                    $currentPage++;
                    $sourceCollection->clear();
                } while ($currentPage <= $pages);
                  $this->echoAndLog(sprintf('Done for %s products', $processingCount));
                  $appEmulation->stopEnvironmentEmulation($initialEnvironmentInfo);
                  echo "Finished run";
            }
        } catch (Exception $e) {
            $this->echoAndLog(sprintf('Stopped processing, Reason Exception: %s', $e->getMessage()));
            $this->echoAndLog("\n\n" . $this->usageHelp());
            exit();
        }
    }

    /**
     * @param \Magento\Catalog\Model\Product $product
     * @return
     */
    protected function getMediaGalleryBackendModel($product) {
        if (is_null($this->productMediaGalleryBackendModel)) {
            $attributes = $product->getTypeInstance(true)->getSetAttributes($product);
            /** @var \Magento\Catalog\Model\ResourceModel\Eav\Attribute $mediaGallery */
            $mediaGallery = $attributes['media_gallery'];
            $backend = $mediaGallery->getBackend();
            $backend->afterLoad($product);
            $this->productMediaGalleryBackendModel = $backend;
        }
        return $this->productMediaGalleryBackendModel;
    }

    /**
     * @return \Magento\Framework\Db\Adapter\AdapterInterface
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    protected function getWriteConnection() {
        return $this->connection;
    }
    /**
     *
     */
    protected function printErrors() {
        if (!empty($this->processingErrors))
            $this->echoAndLog('Errors:');
        foreach ($this->processingErrors as $msg => $count) {
            $this->echoAndLog(sprintf('%s times => "%s"', $count, $msg));
        }
    }

    /**
     * Calculates Option values
     *
     * @param $attributeCode string
     * @param $pattern string
     * @param \Magento\Catalog\Model\Product $product
     * @param \Magento\Store\Model\Store $store
     * @return array
     */
    protected function calculateAttributeValue($attributeCode, $pattern, $product, $store){
        $value = $pattern;
        $return = array('type' => 'attributes');
        $result = array();
        $matchesCount = preg_match_all("/\{(.*?)\}/", $pattern, $result, PREG_PATTERN_ORDER);

        if (strtoupper($pattern) == 'DESCRIPTION') {
            $value = '';
           $groupedAttributes = $this->getExtendedGroupedAttributesForProduct($product);
            foreach ($groupedAttributes as $groupLabel => $groupData) {
              $value .= sprintf('<b>%s</b><br/><br/>', $groupLabel);
                foreach ($groupData as $attrCode => $attrValue) {
                    $itemValue = $attrValue['value'];
                    if (is_array($itemValue)) {
                        $itemValue = implode(',', $itemValue);
                    }
                    $value .= sprintf('%s: %s<br/>', $attrValue['attributeLabel'], $itemValue);
                }
                $value .= '<br/><br/>';
            }
            $return['isOk'] = true;
            $return['value'] = $value;

        } elseif (strpos(strtoupper($attributeCode), 'GALLERY_IMAGE') === 0) {
            $return = $this->replaceVariables($value, $result, $product, $store);
            $return['type'] = 'gallery_images';

        } elseif (count($result) == 2 && !empty($result[0])) {
            $return = array_merge($return, $this->replaceVariables($value, $result, $product, $store));

        } else {
            $return['isOk'] = false; // FAILURE
            $return['value'] = 'Invalid match result by pattern: '.$pattern;
        }
        return $return;
    }
    public function getExtractedTitleFromATag($aTag) {
        if (strpos($aTag, '<a') !== false || strpos($aTag, '< a') !== false) {
            $aTag = str_replace('&', '&amp;', $aTag);
            $simpleXmlElement = @simplexml_load_string($aTag);
            if ($simpleXmlElement) {
                return (string)$simpleXmlElement;
            }
        }
        return false;
    }

    /**
     * @param $value
     * @param $result
     * @param \Magento\Catalog\Model\Product $product
     * @param \Magento\Store\Model\Store $store
     * @return array
     */
    protected function replaceVariables($value, $result, $product, $store) {
        $missingValues = []; $exceptions = [];
        $filterOutOptionsTranslated = array(__('No'), __('Yes'));
        $filterOutOptions = array_merge(array('Yes', 'No', 'Non', 'Nein', 'Ja', 'Aucun', 'Oui', 'Geen', '否', 'Sì', 'oui', '是'), $filterOutOptionsTranslated);
        $variables = $result[1];
        foreach ($variables as $variable) {
            if (strpos($variable, '{{') === 0) continue;
            $replacement = '';
            if ($product->hasData($variable)) {
                try {
                    $replacement = $product->getResource()->getAttribute($variable)->setStoreId($store->getId())
                        ->getFrontend()->getValue($product);
                    if (strlen($replacement) == 0 || in_array($replacement, $filterOutOptions)) {
                         $productModel = $this->product->create();
                        $productWithDefaults = $productModel->setStoreId(0)->load($product->getId());
                        $attrValue = $productWithDefaults->getData($variable);
                        if ($attrValue && ($source =
                                $productWithDefaults->getResource()->getAttribute($variable)->getSource())) {
                            $replacement = $source->getOptionText($attrValue);
                        }
                    }
                } catch (Exception $e) {
                    $exceptions[$variable] = $e->getMessage();
                }
                if (is_null($replacement) || strlen($replacement) == 0) {
                    $replacement = '';
                    $missingValues[] = $variable;

                } elseif ($title = $this->getExtractedTitleFromATag($replacement)) {
                    $replacement = $title;

                } elseif (empty($replacement) || in_array($replacement, $filterOutOptions)) {
                    $replacement = ''; $missingValues[] = $variable;
                }
            } else {
                $missingValues[] = $variable;
            }
            $value = str_ireplace('{' . $variable . '}', $replacement, $value);
        }

        if (count($missingValues) > 0 /*(($missingValues / $matchesCount) * 100) >= 50*/) {
            $return['isOk'] = false; // FAILURE
            $return['value'] = sprintf('%s value(s) missing (%s): %s', count($missingValues),
                implode(',', $missingValues) , $value);
            if (!empty($exceptions)) {
                $msg = '';
                foreach ($exceptions as $var => $exception) {
                    $msg = sprintf('[Exception(%s)="%s"]', $var, $exception); break;
                }
                $return['value'] = $return['value'].' '.$msg;
            }
        } else {
            $return['isOk'] = true;
            $return['value'] = $value;
        }
        return $return;
    }

    /**
     * @param $id
     */
    protected function addError($id) {
        if (array_key_exists($id, $this->processingErrors)) {
            $count = $this->processingErrors[$id];
            $this->processingErrors[$id] = ++$count;
        } else {
            $this->processingErrors[$id] = 1;
        }
    }

    /**
     *
     * @param array $configItem
     * @param \Magento\Catalog\Model\Product $product
     * @return bool|NULL
     */
    protected function configItemMatchesForProduct(array $configItem, $product) {
        $attributeSetName = mb_strtolower($configItem['attributeset']);
        $attributesetCollection = $this->attributesetICollection;
        $productsAttributeSetName = mb_strtolower($attributesetCollection->get($product->getAttributeSetId())->getAttributeSetName());
            
        if ($attributeSetName == self::NOTATION_INCLUDE_ALL_ATTRSET || $productsAttributeSetName == $attributeSetName) {
            return true;
        } elseif (strpos($attributeSetName, self::NOTATION_EXCLUDE_ATTRSET) === 0 &&
                strpos($attributeSetName, $productsAttributeSetName) > 0) {
            return null; // We explicitely don't want to modify this one
        }
        return false;
    }
        //   added from justselling helper class
         /**
     * @param Mage_Catalog_Model_Product $product
     * @return bool|array
     */
    public function getExtendedGroupedAttributesForProduct($product, $backendValues = false) {
        $groups = array();
        $attributeSet = $this->getAttributeSet($product);
        if (is_null($this->supportedAttributSets) ||
            in_array($attributeSet->getAttributeSetName(), $this->supportedAttributSets)) {
            $attributeGroups = $this->getAttributeSetGroups($attributeSet);
            /** @var $attributeGroup Mage_Eav_Model_Entity_Attribute_Group */
            foreach ($attributeGroups as $attributeGroup) {
                $groupLabel = $attributeGroup->getAttributeGroupName();
                $attributes = $this->getAttributesOfGroup($attributeGroup);
                /** @var $attribute Mage_Eav_Model_Entity_Attribute */
                foreach ($attributes as $attribute) {
                    if ($attribute->getIsVisibleOnFront() &&
                            $product->hasData($attribute->getAttributeCode()) &&
                            $product->getData($attribute->getAttributeCode()) != null) {
                        if (!array_key_exists($groupLabel, $groups)) {
                            $groups[$groupLabel] = array();
                        }
                        $frontend = $product->getResource()->getAttribute($attribute->getAttributeCode())
                            ->setStoreId($product->getStoreId())
                            ->getFrontend();
                        $productAttribute = $product->getResource()->getAttribute($attribute->getAttributeCode());
                        $value = $frontend->getValue($product);
                        if ($attribute->getFrontendInput() == 'multiselect') {
                            $optionIds = $product->getData($attribute->getAttributeCode());
                            $value = array();
                            foreach (explode(',', $optionIds) as $optionId) {
                                if ( $backendValues ) {
                                    $value[] = [
                                        'frontend' => $productAttribute->setStoreId(
                                            $product->getStoreId())->getSource()->getOptionText($optionId), 
                                            'backend' => $productAttribute->setStoreId(0)->getSource()->getOptionText($optionId)];
                                } else {
                                    $value[] = $frontend->getOption($optionId);
                                }
                            }
                        }
                        $value = $attribute->getAttributeCode() == 'weight' ? number_format($value, 2, ',', '.').' kg' : $value;
                        $groups[$groupLabel][$attribute->getAttributeCode()] = array(
                            'attributeCode' => $attribute->getAttributeCode(),
                            'value' => $value,
                            'attributeLabel' => $attribute->getStoreLabel($product->getStoreId()),
                            'position' => $attribute->getData('position')
                        ); // may be array
                    }
                }
            }
            return $groups;
        }
        return false;
    }

    /**
     * @param Mage_Catalog_Model_Product $product
     * @return false|Mage_Core_Model_Abstract|Mage_Eav_Model_Entity_Attribute_Set
     */
    public function getAttributeSet($product) {
        $attributeSet = $this->objectManager->create('Magento\Eav\Api\AttributeSetRepositoryInterface');
        $attributeSetRepository = $attributeSet->get($product->getAttributeSetId());
        return  $attributeSetRepository;
    }

    /**
     * @param Mage_Eav_Model_Entity_Attribute_Set $attributeSet
     * @return array|Mage_Eav_Model_Resource_Entity_Attribute_Group_Collection
     */
    protected function getAttributeSetGroups($attributeSet) {
        $groups = array();
        /** @var Mage_Eav_Model_Resource_Entity_Attribute_Group_Collection $attrGroupCollection */
        $attrGroupCollection  = $this->attributeGroupCollection->create();
        $attrGroupCollection->setAttributeSetFilter($attributeSet->getId());
        $attrGroupCollection->setOrder('sort_order', 'asc');
        /** @var $attrGroup Mage_Eav_Model_Entity_Attribute_Group */
        foreach ($attrGroupCollection as $attrGroup) {
            if ($attrGroup->getAttributeGroupName()[0] == '_') {
                $groups[] = $attrGroup;
            }
        }
        return $groups;
    }

    /**
     * @param Mage_Eav_Model_Entity_Attribute_Group $attributeGroup
     * @return Mage_Eav_Model_Resource_Entity_Attribute_Collection
     */
    protected function getAttributesOfGroup($attributeGroup) {
        if (!isset($this->attributeGroups[$attributeGroup->getId()])) {
           $attrCollection =  $this->eavAttribute->getCollection(); 
            $attrCollection->setAttributeSetFilter($attributeGroup->getAttributeSetId());
            $attrCollection->setAttributeGroupFilter($attributeGroup->getId());
            $attrCollection->getSelect()
                ->joinLeft(
                    array("t1" => 'catalog_eav_attribute'),
                    "main_table.attribute_id = t1.attribute_id");
            $this->attributeGroups[$attributeGroup->getId()] = $attrCollection;
        }
        return $this->attributeGroups[$attributeGroup->getId()];
    }

    /**
     * Retrieve Usage Help Message
     *
     */
    public function usageHelp(){
        return <<<USAGE
Usage:  php bin/magento Xtwo:AttributesAutofills --[options]

  --store <store_codes>        Stores to process products for. Multiple separated via comma.
  
  --sku <sku_list>             List of SKUs to limit processing for. Multiple separated via comma.
  
  --attributeset <attributeset_list>    List of Attributesets to process products for. Multiple separated via comma.

  help                  This help
  
  <store_code>  Store code, e.g. "xtwostore_de,xtwostore_com" etc.
  <sku_list>    List of product SKUs. If not provided, all products will be used.
  <attributeset_list>    List of attributesets. If not provided, all products will be processed.

USAGE;
    }
}
